package tecgraf.openbus.browser.scs_offers;

import java.awt.MouseInfo;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.datatransfer.DataFlavor;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.JOptionPane;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreePath;

/**
 * Implementao da API Swing para drag-and-drop em um JTree
 * <p>
 * Esta classe  necessria para viabilizar o drag-and-drop entre ns da rvore.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
final class SCSTreeHandler implements KeyListener, DropTargetListener {

	private final JTree resultTree;

	public SCSTreeHandler(JTree resultTree) {
		this.resultTree = resultTree;
	}

	@Override
	public void keyTyped(KeyEvent e) {
	}

	@Override
	public void keyPressed(KeyEvent e) {
		if (e.getKeyCode() == KeyEvent.VK_F5) {
			RefreshableNode refreshable = getRefreshable(resultTree.getSelectionPath());
			if (refreshable != null) {
				e.consume();
				try {
					refreshable.refreshNode();
				}
				catch (Throwable ex) {
					JOptionPane
					  .showMessageDialog(resultTree, ex.getMessage(), ex.getClass().getName(), JOptionPane.ERROR_MESSAGE);
				}
			}
		}
	}

	@Override
	public void keyReleased(KeyEvent e) {
	}

	@Override
	public void dragEnter(DropTargetDragEvent dtde) {
	}

	@Override
	public void dragOver(DropTargetDragEvent dtde) {

		if (dtde.getDropTargetContext().getComponent() != resultTree)
			return;

		// Faz o efeito de scroll.
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				boolean scrolled = false;

				Point mouse = MouseInfo.getPointerInfo().getLocation();
				Point topLeft = resultTree.getParent().getLocationOnScreen();
				int width = resultTree.getParent().getWidth();
				int height = resultTree.getParent().getHeight();

				if (mouse.y < topLeft.y + 10) {
					Rectangle newRect = resultTree.getVisibleRect();
					newRect.y -= 1;
					if (newRect.y >= 0) {
						resultTree.scrollRectToVisible(newRect);
						scrolled = true;
					}
				}
				else if (mouse.y > topLeft.y + height - 10) {
					Rectangle newRect = resultTree.getVisibleRect();
					newRect.y += 1;
					if (newRect.y <= resultTree.getHeight() - height) {
						resultTree.scrollRectToVisible(newRect);
						scrolled = true;
					}
				}

				if (mouse.x < topLeft.x + 10) {
					Rectangle newRect = resultTree.getVisibleRect();
					newRect.x -= 1;
					if (newRect.x >= 0) {
						resultTree.scrollRectToVisible(newRect);
						scrolled = true;
					}
				}
				else if (mouse.x > topLeft.x + width - 10) {
					Rectangle newRect = resultTree.getVisibleRect();
					newRect.x += 1;
					if (newRect.x <= resultTree.getWidth() - width) {
						resultTree.scrollRectToVisible(newRect);
						scrolled = true;
					}
				}

				if (scrolled)
					SwingUtilities.invokeLater(this);
			}
		});

		checkIfCanDrop(dtde);

	}

	private final void checkIfCanDrop(DropTargetDragEvent dtde) {
		TreePath row = resultTree.getClosestPathForLocation(dtde.getLocation().x, dtde.getLocation().y);
		if (row == null || row.getLastPathComponent() == null) {
			dtde.rejectDrag();
			resultTree.setLeadSelectionPath(resultTree.getSelectionPath());
			return;
		}

		if (!row.equals(resultTree.getLeadSelectionPath()))
			resultTree.setLeadSelectionPath(row);

		NodeWithDropBehaviorInterface droppable = getDroppable(row);

		if (droppable == null) {
			dtde.rejectDrag();
			return;
		}

		int accept = 0;
		for (DataFlavor flavor : dtde.getCurrentDataFlavors()) {
			accept |= droppable.accept(flavor);
		}

		if (accept > 0) {
			dtde.acceptDrag(accept);
		}
		else
			dtde.rejectDrag();
	}

	private NodeWithDropBehaviorInterface getDroppable(TreePath row) {
		NodeWithDropBehaviorInterface droppable = null;
		if (row.getLastPathComponent() instanceof NodeWithDropBehaviorInterface) {
			droppable = (NodeWithDropBehaviorInterface) row.getLastPathComponent();
		}
		else if (row.getLastPathComponent() instanceof DefaultMutableTreeNode) {
			DefaultMutableTreeNode n = (DefaultMutableTreeNode) row.getLastPathComponent();
			if (n.getUserObject() != null && n.getUserObject() instanceof NodeWithDropBehaviorInterface)
				droppable = (NodeWithDropBehaviorInterface) n.getUserObject();
		}
		return droppable;
	}

	private RefreshableNode getRefreshable(TreePath row) {
		RefreshableNode refreshable = null;
		if (row.getLastPathComponent() instanceof RefreshableNode) {
			refreshable = (RefreshableNode) row.getLastPathComponent();
		}
		else if (row.getLastPathComponent() instanceof DefaultMutableTreeNode) {
			DefaultMutableTreeNode n = (DefaultMutableTreeNode) row.getLastPathComponent();
			if (n.getUserObject() != null && n.getUserObject() instanceof RefreshableNode)
				refreshable = (RefreshableNode) n.getUserObject();
		}
		return refreshable;
	}

	@Override
	public void dropActionChanged(DropTargetDragEvent dtde) {
		checkIfCanDrop(dtde);
	}

	@Override
	public void dragExit(DropTargetEvent dte) {
		resultTree.setLeadSelectionPath(resultTree.getSelectionPath());
	}

	@Override
	public void drop(DropTargetDropEvent dtde) {
		TreePath row = resultTree.getClosestPathForLocation(dtde.getLocation().x, dtde.getLocation().y);
		resultTree.setLeadSelectionPath(resultTree.getSelectionPath());

		if (row == null || row.getLastPathComponent() == null) {
			dtde.rejectDrop();
			return;
		}

		NodeWithDropBehaviorInterface droppable = getDroppable(row);

		if (droppable == null) {
			dtde.rejectDrop();
			return;
		}

		int accept = 0;
		for (DataFlavor flavor : dtde.getCurrentDataFlavors()) {
			accept |= droppable.accept(flavor);
		}

		if (accept > 0) {
			dtde.acceptDrop(accept);
			try {
				droppable.doDrop(dtde.getTransferable());
				dtde.dropComplete(true);
			}
			catch (Throwable e) {
				e.printStackTrace(System.err);
				dtde.dropComplete(false);
			}
		}
		else
			dtde.rejectDrop();
	}

}
