package tecgraf.openbus.browser.scs_offers;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.omg.CORBA.ORB;
import org.omg.CORBA.ORBPackage.InvalidName;
import org.omg.PortableServer.POA;
import org.omg.PortableServer.POAHelper;

import tecgraf.openbus.Connection;
import tecgraf.openbus.OpenBusContext;
import tecgraf.openbus.browser.ConnectionProvider;
import tecgraf.openbus.browser.ManagedConnection;
import tecgraf.openbus.browser.ManagedConnection.ConnectionState;
import tecgraf.openbus.browser.ManagedConnection.LogoffSyncListener;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;
import tecgraf.openbus.core.v2_0.services.offer_registry.OfferObserver;
import tecgraf.openbus.core.v2_0.services.offer_registry.OfferObserverHelper;
import tecgraf.openbus.core.v2_0.services.offer_registry.OfferObserverPOA;
import tecgraf.openbus.core.v2_0.services.offer_registry.OfferObserverSubscription;
import tecgraf.openbus.core.v2_0.services.offer_registry.OfferRegistryObserver;
import tecgraf.openbus.core.v2_0.services.offer_registry.OfferRegistryObserverHelper;
import tecgraf.openbus.core.v2_0.services.offer_registry.OfferRegistryObserverPOA;
import tecgraf.openbus.core.v2_0.services.offer_registry.OfferRegistryObserverSubscription;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOffer;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOfferDesc;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceProperty;

/**
 * Pool de observadores do servio de ofertas cujas responsabilidades so:
 * 
 * <ul>
 * <li>Garantir que s exista um observador por oferta, e por busid;
 * <li>Registrar observadores e receber as chamadas remotas do servio de
 * ofertas com as notificaes;
 * <li>Remover observadores amigavelmente quando no forem mais necessrios.
 * </ul>
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
final class ServiceOfferRegistryObserversPool {

	private static final String SERVICE_OFFER_OBSERVERS_KEY = "serviceOfferObservers";

	static final Logger logger = Logger.getLogger(ServiceOfferRegistryObserversPool.class.toString());

	/**
	 * Bean que representa um barramento em especfico (busid) cujos observadores
	 * so gerenciados aqui.
	 * 
	 * Em um busid, a coleo de ofertas que so gerenciadas ficam registradas
	 * aqui atravs de referncias fracas de {@link ServiceOfferManagedBean}.
	 */
	final class BusObserverPoolBean {
		String busId;
		OfferRegistryObserverSubscription subscription;
		public final Map<ServiceOffer, WeakReference<ServiceOfferManagedBean>> controlledBeans =
		  new WeakHashMap<ServiceOffer, WeakReference<ServiceOfferManagedBean>>();

		public ServiceOfferManagedBean getControlledBean(ServiceOffer key) {
			synchronized (controlledBeans) {
				WeakReference<ServiceOfferManagedBean> wBean = controlledBeans.get(key);
				if (wBean == null)
					return null;
				return wBean.get();
			}
		}

		/**
		 * Remover amigavelmente todos os observadores remotos relacionados  este
		 * busid.
		 */
		public void clearSubscriptions() {
			if (subscription != null) {
				logger.fine("Removendo listener do busid=" + busId);
				try {
					subscription.remove();
				}
				catch (Throwable e) {
					logger.log(Level.WARNING, "Exceo ignorada ao remover listener do servio de ofertas.", e);
				}
				subscription = null;
			}

			ServiceOffer monitoredOffers[];
			synchronized (controlledBeans) {
				monitoredOffers = controlledBeans.keySet().toArray(new ServiceOffer[] {});
			}

			for (ServiceOffer o : monitoredOffers) {
				ServiceOfferManagedBean bean = getControlledBean(o);
				if (bean != null) {
					try {
						bean.removeSubscriber();
					}
					catch (Throwable e) {
						logger.log(Level.WARNING, "Exceo ignorada ao remover listener de oferta.", e);
					}
				}
			}
			synchronized (controlledBeans) {
				controlledBeans.clear();
			}
		}
	}

	/**
	 * Interface para notificao interna ocorrida por uma notificao externa de
	 * que uma nova oferta foi registrada no servio de registro.
	 */
	public interface ServiceOfferRegistryObserversPoolListener {
		public void offerRegistered(ManagedConnection cnn, ServiceOfferDesc offer);
	}

	private static final ServiceOfferRegistryObserversPool singleton = new ServiceOfferRegistryObserversPool();

	ConnectionProvider connectionProvider = null;

	private OfferRegistryObserver offerRegistryObserver = null;

	private OfferObserver offerObserver = null;

	private final HashMap<String, BusObserverPoolBean> busSubscriptions =
	  new HashMap<String, BusObserverPoolBean>(4);

	private final CopyOnWriteArrayList<ServiceOfferRegistryObserversPoolListener> offerRegistryListeners =
	  new CopyOnWriteArrayList<ServiceOfferRegistryObserversPoolListener>();

	private ORB orb;

	private final OfferRegistryObserverPOA offerRegistryObserverPOA = new OfferRegistryObserverPOA() {
		@Override
		public void offerRegistered(ServiceOfferDesc offer) {

			OpenBusContext ctx = getOpenBusContext();

			ManagedConnection myCnn = connectionProvider.getManagedConnectionOf(ctx.getCurrentConnection());

			for (ServiceOfferRegistryObserversPoolListener l : offerRegistryListeners) {
				try {
					l.offerRegistered(myCnn, offer);
				}
				catch (Throwable e) {
					logger.log(Level.WARNING, "Exceo ignorada ao chamar listener interno do servio de ofertas.", e);
				}
			}
		}
	};

	private final OfferObserverPOA offerObserverPOA = new OfferObserverPOA() {
		@Override
		public void removed(ServiceOfferDesc offer) {

			OpenBusContext ctx = getOpenBusContext();

			BusObserverPoolBean busBean;
			synchronized (busSubscriptions) {
				busBean = busSubscriptions.get(ctx.getCurrentConnection().busid());
			}

			if (busBean == null) {
				logger.warning("OfferObserver.removed() chamado sem que eu tenha um busid criado: busid="
				  + ctx.getCurrentConnection().busid());
				return;
			}

			synchronized (busBean.controlledBeans) {
				ServiceOfferManagedBean myBean = busBean.getControlledBean(offer.ref);
				if (myBean == null) {
					throw new IllegalStateException("Oferta no identificada no pool: " + offer.ref);
				}
				myBean.notifyRemoved();
				myBean.offerObserverSubscription = null;
				busBean.controlledBeans.remove(offer.ref);
			}
		}

		@Override
		public void propertiesChanged(ServiceOfferDesc offer) {
			OpenBusContext ctx = getOpenBusContext();

			BusObserverPoolBean busBean;
			synchronized (busSubscriptions) {
				busBean = busSubscriptions.get(ctx.getCurrentConnection().busid());
			}

			ServiceOfferManagedBean myBean = busBean.getControlledBean(offer.ref);
			if (myBean == null) {
				throw new IllegalStateException("Oferta no identificada no pool: " + offer.ref);
			}
			myBean.notifyChanged();

		}

	};

	private final LogoffSyncListener connectionObserver = new LogoffSyncListener() {
		@Override
		public void preLogoff(ManagedConnection cnn) {
			if (cnn.getState() == ConnectionState.DISCONNECTING) {
				cnn.setContextCurrentConnection();
				int connectionsWithSameBusID = connectionProvider.getNumActiveConnectionsOfBusID(cnn.busid());
				logger.fine("desconectando: Conexes com o mesmo busid="
				  + connectionsWithSameBusID);
				if (connectionsWithSameBusID == 1) {
					BusObserverPoolBean busBean;
					synchronized (busSubscriptions) {
						busBean = busSubscriptions.remove(cnn.busid());
						if (busBean != null) {
							busBean.clearSubscriptions();
						}
					}
				}
			}
		}
	};

	private ServiceOfferRegistryObserversPool() {}

	public static final ServiceOfferRegistryObserversPool getSingleton() {
		return singleton;
	}

	private final void startup(ORB orb) {

		if (this.orb != null)
			throw new IllegalStateException("No posso inicializar duas vezes!");

		logger.fine("Startup(orb=" + System.identityHashCode(orb) + ")");
		this.orb = orb;

		POA poa;
		try {
			poa = POAHelper.narrow(orb.resolve_initial_references("RootPOA"));
			poa.the_POAManager().activate();

			org.omg.CORBA.Object offerRegistryObserverObject = poa.servant_to_reference(offerRegistryObserverPOA);
			offerRegistryObserver = OfferRegistryObserverHelper.narrow(offerRegistryObserverObject);

			org.omg.CORBA.Object offerObserverObject = poa.servant_to_reference(offerObserverPOA);
			offerObserver = OfferObserverHelper.narrow(offerObserverObject);

		}
		catch (Exception e1) {
			throw new IllegalStateException(e1);
		}

	}

	/**
	 * Registrar uma nova oferta de servio no pool de ofertas, e retornar o bean
	 * que representa o gerenciamento deste observador internamente.
	 * <p>
	 * O bean retornado deve ser guardado (referenciado) pelo chamador, pois a
	 * perda da referncia far com que o observador seja desconectado.
	 * 
	 * @param cnn Conexo Openbus a ser usada.
	 * @param serviceOffer Oferta de servio na qual se deseja que seja
	 *          gerenciada.
	 * @return Instncia do bean interno que representa a oferta gerenciada.
	 * @throws ServiceFailure
	 */
	public ServiceOfferManagedBean addNewServiceOfferBean(Connection cnn, ServiceOffer serviceOffer)
	  throws ServiceFailure {

		// Opo para no utilizar Observers ("-DserviceOfferObservers=false ou true")
		if (!getServiceOfferObserversSystemPropertyValue()) {
			final ServiceOfferManagedBean newBean = new ServiceOfferManagedBean(this, null, serviceOffer, null);
			return newBean;
		}

		ManagedConnection.setContextCurrentConnection(cnn);
		synchronized (this) {
			if (orb == null) {
				startup(cnn.orb());
			}
		}

		final BusObserverPoolBean busBean = getBusObserverPoolBean(cnn);

		ServiceOfferManagedBean res;
		synchronized (busBean.controlledBeans) {
			res = busBean.getControlledBean(serviceOffer);
			boolean nonExistant = false;
			try{
				nonExistant = res.offerObserverSubscription._non_existent();
			}catch(Throwable e){
				nonExistant = true;
			}
			if (res == null || nonExistant) {
				final OfferObserverSubscription offerObserverSubscription = serviceOffer.subscribeObserver(offerObserver);
				logger.fine("Listener de oferta criado: " + offerObserverSubscription.toString());
				final ServiceOfferManagedBean newBean =
				  new ServiceOfferManagedBean(this, busBean, serviceOffer, offerObserverSubscription);
				busBean.controlledBeans.put(serviceOffer, new WeakReference<ServiceOfferManagedBean>(newBean));
				res = newBean;
			}
		}

		// Garantir que a conexo ser observada pelo pool (apenas uma vez!)
		if (cnn instanceof ManagedConnection) {
			ManagedConnection mCnn = (ManagedConnection) cnn;
			mCnn.removeLogoffSyncListener(connectionObserver);
			mCnn.addLogoffSyncListener(connectionObserver);
		}

		return res;
	}

	public static final boolean getServiceOfferObserversSystemPropertyValue() {
		String propertyValue = System.getProperty(SERVICE_OFFER_OBSERVERS_KEY);
		if (propertyValue == null)
			return true;
		else
			propertyValue = propertyValue.toLowerCase().trim();

		return propertyValue.equals("on") || propertyValue.equals("true")
		  || propertyValue.equals("1") || propertyValue.equals("yes");
	}

	private BusObserverPoolBean getBusObserverPoolBean(Connection cnn) throws ServiceFailure {
		synchronized (busSubscriptions) {
			BusObserverPoolBean busBean = busSubscriptions.get(cnn.busid());
			if (busBean == null) {
				busBean = new BusObserverPoolBean();
				busBean.busId = cnn.busid();
				ServiceProperty genericFilter[] = new ServiceProperty[] {
				    new ServiceProperty("openbus.component.facet", "IComponent")
				};

				busBean.subscription = ManagedConnection.getContext(cnn)
				  .getOfferRegistry()
				  .subscribeObserver(offerRegistryObserver, genericFilter);
				logger.fine("Registrado observador do busid=" + cnn.busid());
				busSubscriptions.put(cnn.busid(), busBean);
			}
			return busBean;
		}
	}

	private OpenBusContext getOpenBusContext() {
		OpenBusContext ctx;
		try {
			ctx = (OpenBusContext) orb.resolve_initial_references("OpenBusContext");
		}
		catch (InvalidName e) {
			throw new IllegalStateException("Isto realmente no deveria ter acontecido. Comece a rezar.", e);
		}
		return ctx;
	}

	public void addListener(ServiceOfferRegistryObserversPoolListener listener) {
		offerRegistryListeners.add(listener);
	}

	public void removeListener(ServiceOfferRegistryObserversPoolListener listener) {
		offerRegistryListeners.remove(listener);
	}

	protected void setConnectionProvider(ConnectionProvider connectionProvider) {
		this.connectionProvider = connectionProvider;
	}

	public static final String getOfferID(ServiceOffer offer) {
		ServiceProperty properties[] = offer.properties();
		if (properties == null)
			return null;
		for (ServiceProperty p : properties) {
			if (p != null)
				if ("openbus.offer.id".equals(p.name))
					return p.value;
		}
		return null;
	}

}
