package tecgraf.openbus.browser.scs_offers;

import java.awt.Component;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import javax.swing.JOptionPane;

import org.jacorb.orb.ORB;
import org.jacorb.orb.ParsedIOR;
import org.jacorb.orb.iiop.IIOPAddress;
import org.jacorb.orb.iiop.IIOPProfile;
import org.omg.CORBA.Object;

import tecgraf.openbus.Connection;
import tecgraf.openbus.browser.ManagedConnection;
import tecgraf.openbus.browser.OpenbusBrowser;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOfferDesc;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceProperty;

/**
 * Utilitrio para buscar ofertas bem especficas no servio de registro do
 * Openbus.  til quando se deseja recuperar alguma faceta onde se espera
 * apenas uma instncia, dado o critrio de busca.
 * 
 * <p>
 * Os mtodos desta classe otimizam o tempo de retorno da busca, evitando
 * demoras causadas por temporizaes nas tentativas de chamadas remotas
 * de ofertas invlidas.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
public final class SingleServiceFinder {

	private SingleServiceFinder() {
	}

	private static final class FindResult {
		final ServiceOfferDesc desc;
		final Object ref;
		final Connection cnn;

		public FindResult(Connection cnn, ServiceOfferDesc desc, Object ref) {
			super();
			this.cnn = cnn;
			this.desc = desc;
			this.ref = ref;
		}

		@Override
		public String toString() {

			String host = "";
			try {
				ParsedIOR ior = new ParsedIOR((ORB) cnn.orb(), ref.toString());
				IIOPProfile profile = (IIOPProfile) ior.getProfiles().get(0);
				String ip = ((IIOPAddress) profile.getAddress()).getIP();
				String hostname = ((IIOPAddress) profile.getAddress()).getHostname();
				if (hostname != null && !hostname.isEmpty() && !ip.equals(hostname))
					ip += "(" + hostname + ")";
				host = " do host " + ip;
			}
			catch (Throwable e) {
				e.printStackTrace(System.err);
			}

			String res = "Oferta " + host
			  + " feita ou renovada em "
			  + getProperty("openbus.offer.day")
			  + "/" + getProperty("openbus.offer.month")
			  + "/" + getProperty("openbus.offer.year")
			  + " " + getProperty("openbus.offer.hour")
			  + ":" + getProperty("openbus.offer.minute")
			  + ":" + getProperty("openbus.offer.second")
			  + " por " + getProperty("openbus.offer.entity");

			return res;
		}

		private final String getProperty(String propName) {
			if (desc.properties != null)
				for (ServiceProperty p : desc.properties)
					if (propName.equals(p.name))
						return p.value;
			return "";
		}

	}

	public static ServiceOfferDesc findFirstOffer(final Connection cnn, ServiceProperty properties[])
	  throws InterruptedException, ServiceFailure {
		ManagedConnection.setContextCurrentConnection(cnn);
		ServiceOfferDesc offers[] = ManagedConnection.getContext(cnn)
			.getOfferRegistry().findServices(properties);

		final ArrayList<FindResult> res = new ArrayList<FindResult>();
		final AtomicInteger offersCountDown = new AtomicInteger(offers.length);
		final AtomicBoolean validOfferFound = new AtomicBoolean(false);
		final ArrayList<Thread> threads = new ArrayList<Thread>();
		for (final ServiceOfferDesc offer : offers) {
			Thread t = new Thread() {
				@Override
				public void run() {
					try {
						ManagedConnection.setContextCurrentConnection(cnn);
						Object o = offer.service_ref;
						if (!o._non_existent()) {
							res.add(new FindResult(cnn, offer, o));
							validOfferFound.set(true);
						}
					}
					catch (Throwable e) {
					}
					finally {
						offersCountDown.decrementAndGet();
					}
				}
			};
			t.start();
			threads.add(t);
		}

		while (!validOfferFound.get()) {
			if (offersCountDown.get() == 0)
				break;
			Thread.sleep(10);
		}

		for (Thread t : threads)
			t.interrupt();

		if (res.isEmpty())
			return null;

		return res.get(0).desc;

	}

	/**
	 * Recuperar a nica oferta de servio no servio de ofertas que atende aos critrios
	 * especificados.
	 * 
	 * <p>
	 * Se este mtodo identificar mais de uma oferta com o critrio, um popup ser
	 * exibido para que o usurio escolha entre as ofertas encontradas.
	 * 
	 * @param cnn Conexo Openbus a ser usada para realizar a busca
	 * @param properties Filtro que deve retornar apenas uma oferta
	 * @param mayFindNothing Se for false, exibe um popup para o usurio informando que 
	 *                       nada foi encontrado.
	 * @return A nica oferta de servio encontrada ou null caso nenhuma seja encontrada.
	 * @throws ServiceFailure
	 * @throws InterruptedException
	 */
	public static ServiceOfferDesc findSingleOffer(final ManagedConnection cnn, ServiceProperty properties[],
	  final boolean mayFindNothing)
	  throws ServiceFailure, InterruptedException {
		return findSingleOffer(null, cnn, properties, mayFindNothing);
	}

	/**
	 * Recuperar a nica oferta de servio no servio de ofertas que atende aos critrios
	 * especificados.
	 * 
	 * <p>
	 * Se este mtodo identificar mais de uma oferta com o critrio, um popup ser
	 * exibido para que o usurio escolha entre as ofertas encontradas.
	 * 
	 * @param parentForUI Componente pai para os possveis popups gerados por este mtodo.
	 * @param cnn Conexo Openbus a ser usada para realizar a busca
	 * @param properties Filtro que deve retornar apenas uma oferta
	 * @param mayFindNothing Se for false, exibe um popup para o usurio informando que 
	 *                       nada foi encontrado.
	 * @return A nica oferta de servio encontrada ou null caso nenhuma seja encontrada.
	 * @throws ServiceFailure
	 * @throws InterruptedException
	 */
	public static ServiceOfferDesc findSingleOffer(Component parentForUI, final ManagedConnection cnn,
	  ServiceProperty properties[], final boolean mayFindNothing)
	  throws ServiceFailure, InterruptedException {

		if (parentForUI == null)
			parentForUI = OpenbusBrowser.getSingletonInstance();

		cnn.setContextCurrentConnection();
		ServiceOfferDesc offers[] = cnn.getContext().getOfferRegistry().findServices(properties);

		if (offers == null || offers.length == 0) {
			if (!mayFindNothing)
				JOptionPane.showMessageDialog(parentForUI,
				  "Servio de ofertas da conexo no identificou nenhuma oferta com o critrio: \n"
				    + getPropertiesToString(properties), "Erro", JOptionPane.ERROR_MESSAGE);
			return null;
		}

		final ArrayList<FindResult> res = new ArrayList<FindResult>();
		final HashSet<String> exceptions = new HashSet<String>();
		final AtomicInteger offersCountDown = new AtomicInteger(offers.length);
		final AtomicBoolean validOfferFound = new AtomicBoolean(false);
		final ArrayList<Thread> threads = new ArrayList<Thread>();
		for (final ServiceOfferDesc offer : offers) {
			Thread t = new Thread() {
				@Override
				public void run() {
					try {
						Object o = offer.service_ref;
						if (!o._non_existent()) {
							res.add(new FindResult(cnn, offer, o));
							validOfferFound.set(true);
						}
					}
					catch (Throwable e) {
						String desc = e.getClass().getName();
						if (e instanceof ServiceFailure) {
							desc += ": " + ((ServiceFailure) e).message;
						}
						else if (e instanceof tecgraf.openbus.data_service.core.v1_01.ServiceFailure) {
							desc += ": " + ((ServiceFailure) e).message;
						}
						else
							desc += ": " + e.getMessage();
						exceptions.add(desc);
					}
					finally {
						offersCountDown.decrementAndGet();
					}
				}
			};
			t.start();
			threads.add(t);
		}

		while (!validOfferFound.get()) {
			if (offersCountDown.get() == 0)
				break;
			Thread.sleep(10);
		}
		if (offersCountDown.get() > 0 && validOfferFound.get())
			Thread.sleep(1000); // Chance para mais algum entrar
		for (Thread t : threads) {
			t.interrupt();
		}

		if (res.size() == 0) {
			String exceptionsDesc = ".";
			if (!exceptions.isEmpty()) {
				exceptionsDesc = " pelas seguintes causas:";
				for (String e : exceptions.toArray(new String[] {}))
					exceptionsDesc += "\n" + e;
			}
			JOptionPane.showMessageDialog(parentForUI, "Servio de ofertas da conexo identificou " + offers.length
			  + " oferta(s) com o critrio: \n"
			  + getPropertiesToString(properties)
			  + "\nPorm nenhuma delas  vlida" + exceptionsDesc, "Erro", JOptionPane.ERROR_MESSAGE);
			return null;
		}
		else if (res.size() == 1) {
			return res.get(0).desc; // Resultado ideal!
		}
		else if (res.size() > 1) {
			// Ambiguidade detectada. Exibir dilogo para o usurio escolher a oferta adequada.
			String message =
			  "Servio de ofertas da conexo identificou " + res.size() + " ofertas vlidas com o critrio: \n"
			    + getPropertiesToString(properties)
			    + "\n Voc dever escolher uma.";

			FindResult sel =
			  (FindResult) JOptionPane.showInputDialog(
			    parentForUI, message, "Ambiguidade detectada", JOptionPane.QUESTION_MESSAGE, null,
			    res.toArray(new FindResult[] {}),
			    null);

			if (sel != null)
				return sel.desc;
		}

		return null;
	}

	private static final String getPropertiesToString(ServiceProperty properties[]) {
		StringBuilder res = new StringBuilder();

		for (ServiceProperty p : properties) {
			if (res.length() > 0)
				res.append("\n");
			res.append(p.name + "=" + p.value);
		}

		return res.toString();
	}

}
