package tecgraf.openbus.browser.scs_offers.basic_nodes;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.dnd.DnDConstants;
import java.lang.ref.WeakReference;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import scs.core.IMetaInterface;
import scs.core.IMetaInterfaceHelper;
import scs.core.IReceptacles;
import scs.core.ReceptacleDescription;
import tecgraf.openbus.Connection;
import tecgraf.openbus.browser.ManagedConnection;
import tecgraf.openbus.browser.SlowRequester;
import tecgraf.openbus.browser.TipPanelInterface;
import tecgraf.openbus.browser.SlowRequester.SlowRunnable;
import tecgraf.openbus.browser.scs_offers.NodeWithDetailsInterface;
import tecgraf.openbus.browser.scs_offers.NodeWithDropBehaviorInterface;
import tecgraf.openbus.browser.scs_offers.NodeWithHelpTipInterface;
import tecgraf.openbus.browser.scs_offers.NodeWithIconsInterface;
import tecgraf.openbus.browser.scs_offers.OfferRegistryPanel;
import tecgraf.openbus.browser.scs_offers.RefreshableNode;
import tecgraf.openbus.browser.scs_offers.basic_nodes.FacetNodeBean.TransferableFacetDescription;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOfferDesc;

/**
 * Tipo de n que representa um receptculo identificado em um componente SCS.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
public final class ReceptacleNodeBean implements NodeWithDropBehaviorInterface, RefreshableNode, NodeWithIconsInterface,
  NodeWithDetailsInterface, NodeWithHelpTipInterface {

	private final ServiceOfferDesc service;
	private ReceptacleDescription receptacleDescription;
	private final IReceptacles iReceptacles;
	private final Connection cnn;
	private WeakReference<ReceptacleNodeDetailsPanel> detailsPanel = null;

	public ReceptacleNodeBean(Connection cnn, ServiceOfferDesc service,
	  ReceptacleDescription receptacleDescription,
	  IReceptacles iReceptacles) {
		super();
		this.cnn = cnn;
		this.service = service;
		this.receptacleDescription = receptacleDescription;
		this.iReceptacles = iReceptacles;
	}

	/**
	 * @return the receptacleDescription
	 */
	public final ReceptacleDescription getReceptacleDescription() {
		return receptacleDescription;
	}

	/**
	 * @return the iReceptacles
	 */
	public IReceptacles getIReceptacles() {
		return iReceptacles;
	}

	@Override
	public String toString() {
		StringBuilder res = new StringBuilder();
		res.append(receptacleDescription.name);
		res.append("(");
		res.append(receptacleDescription.interface_name);
		res.append(")");
		if (receptacleDescription.is_multiplex)
			res.append("(multiplex)");
		res.append("[");
		res.append(receptacleDescription.connections.length);
		res.append("]");
		return res.toString();
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result =
		  prime
		    * result
		    + ((receptacleDescription == null) ? 0 : receptacleDescription
		      .hashCode());
		return result;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (!(obj instanceof ReceptacleNodeBean)) {
			return false;
		}
		ReceptacleNodeBean other = (ReceptacleNodeBean) obj;
		if (receptacleDescription == null) {
			if (other.receptacleDescription != null) {
				return false;
			}
		}
		else if (!receptacleDescription.name.equals(other.receptacleDescription.name)) {
			return false;
		}
		else if (!receptacleDescription.interface_name.equals(other.receptacleDescription.interface_name)) {
			return false;
		}
		return true;
	}

	@Override
	public int accept(DataFlavor dataFlavor) {
		if (FacetNodeBean.FACET_DESCRIPTION_DATA_FLAVOR.equals(dataFlavor)) {
			if (dataFlavor.getHumanPresentableName() == null || dataFlavor.getHumanPresentableName().trim().isEmpty())
				return DnDConstants.ACTION_COPY_OR_MOVE;
			return dataFlavor.getHumanPresentableName().equals(receptacleDescription.interface_name) ?
			  DnDConstants.ACTION_COPY_OR_MOVE : DnDConstants.ACTION_NONE;
		}
		return DnDConstants.ACTION_NONE;
	}

	@Override
	public boolean doDrop(final Transferable transferable) throws Exception {
		if (!transferable.isDataFlavorSupported(FacetNodeBean.FACET_DESCRIPTION_DATA_FLAVOR))
			return false;

		SlowRequester.run(new SlowRunnable() {
			@Override
			public void run() throws Exception {
				try {
					ManagedConnection.setContextCurrentConnection(cnn);

					final TransferableFacetDescription facetDescription =
					  (TransferableFacetDescription) transferable.getTransferData(FacetNodeBean.FACET_DESCRIPTION_DATA_FLAVOR);

					org.omg.CORBA.Object facet_ref = cnn.orb().string_to_object(facetDescription.ior);

					iReceptacles.connect(receptacleDescription.name, facet_ref);
				}
				finally {
					refreshNode();
				}
			}
		}, true);

		return true;
	}

	@Override
	public void refreshNode() throws Exception {
		ManagedConnection.setContextCurrentConnection(cnn);
		org.omg.CORBA.Object metaInterfaceObject = service.service_ref.getFacetByName("IMetaInterface");
		IMetaInterface metaInterface = IMetaInterfaceHelper.narrow(metaInterfaceObject);
		receptacleDescription = metaInterface.getReceptaclesByName(new String[] { receptacleDescription.name })[0];
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				((ReceptacleNodeDetailsPanel) getDetailsPanel()).reloadConnectionList();
			}
		});
	}

	private static final ImageIcon icon = new ImageIcon(OfferRegistryPanel.class.getResource("receptacle.png"));

	@Override
	public Icon getClosedIcon() {
		return icon;
	}

	@Override
	public Icon getLeafIcon() {
		return icon;
	}

	@Override
	public Icon getOpenedIcon() {
		return icon;
	}

	@Override
	public JPanel getDetailsPanel() {
		ReceptacleNodeDetailsPanel res = null;
		if (detailsPanel != null)
			res = detailsPanel.get();
		if (res == null) {
			res = new ReceptacleNodeDetailsPanel(cnn, this);
			detailsPanel = new WeakReference<ReceptacleNodeDetailsPanel>(res);
		}
		return res;
	}

	@Override
	public void configureHelpTip(TipPanelInterface tipPanel) {
		tipPanel.setTips(new String[] {
		    "Para conectar a faceta a um receptculo que a suporte, voc pode arrastar " +
		      "o n da rvore at um n do receptculo desejado.",

		    "Para conectar a faceta a um receptculo, voc pode \"copiar e colar\" o n da faceta " +
		      "no n do receptculo.",
		}, new String[] {
		  "No desconecte os receptculos dos coleguinhas."
		});
	}

}
