package tecgraf.openbus.browser.scs_offers.basic_nodes;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.BorderFactory;
import javax.swing.DefaultListModel;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

import net.miginfocom.swing.MigLayout;

import org.jacorb.orb.ORB;
import org.jacorb.orb.ParsedIOR;
import org.jacorb.orb.iiop.IIOPAddress;
import org.jacorb.orb.iiop.IIOPProfile;

import scs.core.ConnectionDescription;
import tecgraf.openbus.Connection;
import tecgraf.openbus.browser.ManagedConnection;
import tecgraf.openbus.browser.SlowRequester;
import tecgraf.openbus.browser.SlowRequester.SlowRunnable;

/**
 * Painel de detalhes para um n do tipo receptculo, que d a oportunidade
 * para o usurio desconectar receptculos.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
@SuppressWarnings("serial")
final class ReceptacleNodeDetailsPanel extends JPanel {

	private final ReceptacleNodeBean bean;
	private final JList list;
	private final Connection cnn;
	private JLabel lblName;
	private JLabel lblInterface;

	private final class ConnectionItem {
		private final ConnectionDescription receptacleCnn;
		private final String label;

		public ConnectionDescription getReceptacleCnn() {
			return receptacleCnn;
		}

		public ConnectionItem(ConnectionDescription receptacleCnn) {
			super();
			this.receptacleCnn = receptacleCnn;

			String host = "";
			try {
				ManagedConnection.setContextCurrentConnection(cnn);
				ParsedIOR ior = new ParsedIOR((ORB) cnn.orb(), receptacleCnn.objref.toString());
				IIOPProfile profile = (IIOPProfile) ior.getProfiles().get(0);
				String ip = ((IIOPAddress) profile.getAddress()).getIP();
				String hostname = ((IIOPAddress) profile.getAddress()).getHostname();
				if (hostname != null && !hostname.isEmpty() && !ip.equals(hostname))
					ip += "(" + hostname + ")";
				host = " Host " + ip;
			}
			catch (Throwable e) {
				e.printStackTrace(System.err);
			}

			label = "[" + receptacleCnn.id + "]" + host;
		}

		@Override
		public String toString() {
			return label;
		}
	}

	public ReceptacleNodeDetailsPanel(Connection cnn, ReceptacleNodeBean receptacleNodeBean) {

		super();

		this.cnn = cnn;
		this.bean = receptacleNodeBean;

		setLayout(new MigLayout("", "[grow]", "[][][grow]"));
		setMinimumSize(new Dimension(0, 0));

		lblName = new JLabel("Nome do receptculo");
		lblName.setMinimumSize(new Dimension(0, 20));
		add(lblName, "cell 0 0,growx");

		lblInterface = new JLabel("Interface do receptculo");
		lblInterface.setFont(new Font(Font.MONOSPACED, Font.PLAIN, 10));
		lblInterface.setMinimumSize(new Dimension(0, 16));
		add(lblInterface, "cell 0 1,growx");

		JScrollPane scrollPane = new JScrollPane();
		scrollPane.setBorder(BorderFactory.createTitledBorder("Conexes"));
		add(scrollPane, "cell 0 2,grow");

		list = new JList();
		list.setModel(new DefaultListModel());
		list.setBackground(UIManager.getColor("Window.background"));
		list.addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(KeyEvent e) {

				if (e.getKeyCode() == KeyEvent.VK_DELETE) {

					final Object[] selection = list.getSelectedValues();
					if (selection != null && selection.length > 0) {
						String message;
						if (selection.length == 1)
							message = "Confirma uma desconexo?";
						else
							message = "Confirma " + selection.length + " desconexes?";

						if (JOptionPane.showConfirmDialog(getRootPane(), message, "Desconexo", JOptionPane.YES_NO_OPTION) == JOptionPane.NO_OPTION)
							return;

						doDisconnect(selection);

						e.consume();
					}
				}

			}
		});

		list.addMouseListener(new MouseAdapter() {
			@Override
			public void mousePressed(MouseEvent e) {
				final Object[] selection = list.getSelectedValues();
				if (selection == null || selection.length == 0)
					return;

				if (e.isPopupTrigger() || SwingUtilities.isRightMouseButton(e)) {
					e.consume();

					JPopupMenu menu = new JPopupMenu();
					JMenuItem disconnectItem = new JMenuItem("Desconectar");
					menu.add(disconnectItem);
					disconnectItem.addActionListener(new ActionListener() {
						@Override
						public void actionPerformed(ActionEvent e) {
							doDisconnect(selection);
						}
					});

					menu.show(e.getComponent(), e.getX(), e.getY());
				}
			}
		});

		scrollPane.setViewportView(list);

		reloadConnectionList();

	}

	private void doDisconnect(final Object[] selection) {
		SlowRequester.run(new SlowRunnable() {
			@Override
			public void run() throws Exception {
				ManagedConnection.setContextCurrentConnection(cnn);
				for (Object o : selection) {
					bean.getIReceptacles().disconnect(((ConnectionItem) o).getReceptacleCnn().id);
				}
				bean.refreshNode();
			}
		}, true);
	}

	public void reloadConnectionList() {

		lblName.setText(bean.getReceptacleDescription().name);
		lblInterface.setText(bean.getReceptacleDescription().interface_name);

		DefaultListModel model = (DefaultListModel) list.getModel();
		model.clear();

		ConnectionDescription cnns[] = bean.getReceptacleDescription().connections;

		for (ConnectionDescription cnn : cnns)
			model.addElement(new ConnectionItem(cnn));

	}

}
