package tecgraf.openbus.browser.scs_offers.data_service;

import java.awt.Rectangle;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.dnd.DnDConstants;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.nio.charset.Charset;
import java.util.LinkedList;
import java.util.List;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.tree.DefaultMutableTreeNode;

import tecgraf.openbus.Connection;
import tecgraf.openbus.browser.ManagedConnection;
import tecgraf.openbus.browser.OpenbusBrowser;
import tecgraf.openbus.browser.SlowRequester;
import tecgraf.openbus.browser.SlowRequester.SlowRunnable;
import tecgraf.openbus.browser.TipPanelInterface;
import tecgraf.openbus.browser.scs_offers.AsyncExpandableTreeNode;
import tecgraf.openbus.browser.scs_offers.NodeWithDetailsInterface;
import tecgraf.openbus.browser.scs_offers.NodeWithDropBehaviorInterface;
import tecgraf.openbus.browser.scs_offers.NodeWithHelpTipInterface;
import tecgraf.openbus.browser.scs_offers.NodeWithIconsInterface;
import tecgraf.openbus.browser.scs_offers.SCSTree;
import tecgraf.openbus.browser.scs_offers.basic_nodes.FacetNodeBean;
import tecgraf.openbus.data_service.core.v1_01.DataDescription;
import tecgraf.openbus.data_service.hierarchical.v1_01.IHierarchicalManagementDataService;
import tecgraf.openbus.data_service.hierarchical.v1_01.IHierarchicalNavigationDataService;
import tecgraf.openbus.data_service.hierarchical.v1_01.IHierarchicalNavigationDataServiceHelper;
import tecgraf.openbus.data_service.hierarchical.v1_01.IHierarchicalTransferDataService;

/**
 * Tipo de n que representa um DataKey Openbus. Como ele aparecer em uma
 * rvore de HDataService, este n poder ser expandido, o que ocasionar
 * o carregamento assncrono dos filhos do n atravs de seu
 * {@link IHierarchicalNavigationDataService}.
 * <p>
 * Um n deste tipo tambm possui implementaes para o uso das APIs de
 * gesto do dataservice como IHierarchicalManagementDataService e
 * {@link IHierarchicalTransferDataService}. Tais aes podem ser feitas
 * atravs de drag-and-drop ou com o menu de contexto (popup).
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
@SuppressWarnings("serial")
final class DataKeyNodeBean extends AsyncExpandableTreeNode implements NodeWithDetailsInterface,
  NodeWithIconsInterface, Transferable, NodeWithDropBehaviorInterface, NodeWithHelpTipInterface {

	private final FacetNodeBean myFacetBean;
	private final DataDescription dataDescription;
	private final Connection cnn;
	private WeakReference<DataKeyNodeDetailsPanel> detailsPanel = null;
	
	protected DataKeyNodeBean(SCSTree tree, Connection cnn,
	  FacetNodeBean myFacetBean,
	  DataDescription dataDescription) {
		super(tree);
		this.myFacetBean = myFacetBean;
		this.dataDescription = dataDescription;
		this.cnn = cnn;
		
	}

	@Override
	protected String getLoadingDescription() {
		return "Aguardando retorno de IHierarchicalNavigationDataService.getChildren()...";
	}

	@Override
	protected List<DefaultMutableTreeNode> loadChildren() throws Exception {
		ManagedConnection.setContextCurrentConnection(cnn);
		IHierarchicalNavigationDataService hdsn =
		  IHierarchicalNavigationDataServiceHelper.narrow(myFacetBean.getFacetDescription().facet_ref);
		DataDescription roots[] = hdsn.getChildren(dataDescription.fKey);
		LinkedList<DefaultMutableTreeNode> res = new LinkedList<DefaultMutableTreeNode>();
		if (roots != null) {
			for (DataDescription ds : roots) {
				res.add(new DataKeyNodeBean(getMyTree(), cnn, myFacetBean, ds));
			}
		}

		return res;
	}

	@Override
	public String toString() {
		return dataDescription.fName;
	}

	@Override
	public JPanel getDetailsPanel() {
		DataKeyNodeDetailsPanel res = null;
		if (detailsPanel != null)
			res = detailsPanel.get();

		if (res == null) {
			res = new DataKeyNodeDetailsPanel(this);
			detailsPanel = new WeakReference<DataKeyNodeDetailsPanel>(res);
		}

		return res;
	}

	public FacetNodeBean getMyFacetBean() {
		return myFacetBean;
	}

	public DataDescription getDataDescription() {
		return dataDescription;
	}

	public Connection getCnn() {
		return cnn;
	}

	@Override
	public Object getUserObject() {
		return this;
	}

	private static final ImageIcon iconFolderEmpty = new ImageIcon(DataKeyNodeBean.class.getResource("folder.png"));
	private static final ImageIcon iconFolderFilled = new ImageIcon(
	  DataKeyNodeBean.class.getResource("folder_page_white.png"));
	private static final ImageIcon iconLeafOneDataView = new ImageIcon(
	  DataKeyNodeBean.class.getResource("page_white.png"));
	private static final ImageIcon iconLeafMoreDataViews = new ImageIcon(
	  DataKeyNodeBean.class.getResource("page_white_stack.png"));
	private static final ImageIcon iconLeafNoDataViews = new ImageIcon(
	  DataKeyNodeBean.class.getResource("document_empty.png"));

	private final int getNumDataViews() {
		if (dataDescription == null)
			return 0;
		int c = 0;
		if (dataDescription.fDefaultView != null
		  && dataDescription.fDefaultView.fInterfaceName != null
		  && !dataDescription.fDefaultView.fInterfaceName.trim().isEmpty()) {
			c += 1;
		}
		if (dataDescription.fOthersViews != null) {
			for (String v : dataDescription.fOthersViews) {
				if (v != null && !v.trim().isEmpty())
					c += 1;
			}
		}
		return c;
	}

	@Override
	public Icon getClosedIcon() {
		return getOpenedIcon();
	}

	@Override
	public Icon getLeafIcon() {
		switch (getNumDataViews()) {
			case 0:
				return iconLeafNoDataViews;
			case 1:
				return iconLeafOneDataView;
			default:
				return iconLeafMoreDataViews;
		}
	}

	@Override
	public Icon getOpenedIcon() {
		switch (getNumDataViews()) {
			case 0:
				return iconFolderEmpty;
			default:
				return iconFolderFilled;
		}
	}

	@Override
	public int accept(DataFlavor dataFlavor) {
		return DataServiceUtils.DATAKEY_FLAVOR_BIN.equals(dataFlavor) ?
		  DnDConstants.ACTION_COPY_OR_MOVE : DnDConstants.ACTION_NONE;
	}

	@Override
	public boolean doDrop(final Transferable transferable) throws Exception {

		if (!transferable.isDataFlavorSupported(DataServiceUtils.DATAKEY_FLAVOR_BIN)) {
			throw new UnsupportedFlavorException(DataServiceUtils.DATAKEY_FLAVOR_BIN);
		}

		//tecgraf.openbus.data_service.hierarchical.v1_01.IHierarchicalTransferDataServiceOperations transfer;

		SlowRequester.run(new SlowRunnable() {
			@Override
			public void run() throws Exception {

				setProgress("Lendo DataDescription do objeto trazido");
				final DataDescription theirDS = DataServiceUtils.getDataDescription(cnn, transferable);

				if (theirDS == null) {
					JOptionPane.showMessageDialog(OpenbusBrowser.getSingletonInstance(), "No foi possvel localizar " +
					  "o servio DataService responsvel pelo Datakey arrastado at aqui. Nada a fazer", "Erro",
					  JOptionPane.ERROR_MESSAGE);
					return;
				}

				// IHierarchicalManagementDataService --------------------------------------------------------------
				setProgress("Procurando IHierarchicalManagementDataService de " + dataDescription.fName);
				IHierarchicalManagementDataService myManagement =
				  DataServiceUtils.getHManagementDataServiceFor(cnn, dataDescription.fKey);
				setProgress("Procurando IHierarchicalManagementDataService de " + theirDS.fName);
				IHierarchicalManagementDataService theirManagement =
				  DataServiceUtils.getHManagementDataServiceFor(cnn, theirDS.fKey);

				JPopupMenu popup = new JPopupMenu(dataDescription.fName + "->" + theirDS.fName);
				if (myManagement != null && theirManagement != null && myManagement._is_equivalent(theirManagement)) {
					popup.add(getMenuForHManagementDataService("IHierarchicalManagementDataService", myManagement,
					  dataDescription,
					  theirDS));
				}
				else {
					if (myManagement != null)
						popup.add(getMenuForHManagementDataService(
						  "IHierarchicalManagementDataService de " + dataDescription.fName,
						  myManagement, dataDescription, theirDS));
					if (theirManagement != null)
						popup.add(getMenuForHManagementDataService("IHierarchicalManagementDataService de " + theirDS.fName,
						  theirManagement, dataDescription, theirDS));
				}

				if (popup.getSubElements() == null || popup.getSubElements().length == 0) {
					JOptionPane.showMessageDialog(OpenbusBrowser.getSingletonInstance(),
					  "Nenhuma operao elegvel encontrada.\n" +
					    "No h IHierarchicalManagementDataService disponvel para os DataKeys selecionados.", "Nada a fazer",
					  JOptionPane.INFORMATION_MESSAGE);
				}
				else {
					int x, y;
					if (getMyTree().getSelectionRows() == null || getMyTree().getSelectionRows().length != 1) {
						x = getMyTree().getMousePosition().x;
						y = getMyTree().getMousePosition().y;
					}
					else {
						int sel = getMyTree().getSelectionRows()[0];
						Rectangle bounds = getMyTree().getRowBounds(sel);
						x = (int) bounds.getCenterX();
						y = (int) bounds.getMaxY();
					}
					popup.show(getMyTree(), x, y);
				}
			}
		}, false);

		return true;
	}

	private JMenu getMenuForHManagementDataService(String label, final IHierarchicalManagementDataService management,
	  final DataDescription ds1, final DataDescription ds2) {

		JMenu res = new JMenu(label);
		res.add(getPopupItemForHMDSCopy(management, ds1, ds2));
		res.add(getPopupItemForHMDSMove(management, ds1, ds2));

		return res;
	}

	private JMenuItem getPopupItemForHMDSCopy(final IHierarchicalManagementDataService management,
	  final DataDescription ds1, final DataDescription ds2) {
		JMenuItem res = new JMenuItem("copyData(" + ds1.fName + ", " + ds2.fName + ")");
		res.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				SlowRequester.run(new SlowRunnable() {
					@Override
					public void run() throws Exception {
						ManagedConnection.setContextCurrentConnection(cnn);
						if (management.copyData(ds1.fKey, ds2.fKey) != null)
							refreshNode();
						JOptionPane.showMessageDialog(getMyTree(),
						  "copyData() executado sem lanamento de exceo.",
						  "DataService", JOptionPane.INFORMATION_MESSAGE);
					}
				}, true);
			}
		});
		return res;
	}

	private JMenuItem getPopupItemForHMDSMove(final IHierarchicalManagementDataService management,
	  final DataDescription ds1, final DataDescription ds2) {
		JMenuItem res = new JMenuItem("moveData(" + ds1.fName + ", " + ds2.fName + ")");
		res.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				SlowRequester.run(new SlowRunnable() {
					@Override
					public void run() throws Exception {
						ManagedConnection.setContextCurrentConnection(cnn);
						management.moveData(ds1.fKey, ds2.fKey);
						JOptionPane.showMessageDialog(getMyTree(),
						  "moveData() executado sem lanamento de exceo.",
						  "DataService", JOptionPane.INFORMATION_MESSAGE);
					}
				}, true);
			}
		});
		return res;
	}

	@Override
	public DataFlavor[] getTransferDataFlavors() {
		return new DataFlavor[] { DataServiceUtils.DATAKEY_FLAVOR_BIN, DataFlavor.getTextPlainUnicodeFlavor() };
	}

	@Override
	public boolean isDataFlavorSupported(DataFlavor flavor) {
		return DataServiceUtils.DATAKEY_FLAVOR_BIN.equals(flavor) || DataFlavor.getTextPlainUnicodeFlavor().equals(flavor);
	}

	@Override
	public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException {

		if (DataServiceUtils.DATAKEY_FLAVOR_BIN.equals(flavor)) {
			return dataDescription.fKey;
		}

		if (DataFlavor.getTextPlainUnicodeFlavor().equals(flavor)) {
			return new ByteArrayInputStream(dataDescription.fName.getBytes(Charset.forName(DataFlavor
			  .getTextPlainUnicodeFlavor().getParameter("charset"))));
		}

		throw new UnsupportedFlavorException(flavor);
	}

	@Override
	public void configureHelpTip(TipPanelInterface tipPanel) {
		tipPanel
		  .setTips(
		    new String[] {
		        "Para usar os servios do IHierarchicalManagementDataService, voc pode arrastar um n de DataService para outro.",

		        "Para usar os servios do IHierarchicalManagementDataService, voc pode copiar e colar ns de DataService entre si.",

		        "Se voc quiser recarregar algum n (reexecutar a chamada getChildren), basta clicar sobre o n e teclar F5."
		    }, new String[] {});
	}

}
