package tecgraf.openbus.browser.scs_offers.data_service;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;

import tecgraf.openbus.Connection;
import tecgraf.openbus.browser.scs_offers.SingleServiceFinder;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOfferDesc;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceProperty;
import tecgraf.openbus.data_service.core.v1_01.DataAccessDenied;
import tecgraf.openbus.data_service.core.v1_01.DataDescription;
import tecgraf.openbus.data_service.core.v1_01.DataKeyWrapper;
import tecgraf.openbus.data_service.core.v1_01.DataNotFound;
import tecgraf.openbus.data_service.core.v1_01.IDataService;
import tecgraf.openbus.data_service.core.v1_01.IDataServiceHelper;
import tecgraf.openbus.data_service.core.v1_01.InvalidDataKey;
import tecgraf.openbus.data_service.hierarchical.v1_01.IHierarchicalManagementDataService;
import tecgraf.openbus.data_service.hierarchical.v1_01.IHierarchicalManagementDataServiceHelper;

/**
 * Convenincias para trabalhar com o DataService.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
final class DataServiceUtils {

	public static final DataFlavor DATAKEY_FLAVOR_BIN = new DataFlavor("openbus-datakey/octet-stream", "Datakey Openbus");

	private DataServiceUtils() {
	}

	/**
	 * Recuperar a instncia de {@link IDataService} relacionada ao datakey informado.
	 * A busca no servio de ofertas  feita de forma eficiente (multithread) e
	 * tratando eventuais ambiguidades no critrio de busca deduzido do datakey.
	 * 
	 * @param cnn Conexo Openbus que ser usada nas chamadas remotas
	 * @param dataKey Datakey que ser resolvido.
	 * @return Instncia do IDataService ou null caso nada seja encontrado.
	 * @throws ServiceFailure
	 * @throws InvalidDataKey
	 * @throws InterruptedException
	 */
	public static IDataService getDataService(Connection cnn, byte[] dataKey) throws ServiceFailure,
	  InvalidDataKey,
	  InterruptedException {
		DataKeyWrapper wrapper = new DataKeyWrapper(dataKey);
		ServiceOfferDesc offer = SingleServiceFinder.findFirstOffer(cnn, new ServiceProperty[] {
		    new ServiceProperty("openbus.offer.entity", wrapper.getSystemDeploymentId()),
		    new ServiceProperty("openbus.data_source_id", wrapper.getDataSourceId()),
		    new ServiceProperty("openbus.component.interface", IDataServiceHelper.id())
		});

		if (offer == null)
			return null;

		return IDataServiceHelper.narrow(offer.service_ref.getFacet(IDataServiceHelper.id()));
	}

	/**
	 * Recuperar a instncia de {@link IHierarchicalManagementDataService} 
	 * relacionada ao datakey informado. A busca no servio de ofertas  feita 
	 * de forma eficiente (multithread) e tratando eventuais ambiguidades no 
	 * critrio de busca deduzido do datakey.
	 * 
	 * @param cnn Conexo Openbus que ser usada nas chamadas remotas
	 * @param dataKey Datakey que ser resolvido.
	 * @return Instncia do IDataService ou null caso nada seja encontrado.
	 * @throws ServiceFailure
	 * @throws InvalidDataKey
	 * @throws InterruptedException
	 */
	public static IHierarchicalManagementDataService getHManagementDataServiceFor(Connection cnn, byte[] dataKey)
	  throws ServiceFailure,
	  InterruptedException, InvalidDataKey {
		DataKeyWrapper wrapper = new DataKeyWrapper(dataKey);

		ServiceOfferDesc offer = SingleServiceFinder.findFirstOffer(cnn, new ServiceProperty[] {
		    new ServiceProperty("openbus.offer.entity", wrapper.getSystemDeploymentId()),
		    new ServiceProperty("openbus.data_source_id", wrapper.getDataSourceId()),
		    new ServiceProperty("openbus.component.interface", IHierarchicalManagementDataServiceHelper.id())
		});

		if (offer == null)
			return null;

		return IHierarchicalManagementDataServiceHelper.narrow(offer.service_ref
		  .getFacet(IHierarchicalManagementDataServiceHelper.id()));
	}

	/**
	 * Recuperar o {@link DataDescription} do dataKey informado. Isto implica na
	 * realizao de chamadas remotas para recuperao do DataDescription em questo.
	 * @param cnn Conexo Openbus a ser usada para recuperar o DataDescription.
	 * @param dataKey Datakey cujo DataDescription ser recuperado.
	 * @return Instncia do DataDescription recuperada ou null se no for encontrada.
	 * @throws ServiceFailure
	 * @throws InvalidDataKey
	 * @throws InterruptedException
	 * @throws tecgraf.openbus.data_service.core.v1_01.ServiceFailure
	 * @throws DataAccessDenied
	 * @throws DataNotFound
	 */
	public static DataDescription getDataDescription(Connection cnn, byte[] dataKey) throws ServiceFailure,
	  InvalidDataKey, InterruptedException, tecgraf.openbus.data_service.core.v1_01.ServiceFailure, DataAccessDenied,
	  DataNotFound {
		IDataService ds = getDataService(cnn, dataKey);
		if (ds == null)
			return null;
		return ds.getDataDescription(dataKey);
	}

	/**
	 * Recuperar o {@link DataDescription} do dataKey implcito no {@link DataFlavor}
	 * do {@link Transferable} informado (vide api de drag-and-drop). Isto implica na
	 * realizao de chamadas remotas para recuperao do DataDescription em questo.
	 * @param cnn Conexo Openbus a ser usada para recuperar o DataDescription.
	 * @param transferable Objeto de onde o datakey para recuperao ser extrado
	 * @return Instncia do DataDescription recuperada ou null se no for encontrada.
	 * @throws ServiceFailure
	 * @throws InvalidDataKey
	 * @throws InterruptedException
	 * @throws tecgraf.openbus.data_service.core.v1_01.ServiceFailure
	 * @throws DataAccessDenied
	 * @throws DataNotFound
	 */
	public static DataDescription getDataDescription(Connection cnn, Transferable transferable)
	  throws UnsupportedFlavorException, ServiceFailure, InvalidDataKey,
	  tecgraf.openbus.data_service.core.v1_01.ServiceFailure, DataAccessDenied, DataNotFound, InterruptedException,
	  IOException {
		if (!transferable.isDataFlavorSupported(DATAKEY_FLAVOR_BIN))
			throw new UnsupportedFlavorException(DATAKEY_FLAVOR_BIN);

		byte[] dataKey = (byte[]) transferable.getTransferData(DATAKEY_FLAVOR_BIN);

		return getDataDescription(cnn, dataKey);
	}

}
