/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.client.algorithms.parameters;

import java.awt.Component;
import java.awt.GridLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.LinkedList;
import java.util.List;

import javax.swing.ComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

import tecgraf.javautils.gui.GBC;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationParameter;
import csbase.logic.algorithms.parameters.EnumerationParameterListener;

/**
 * Viso para o parmetro do tipo enumerao.
 * 
 * @author lmoreira
 */
public final class EnumerationParameterView extends
  SimpleParameterView<EnumerationItem> {

  /**
   * Cria uma viso em modo {@link ParameterView.Mode#CONFIGURATION} para
   * parmetro do tipo enumerao.
   * 
   * @param window NO EST SENDO UTILIZADO. Existe somente para manter
   *        compatibilidade com o WebSintesi.
   * @param parameter O parmetro (No aceita {@code null}).
   * 
   * @deprecated para manter compatibilidade com o WebSintesi
   */
  @Deprecated
  public EnumerationParameterView(Window window, EnumerationParameter parameter) {
    this(parameter, Mode.CONFIGURATION);
  }

  /**
   * Cria uma viso para parmetro do tipo enumerao.
   * 
   * @param parameter O parmetro (No aceita {@code null}).
   * @param mode Modo de visualizao. No aceita {@code null}, os possveis
   *        valores so: {@link ParameterView.Mode#CONFIGURATION} ou
   *        {@link ParameterView.Mode#REPORT}.
   */
  public EnumerationParameterView(EnumerationParameter parameter, Mode mode) {
    super(parameter, mode);

    getParameter().addEnumerationParameterListener(
      new EnumerationParameterListener() {
        @Override
        public void enumerationItemWasSetVisible(EnumerationParameter param,
          EnumerationItem item) {
          updateViewContents();
        }
      });

    updateCapabilityView();
    updateVisibilyView();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EnumerationParameter getParameter() {
    return (EnumerationParameter) super.getParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createConfigurationComponent(Object... componentArgs) {
    return new EnumerationConfigurationParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createReportComponent(Object... componentArgs) {
    return new EnumerationReportParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void updateViewContents() {
    ((IEnumerationParameterComponent) getComponent()).updateViewContents();
  }

  /**
   * Interface comum s diferentes vises do parmetro.
   */
  private interface IEnumerationParameterComponent {
    /**
     * Atualiza o contedo exibido pela viso.
     */
    void updateViewContents();
  }

  /**
   * Viso do parmetro em modo {@link ParameterView.Mode#REPORT}.
   */
  private final class EnumerationReportParameter extends JTextField implements
    IEnumerationParameterComponent {

    /**
     * Construtor.
     */
    EnumerationReportParameter() {
      setToolTipText(getParameter().getDescription());
      ComponentProperties.setProperties(this, Mode.REPORT, true);
      super.setEditable(false);
      updateViewContents();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      EnumerationItem selectedItem = getParameter().getValue();
      if (selectedItem == null) {
        setToolTipText(null);
        setText(" ");
      }
      else {
        setText(selectedItem.getLabel());
        if (selectedItem.getDescription() != null) {
          setToolTipText(selectedItem.getDescription());
        }
        else {
          setToolTipText(getParameter().getDescription());
        }
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setEditable(boolean enable) {
      // Este mtodo no faz nada por que este componente no pode ser editvel.
    }
  }

  /**
   * Viso do parmetro em modo {@link ParameterView.Mode#CONFIGURATION}.
   */
  private final class EnumerationConfigurationParameter extends JPanel
    implements IEnumerationParameterComponent {

    /**
     * A combobox.
     */
    private JComboBox comboBox;

    /**
     * O modelo da combobox.
     */
    private MyComboBoxModel comboBoxModel;

    /**
     * Construtor.
     */
    EnumerationConfigurationParameter() {
      super();
      setLayout(new GridLayout());

      comboBoxModel = new MyComboBoxModel();
      comboBox = new JComboBox(comboBoxModel);
      comboBox.setToolTipText(getParameter().getDescription());
      add(comboBox, new GBC(0, 0).both());

      comboBox.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          updateModel();
        }
      });
      comboBox.setRenderer(new DefaultListCellRenderer() {
        @Override
        public Component getListCellRendererComponent(JList list, Object value,
          int index, boolean isSelected, boolean cellHasFocus) {
          super.getListCellRendererComponent(list, value, index, isSelected,
            cellHasFocus);
          EnumerationItem item = (EnumerationItem) value;
          if (item == null) {
            setToolTipText(null);
            setText("");
          }
          else {
            setText(item.getLabel());
            if (item.getDescription() != null) {
              setToolTipText(item.getDescription());
            }
            else {
              setToolTipText(getParameter().getDescription());
            }
          }
          return this;
        }
      });

      updateViewContents();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      comboBoxModel.updateFromParameter();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setEnabled(boolean isEnabled) {
      comboBox.setEnabled(isEnabled);
    }

    /**
     * Reflete as alteraes da viso para o modelo (o parmetro propriamente
     * dito).
     */
    private void updateModel() {
      EnumerationItem selectedItem =
        (EnumerationItem) comboBox.getSelectedItem();
      getParameter().setValue(selectedItem);
    }

    /**
     * O modelo da combobox.
     */
    private final class MyComboBoxModel implements ComboBoxModel {
      /**
       * Os items que esto na combobox.
       */
      private List<EnumerationItem> items;
      /**
       * Os observadores do modelo.
       */
      private List<ListDataListener> listeners;
      /**
       * O item selecionado.
       */
      private EnumerationItem selectedItem;

      /**
       * Cria o modelo.
       */
      MyComboBoxModel() {
        listeners = new LinkedList<ListDataListener>();
        items = new LinkedList<EnumerationItem>();
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public void addListDataListener(ListDataListener l) {
        synchronized (listeners) {
          listeners.add(l);
        }
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Object getElementAt(int index) {
        return items.get(index);
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Object getSelectedItem() {
        return selectedItem;
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public int getSize() {
        return items.size();
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public void removeListDataListener(ListDataListener l) {
        synchronized (listeners) {
          listeners.remove(l);
        }
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public void setSelectedItem(Object anItem) {
        selectedItem = (EnumerationItem) anItem;
        fireContentsChanged();
      }

      /**
       * Obtm os valores do parmetro e os coloca no modelo.
       */
      void updateFromParameter() {
        items.clear();
        for (EnumerationItem item : getParameter().getItems()) {
          if (item.isVisible()) {
            items.add(item);
          }
        }
        selectedItem = getParameter().getValue();
        fireContentsChanged();
      }

      /**
       * Dispara o evento
       * {@link ListDataListener#contentsChanged(ListDataEvent)}.
       */
      private void fireContentsChanged() {
        ListDataListener[] listenersArray;
        synchronized (listeners) {
          listenersArray =
            listeners.toArray(new ListDataListener[listeners.size()]);
        }
        for (ListDataListener listener : listenersArray) {
          ListDataEvent event =
            new ListDataEvent(EnumerationConfigurationParameter.this,
              ListDataEvent.CONTENTS_CHANGED, -1, -1);
          listener.contentsChanged(event);
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean fillVerticalSpace() {
    return false;
  }
}
