package csbase.client.algorithms.parameters;

import java.text.MessageFormat;
import java.util.HashSet;
import java.util.Set;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.desktop.DesktopFrame;
import csbase.client.kernel.ClientException;
import csbase.client.project.ProjectFileChooser;
import csbase.client.project.ProjectFileChooserSave;
import csbase.client.project.ProjectTreePath;
import csbase.logic.ClientProjectFile;
import csbase.logic.ProjectFileType;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.OutputFileListParameter;
import csbase.logic.algorithms.parameters.FileURLValue;

/**
 * Viso para {@link OutputFileListParameter Parmetro do Tipo Arquivo de Sada}
 * com Seleo Mltipla.
 *
 */
public class OutputFileListParameterView extends FileListParameterView {

  /**
   * Cria uma viso.
   *
   * @param parameter O parmetro (No aceita {@code null}).
   * @param mode Modo de visualizao. No aceita {@code null}, os possveis
   *        valores so: {@link ParameterView.Mode#CONFIGURATION} ou
   *        {@link ParameterView.Mode#REPORT}.
   */
  public OutputFileListParameterView(OutputFileListParameter parameter,
    Mode mode) {
    super(parameter, mode);
  }

  /**
   * Solicita os arquivos ao usurio.
   *
   * @return O conjunto de arquivos selecionados (Se o usurio cancelar a
   *         seleo, o conjunto estar vazio; se houver um erro durante a
   *         seleo, retorna {@code null}).
   */
  @Override
  protected Set<FileURLValue> askForFile() {
    try {
      Set<FileURLValue> files = new HashSet<FileURLValue>();
      ProjectFileChooserSave chooser =
        new ProjectFileChooserSave(getWindow(), DesktopFrame.getInstance()
          .getProject(), getFileChooserMode(), getParameter().getFileType(),
          null, null, getParameter().usesFilter());
      ProjectTreePath[] projectTreePaths = chooser.getSelectedPaths();
      if ((projectTreePaths == null) || (projectTreePaths.length == 0)) {
        return files;
      }
      for (ProjectTreePath projectTreePath : projectTreePaths) {
        ClientProjectFile clientProjectFile = projectTreePath.getFile();
        String type;
        if (clientProjectFile == null) {
          type = getParameter().getFileType();
        }
        else if (clientProjectFile.isDirectory()) {
          type = ProjectFileType.DIRECTORY_TYPE;
        }
        else {
          type = clientProjectFile.getType();
        }
        FileURLValue file =
          new FileURLValue(FileUtils.joinPath('/', projectTreePath.getPath()), type);
        files.add(file);
      }
      return files;
    }
    catch (ClientException e) {
      String errorMessage =
        "Ocorreu um erro ao abrir o dilogo de seleo de arquivos.";
      if (e.getLocalizedMessage() != null) {
        errorMessage += "\nDetalhes:\n";
        errorMessage += e.getLocalizedMessage();
      }
      StandardDialogs.showErrorDialog(getWindow(),
        "Erro ao abrir o dilogo de seleo de arquivo", errorMessage);
      return null;
    }
  }

  /**
   * Obtm o modo do {@link ProjectFileChooser} que ser utilizado para
   * solicitar os arquivos ao usurio.
   *
   * @return .
   */
  protected int getFileChooserMode() {
    int mode;
    FileParameterMode parameterMode = getParameter().getMode();
    if (parameterMode.equals(FileParameterMode.REGULAR_FILE)) {
      mode = ProjectFileChooser.FILE_ONLY;
    }
    else if (parameterMode.equals(FileParameterMode.DIRECTORY)) {
      mode = ProjectFileChooser.DIRECTORY_ONLY;
    }
    else {
      String errorMessage =
        MessageFormat.format("O modo {0} no  vlido.", parameterMode);
      throw new IllegalStateException(errorMessage);
    }
    return mode;
  }

}
