/*
 * $Id$
 */

package csbase.client.applications.commandsmonitor.table.column;

import java.awt.Component;
import java.rmi.RemoteException;
import java.util.Locale;

import javax.swing.DefaultCellEditor;
import javax.swing.JComboBox;
import javax.swing.JTable;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import csbase.client.algorithms.commands.cache.CommandsCache;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.RemoteTask;
import csbase.logic.CommandInfo;
import csbase.logic.Priority;
import csbase.logic.User;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Coluna contendo a prioridade do comando. <br>
 * Este valor  utilizado pelo agendador de comandos.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class PriorityColumn extends AbstractCommandsTableColumn {

  /**
   * Construtor.
   * 
   * @param id Identificador da coluna.
   * @param appRegistry registro da aplicao que ser utilizado para se obter o
   *        nome da coluna de acordo com o {@link Locale}.
   */
  public PriorityColumn(String id, ApplicationRegistry appRegistry) {
    super(Priority.class, id, appRegistry, User.getLoggedUser().isAdmin());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValue(CommandInfo cmd) {
    return cmd.getPriority();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValue(final CommandInfo cmd, final Object value) {
    Priority previousPriority = cmd.getPriority();

    String priorityName = value.toString();
    cmd.setPriority(getPriority(priorityName));

    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws RemoteException {
        setResult(CommandsCache.getInstance().updatePriority(cmd));
      }
    };
    String taskTitle = getGenericString("task.command.update.title");
    String taskMessage = getGenericString("task.command.update.message");
    boolean success = task.execute(null, taskTitle, taskMessage);
    if (!success || !task.getResult()) {
      cmd.setPriority(previousPriority);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TableCellRenderer createTableCellRenderer() {
    return new DefaultTableCellRenderer() {
      @Override
      public Component getTableCellRendererComponent(JTable table,
        Object value, boolean isSelected, boolean hasFocus, int row, int column) {
        Component comp =
          super.getTableCellRendererComponent(table, value, isSelected,
            hasFocus, row, column);

        DefaultTableCellRenderer renderer = (DefaultTableCellRenderer) comp;
        renderer.setText(getPriorityName((Priority) value));
        if (isColumnEditable()) {
          renderer.setIcon(ApplicationImages.ICON_EDITABLE_TABLE_CELL_14);
        }
        return this;
      }
    };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TableCellEditor createTableCellEditor() {
    String[] items = new String[Priority.values().length];
    for (int inx = 0; inx < items.length; inx++) {
      items[inx] = getPriorityName(Priority.values()[inx]);
    }
    JComboBox comboBox = new JComboBox(items);
    return new DefaultCellEditor(comboBox);
  }

  /**
   * Obtm uma prioridade dado seu nome internacionalizado.
   * 
   * @param name nome internacionalizado da prioridade.
   * 
   * @return uma prioridade dado seu nome internacionalizado.
   */
  private Priority getPriority(String name) {
    for (Priority aPriority : Priority.values()) {
      if (getPriorityName(aPriority).equals(name)) {
        return aPriority;
      }
    }

    // Na dvida retorna prioridade mdia para evitar erros.
    return Priority.MEDIUM;
  }

  /**
   * Obtm o nome internacionalizado de uma prioridade.
   * 
   * @param priority a prioridade.
   * 
   * @return o nome internacionalizado de uma prioridade.
   */
  private String getPriorityName(Priority priority) {
    String keySufix = String.format("value.%s", priority.name().toLowerCase());
    return getString(keySufix);
  }
}
