/*
 * $Id: Notepad.java 147516 2013-12-02 21:14:26Z isabella $
 */
package csbase.client.applications.notepad;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.nio.charset.Charset;
import java.text.MessageFormat;
import java.util.Date;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.ButtonGroup;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JRootPane;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.ScrollPaneConstants;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.Client;
import csbase.client.applications.ApplicationAboutAction;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.ApplicationProject;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.Task;
import csbase.client.project.ClientFileLock;
import csbase.client.project.ClientFileLock.LockStatus;
import csbase.client.project.ExportStringTask;
import csbase.client.util.ClientUtilities;
import csbase.client.util.SingletonFileChooser;
import csbase.client.util.charset.CharsetRadioMenu;
import csbase.client.util.charset.CharsetRadioMenuChangeListener;
import csbase.client.util.iostring.TextDealerStatusInterface;
import csbase.client.util.iostring.TextReader;
import csbase.client.util.iostring.TextWriter;
import csbase.exception.project.FileLockedException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.Utilities;

/**
 * Classe que representa a aplicao bloco de notas.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class Notepad extends ApplicationProject {

  /**
   * Classe controladora da recarga automatica do documento
   * 
   * @author Daniel
   */
  private class ReloadController {

    /** Timer de reload */
    public Timer timer;

    /** Controle de atividade/inatividade do reloader */
    public boolean isReloading;

    /** View no menu do estado do reloader */
    public JMenuItem menuItem;

    /** View no toolbar do estado do reloader */
    public JToggleButton toggleButton;

    /** Controla estado de sincrodizacao das views (evita eventos em cascata) */
    private boolean isBeingSync;

    /**
     * Cancela a recarga periodica do documento.
     */
    public void cancelReloadSchedule() {
      if (isReloading) {
        timer.cancel();
        timer = new Timer();
      }
      /* Garante a sincronia entre o estado do reloader e as suas views */
      setReloading(false);
    }

    /**
     * Altera o estado atual do recarregador. Como so h dois estados, o metodo
     * nao pede parametros.
     */
    private void changeState() {
      if (isReloading()) {
        cancelReloadSchedule();
        if (currentFile != null) {
          loadFile(currentFile);
        }
        setReadOnlyMode(false);
      }
      else {
        setReadOnlyMode(true);
        startReloadSchedule();
      }
    }

    /**
     * Informa se o reloader esta ativo, isto , se ele est agendando novas
     * recargas ou nao.
     * 
     * @return estado do reloader
     */
    public boolean isReloading() {
      return isReloading;
    }

    /**
     * Schedula uma nova acao de recarga do arquivo. Ao final de cada acao, uma
     * nova acao sera preparada, a menos que o processo seja interrompido
     */
    private void scheduleAnotherReload() {
      timer.schedule(new ReloaderTask(), Notepad.PERIOD * 1000);
    }

    /**
     * Muda o estado do reloader entre ativo / inativo, mandendo suas views
     * sincronizadas.
     * 
     * @param willBeReloading novo estado do reloader
     */
    private void setReloading(final boolean willBeReloading) {
      this.isBeingSync = true;
      this.isReloading = willBeReloading;
      this.toggleButton.setSelected(willBeReloading);
      if (willBeReloading) {
        final String text = getString("menu.file.reload.stop");
        this.toggleButton.setToolTipText(text);
        this.menuItem.setText(text);
      }
      else {
        final String text = getString("menu.file.reload.start");
        this.toggleButton.setToolTipText(text);
        this.menuItem.setText(text);
      }
      this.isBeingSync = false;
      setReadOnlyMode(willBeReloading);
    }

    /**
     * Inicia uma nova rotina de recarga periodica do arquivo
     */
    public void startReloadSchedule() {
      /* Checa se havera perda de dados */
      if (needSave) {
        final JFrame mainFrame = getApplicationFrame();
        final String appName = getName();
        final String fmt = getString("msg.warn.start.reload");
        final int answer =
          StandardDialogs.showYesNoDialog(mainFrame, appName, fmt);
        if (answer != JOptionPane.YES_OPTION) {
          cancelReloadSchedule();
          return;
        }
      }
      /* Cancela qualquer scheduling previo */
      cancelReloadSchedule();
      /* Seta o novo estado e sincroniza os views */
      setReloading(true);
      /* Faz uma primeira atualizao */
      actionReload();
      /* Comeca a schedular */
      scheduleAnotherReload();
    }

    /**
     * Construtor da classe. Prepara o controlador de reloader no estado parado.
     */
    public ReloadController() {
      isBeingSync = false;
      timer = new Timer();
      final String text = getString("menu.file.reload.start");
      menuItem = new JMenuItem(text);
      menuItem.setIcon(ApplicationImages.ICON_REFRESH_16);
      toggleButton = new JToggleButton(ApplicationImages.ICON_REFRESH_16);
      toggleButton.setToolTipText(text);
      isReloading = false;
      /* Cria listener das duas views do estado do reloader (menu e toggle) */
      toggleButton.addItemListener(new ItemListener() {
        @Override
        public void itemStateChanged(final ItemEvent e) {
          if (!isBeingSync) {
            changeState();
          }
        }
      });
      menuItem.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(final ActionEvent e) {
          if (!isBeingSync) {
            changeState();
          }
        }
      });
    }

  }

  /**
   * Classe para a tarefa de recarga do documento. Cada instancia dela
   * representa uma tarefa de recarregar o documento.
   * 
   * @author daniel
   */
  class ReloaderTask extends TimerTask {
    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      if (reloadController.timer != null && reloadController.isReloading) {
        try {
          SwingThreadDispatcher.invokeAndWait(new Runnable() {
            @Override
            public void run() {
              actionReload();
            }
          });
        }
        catch (final InterruptedException e) {
          /* Nao faca nada, apenas agende um novo reload */
        }
        catch (final InvocationTargetException e) {
          /* Nao faca nada, apenas agende um novo reload */
        }
        if (reloadController.isReloading()) {
          reloadController.scheduleAnotherReload();
        }
      }
    }
  }

  /** Tempo mximo de espera pelo lock */
  private static final int WAIT_LOCK = 5000;

  /**
   * rea de texto
   */
  private JTextArea textArea = null;

  /**
   * Flag de arquivo alterado
   */
  private boolean needSave = false;

  /**
   * Flag indicando se o arquivo foi aberto em modo de leitura apenas.
   */
  private boolean readOnlyFile = false;

  /**
   * Arquivo corrente
   */
  private ClientProjectFile currentFile = null;

  /**
   * Gerenciador de reloads
   */
  private final ReloadController reloadController;

  /**
   * Periodo entre recargas (em segundos)
   */
  private static final int PERIOD = 20;

  /**
   * Tempo em que a mensagem de sucesso na gravao do arquivo fica visvel na
   * statusBar
   */
  private static final int SUCCESS_MSG_TIMEOUT = 5;

  /**
   * Painel com a rea de texto e com o painel de procura de textos
   */
  private JPanel textAreaPanel;

  /**
   * ScrollPane para a rea de texto
   */
  private JScrollPane textScrollPane;

  /**
   * Painel de procura de textos
   */
  private SearchPanel searchPanel;

  /**
   * Boto de salvar.
   */
  private JButton saveButton;

  /**
   * Item de menu de salvar
   */
  private JMenuItem saveItem;

  /**
   * Menu de escolha de encoding
   */
  final private CharsetRadioMenu charsetMenu = new CharsetRadioMenu();

  /**
   * Tamanho da fonte em porcentagem do tamanho original
   */
  private float fontSizePercent = 100;

  /**
   * Tamanho da fonte
   */
  private static final int ORIGINAL_FONT_SIZE = 12;

  /**
   * Fonte fixed
   */
  private static final Font FIXED_SIZE_FONT = new Font("Monospaced",
    Font.PLAIN, Notepad.ORIGINAL_FONT_SIZE);

  /**
   * Fonte normal
   */
  private static final Font NORMAL_FONT = new Font("Verdana", Font.PLAIN,
    Notepad.ORIGINAL_FONT_SIZE);

  /** Lock do arquivo */
  private ClientFileLock clientFileLock;

  /**
   * Ao para o CTRL+F, reutilizada para o menu 'Editar - Procurar'
   */
  private final AbstractAction searchAction = new AbstractAction(
    getString("menu.edit.find")) {
    @Override
    public void actionPerformed(final ActionEvent e) {
      if (!searchPanel.isVisible()) {
        searchPanel.setVisible(true);
      }
      else {
        searchPanel.editSearchString();
      }
    }
  };

  /**
   * Ao para o F3, reutilizada para o menu 'Editar - Procurar Novamente'
   */
  private final AbstractAction searchAgainAction = new AbstractAction(
    getString("menu.edit.findAgain")) {
    @Override
    public void actionPerformed(final ActionEvent e) {
      searchPanel.searchAgain();
    }
  };

  /**
   * Charset corrente a ser usado no programa.
   */
  private Charset currentCharset;

  /**
   * Pede o lock exclusivo para o arquivo.
   * 
   * @param file arquivo sobre o qual ser obtido o lock
   */
  private void acquireExclusiveLock(final ClientProjectFile file) {
    if (file.equals(currentFile)) {
      if (null != clientFileLock
        && clientFileLock.getLockStatus() == LockStatus.LOCK_EXCLUSIVE) {
        return;
      }
      removeLock();
    }
    else {
      removeLock();
    }

    clientFileLock =
      ClientFileLock.acquireExclusiveLock(getApplicationFrame(), file,
        Notepad.WAIT_LOCK);
  }

  /**
   * Abertura de arquivo.
   */
  private void actionNew() {
    if (canCloseFile()) {
      resetFile();
    }
  }

  /**
   * Abertura de arquivo.
   */
  private void actionOpen() {
    try {
      final String[] fileTypesArray = getAssociatedFileTypes();
      final ClientProjectFile newFile =
        browseFileOpen(fileTypesArray, fileTypesArray[0]);
      if (newFile == null) {
        return;
      }
      final String newFileType = newFile.getType();
      final boolean logFile = looksLikeLogFile(newFileType);
      openFile(newFile, logFile);
      if (logFile) {
        reloadController.startReloadSchedule();
        reloadController.toggleButton.setSelected(true);
      }
    }
    catch (final Exception e) {
      handleError(e);
    }
  }

  /**
   * Recarga do arquivo corrente
   */
  private void actionReload() {
    if (currentFile != null) {
      final int caretPosition = textArea.getCaretPosition();
      loadFile(currentFile);
      final int textLength = textArea.getText().length();
      if (textLength > 0 && caretPosition > 0) {
        if (caretPosition < textLength) {
          textArea.setCaretPosition(caretPosition);
        }
        else {
          textArea.setCaretPosition(textLength - 1);
        }
      }
    }
  }

  /**
   * Gravao de arquivo.
   * 
   * @return o status da operao
   */
  private boolean actionSave() {
    try {
      if (currentFile == null) {
        final String[] fileTypesArray = getAssociatedFileTypes();
        final ClientProjectFile chosenFile = browseFileSave(fileTypesArray[0]);
        if (chosenFile != null) {
          saveFileAs(chosenFile);
          return true;
        }
        return false;
      }
      saveFileAs(currentFile);
      return true;
    }
    catch (final Exception e) {
      handleError(e);
      return false;
    }
  }

  /**
   * Monta o array de cdigos de tipos associados ao aplicativo.
   * 
   * @return array de cdigos de tipo.
   */
  private String[] getAssociatedFileTypes() {
    final List<String> fileTypes = getFileTypes();
    if (fileTypes.size() == 0) {
      final String err = "no file types linked to notepad!";
      throw new RuntimeException(err);
    }
    final String[] preArray = new String[fileTypes.size()];
    final String[] fileTypesArray = fileTypes.toArray(preArray);
    return fileTypesArray;
  }

  /**
   * Abertura de arquivo.
   */
  private void actionSaveAs() {
    try {
      final String[] fileTypesArray = getAssociatedFileTypes();
      final ClientProjectFile newFile = browseFileSave(fileTypesArray[0]);
      if (newFile == null) {
        return;
      }
      saveFileAs(newFile);
    }
    catch (final Exception e) {
      handleError(e);
    }
  }

  /**
   * Cria menu 'Editar'
   * 
   * @return o menu 'Editar'
   */
  private JMenu buildEditMenu() {
    final JMenu menu = new JMenu(getString("menu.edit"));

    JMenuItem item = new JMenuItem(searchAction);
    item.setIcon(ApplicationImages.ICON_FIND_16);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F,
      InputEvent.CTRL_DOWN_MASK));
    menu.add(item);

    item = new JMenuItem(searchAgainAction);
    item.setIcon(ApplicationImages.ICON_FIND_16);
    item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F3, 0));
    menu.add(item);
    return menu;
  }

  /**
   * Montagem do menu de arquivos.
   * 
   * @return o menu de arquivos.
   */
  private JMenu buildFileMenu() {
    final JMenu fileMenu = new JMenu(getString("menu.file"));
    final JMenuItem openItem = fileMenu.add(getString("menu.file.open"));
    openItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        actionOpen();
      }
    });
    openItem.setIcon(ApplicationImages.ICON_OPEN_16);

    fileMenu.add(reloadController.menuItem);

    final JMenuItem closeItem = fileMenu.add(getString("menu.file.close"));
    closeItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        actionNew();
      }
    });
    closeItem.setIcon(ApplicationImages.ICON_CLOSE_16);

    saveItem = fileMenu.add(getString("menu.file.save"));
    saveItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        actionSave();
      }
    });
    saveItem.setIcon(ApplicationImages.ICON_SAVE_16);

    final JMenuItem saveAsItem = fileMenu.add(getString("menu.file.save.as"));
    saveAsItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        actionSaveAs();
      }
    });
    saveAsItem.setIcon(ApplicationImages.ICON_SAVEAS_16);

    fileMenu.addSeparator();

    final JMenuItem exportItem = fileMenu.add(getString("menu.file.export"));
    exportItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        export();
      }
    });
    exportItem.setIcon(ApplicationImages.ICON_BLANK_16);

    fileMenu.addSeparator();

    fileMenu.add(new ApplicationExitAction(this));
    return fileMenu;
  }

  /**
   * Cria menu 'Fonte'
   * 
   * @return o menu 'Fonte'
   */
  private JMenu buildFontMenu() {
    final JMenu menu = new JMenu(getString("menu.config.font"));

    final JRadioButtonMenuItem fixed =
      new JRadioButtonMenuItem(new AbstractAction(
        getString("menu.config.font.fixed")) {
        @Override
        public void actionPerformed(final ActionEvent e) {
          modifyFontType(true);
        }
      });
    menu.add(fixed);

    final JRadioButtonMenuItem normal =
      new JRadioButtonMenuItem(new AbstractAction(
        getString("menu.config.font.normal")) {
        @Override
        public void actionPerformed(final ActionEvent e) {
          modifyFontType(false);
        }
      });
    menu.add(normal);

    final ButtonGroup g = new ButtonGroup();
    g.add(fixed);
    g.add(normal);
    fixed.setSelected(true);

    return menu;
  }

  /**
   * Montagem do dilogo principal.
   */
  private void buildFrame() {
    final JFrame mainFrame = getApplicationFrame();
    mainFrame.setJMenuBar(buildMenuBar());
    mainFrame.getContentPane().add(buildPanel());
    mainFrame.setSize(new Dimension(800, 600));

    getApplicationFrame().getStatusBar().showStatusBar();
  }

  /**
   * Montagem do menu de ajuda
   * 
   * @return um menu de about.
   */
  private JMenu buildHelpMenu() {
    final JMenu helpMenu = new JMenu(getString("menu.help"));
    helpMenu.add(new ApplicationAboutAction(this));
    return helpMenu;
  }

  /**
   * Montagem do menu de ajuda
   * 
   * @return um menu de about.
   */
  private JMenu buildConfigMenu() {
    charsetMenu.addCharsetChangeListener(new CharsetRadioMenuChangeListener() {
      @Override
      public void charsetChanged(final Charset charset) {
        currentCharset = charset;
        actionReload();
      }
    });

    final JMenu menu = new JMenu(getString("menu.config"));
    menu.add(buildFontMenu());
    charsetMenu.setText(getString("menu.config.charset"));
    menu.add(charsetMenu);
    return menu;
  }

  /**
   * Montagem do menu.
   * 
   * @return o menu do programa.
   */
  private JMenuBar buildMenuBar() {
    final JMenuBar menuBar = new JMenuBar();
    menuBar.add(buildFileMenu());
    menuBar.add(buildEditMenu());
    menuBar.add(buildConfigMenu());

    menuBar.add(buildHelpMenu());
    return menuBar;
  }

  /**
   * montagem do painel principal.
   * 
   * @return o painel.
   */
  private JPanel buildPanel() {
    final JPanel panel = new JPanel();
    panel.setLayout(new BorderLayout());
    panel.add(buildToolBar(), BorderLayout.NORTH);
    panel.add(buildTextArea(), BorderLayout.CENTER);
    return panel;
  }

  /**
   * Montagem da rea de texto.
   * 
   * @return a rea de rolamento.
   */
  private JPanel buildTextArea() {

    // painel novo que engloba a rea de texto e um painel de busca, esse ltimo
    // muitas vezes ficando invisvel
    textAreaPanel = new JPanel(new GridBagLayout());

    // rea de texto mesmo
    textArea = new JTextArea();
    textArea.setEditable(true);
    textArea.setFont(Notepad.FIXED_SIZE_FONT);
    textArea.setTabSize(4);
    ClientUtilities.applyUndoRedoActions(textArea);

    // listener para aumentar/diminuir fonte:
    textArea.addMouseWheelListener(new MouseWheelListener() {
      @Override
      public void mouseWheelMoved(final MouseWheelEvent e) {
        if ((e.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) != 0) {
          if (e.getScrollType() == MouseWheelEvent.WHEEL_UNIT_SCROLL) {
            final int totalScrollAmount = e.getUnitsToScroll();
            modifyFontSize(totalScrollAmount);
          }
        }
        else {
          textScrollPane.getMouseWheelListeners()[0].mouseWheelMoved(e);
        }
      }
    });

    textArea.getDocument().addDocumentListener(new DocumentListener() {
      @Override
      public void changedUpdate(final DocumentEvent evt) {
      }

      @Override
      public void insertUpdate(final DocumentEvent evt) {
        needSave = true;
        indicateModification();
      }

      @Override
      public void removeUpdate(final DocumentEvent evt) {
        needSave = true;
        indicateModification();
      }
    });

    textScrollPane = new JScrollPane(textArea);
    textScrollPane
      .setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
    final Dimension scrDim = new Dimension(600, 400);
    textScrollPane.setSize(scrDim);
    textScrollPane.setPreferredSize(scrDim);
    textScrollPane.setMinimumSize(scrDim);

    // painel de busca:
    searchPanel = new SearchPanel(this, textArea);
    textAreaPanel.add(searchPanel, new GBC().horizontal());
    textAreaPanel.add(textScrollPane, new GBC(0, 1).both());
    searchPanel.setVisible(false);

    return textAreaPanel;
  }

  /**
   * Montagem da toolbar.
   * 
   * @return o painel.
   */
  private JToolBar buildToolBar() {
    final String appName = getName();
    final JToolBar toolBar = new JToolBar(appName);
    toolBar.setFloatable(false);

    final JButton openButton = new JButton(ApplicationImages.ICON_OPEN_16);
    openButton.setToolTipText(getString("menu.file.open"));
    openButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        actionOpen();
      }
    });

    saveButton = new JButton(ApplicationImages.ICON_SAVE_16);
    saveButton.setToolTipText(getString("menu.file.save"));
    saveButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        actionSave();
      }
    });

    final JButton saveAsButton = new JButton(ApplicationImages.ICON_SAVEAS_16);
    saveAsButton.setToolTipText(getString("menu.file.save.as"));
    saveAsButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        actionSaveAs();
      }
    });

    final JButton closeButton = new JButton(ApplicationImages.ICON_CLOSE_16);
    closeButton.setToolTipText(getString("menu.file.close"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        actionNew();
      }
    });

    final JButton cutButton = new JButton(ApplicationImages.ICON_CUT_16);
    cutButton.setToolTipText(getString("menu.edit.cut"));
    cutButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
      }
    });

    final JButton copyButton = new JButton(ApplicationImages.ICON_COPY_16);
    copyButton.setToolTipText(getString("menu.edit.copy"));
    copyButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
      }
    });

    final JButton pasteButton = new JButton(ApplicationImages.ICON_PASTE_16);
    pasteButton.setToolTipText(getString("menu.edit.paste"));
    pasteButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
      }
    });

    final JButton pfsButton =
      new JButton(ApplicationImages.ICON_PREFERENCES_16);
    pfsButton.setToolTipText(getString("menu.edit.prefs"));
    pfsButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
      }
    });

    final JButton aboutButton = new JButton(new ApplicationAboutAction(this));
    aboutButton.setText(null);

    toolBar.add(openButton);
    toolBar.add(reloadController.toggleButton);
    toolBar.add(closeButton);
    toolBar.add(saveButton);
    toolBar.add(saveAsButton);
    toolBar.addSeparator();
    // toolBar.add(cutButton);
    // toolBar.add(copyButton);
    // toolBar.add(pasteButton);
    // toolBar.addSeparator();
    // toolBar.add(pfsButton);
    // toolBar.addSeparator();
    final JButton button = new JButton(ApplicationImages.ICON_FIND_16);
    button.addActionListener(searchAction);
    button.setToolTipText(getString("menu.edit.find"));
    toolBar.add(button);
    toolBar.addSeparator();

    toolBar.add(aboutButton);
    return toolBar;
  }

  /**
   * Verifica se o arquivo foi modificado e no salvo; em caso positivo,
   * confirma com o usurio o fechamento com perda das modificaes.
   * 
   * @return um indicativo de possibilidade de fechamento do arquivo.
   */
  private boolean canCloseFile() {
    boolean result = true;
    if (needSave) {
      final String noName = getString("label.no.name");
      final String fileName =
        (currentFile != null) ? currentFile.getName() : noName;

      final String fmt = getString("msg.close.without.save");
      final MessageFormat formatter = new MessageFormat(fmt);
      final Object[] arg = new Object[] { fileName };
      final String confirmMessage = formatter.format(arg);
      final JFrame mainFrame = getApplicationFrame();
      final String appName = getName();
      final int answer =
        StandardDialogs.showYesNoCancelDialog(mainFrame, appName,
          confirmMessage);

      switch (answer) {
        case JOptionPane.YES_OPTION:
          result = actionSave();
          break;
        case JOptionPane.NO_OPTION:
          result = true;
          break;
        case JOptionPane.CANCEL_OPTION:
          result = false;
          break;
        case JOptionPane.CLOSED_OPTION:
          result = false;
          break;
      }
    }
    if (result) {
      removeLock();
    }
    return result;
  }

  /**
   * Inicializa teclas de atalho para o mecanismo de busca.
   */
  private void configHotKeys() {
    // inserir aes de teclas que sejam ativadas com o foco em qualquer
    // componente da janela:
    final JRootPane jRootPane = getApplicationFrame().getRootPane();
    final InputMap inputMap =
      jRootPane.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
    final ActionMap aMap = jRootPane.getActionMap();

    // CTRL + S: salvar arquivo
    final KeyStroke ctrlS =
      KeyStroke.getKeyStroke(KeyEvent.VK_S, InputEvent.CTRL_DOWN_MASK);
    inputMap.put(ctrlS, ctrlS.toString());
    aMap.put(ctrlS.toString(), new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        if (needSave) {
          actionSave();
        }
      }
    });

    // CTRL + F: mostrar painel de procura de textos
    final KeyStroke ctrlF =
      KeyStroke.getKeyStroke(KeyEvent.VK_F, InputEvent.CTRL_DOWN_MASK);
    inputMap.put(ctrlF, ctrlF.toString());
    aMap.put(ctrlF.toString(), searchAction);

    // F3 para continuar search !
    final KeyStroke f3 = KeyStroke.getKeyStroke(KeyEvent.VK_F3, 0);
    inputMap.put(f3, f3.toString());
    aMap.put(f3.toString(), searchAgainAction);

    // ESC esconde painel de busca [se estiver visvel]
    final KeyStroke esc = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);
    inputMap.put(esc, esc.toString());
    aMap.put(esc.toString(), new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        if (searchPanel.isVisible()) {
          searchPanel.setVisible(false);
        }
      }
    });
  }

  /**
   * Exporta o texto atual para um arquivo na mquina local do usurio.
   */
  protected void export() {

    final JFileChooser chooser = SingletonFileChooser.getInstance();
    chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
    chooser.setMultiSelectionEnabled(false);
    final Window window = getApplicationFrame();
    final int result = chooser.showSaveDialog(window);
    if (result != JFileChooser.APPROVE_OPTION) {
      return;
    }
    final File destination = chooser.getSelectedFile();

    // ok, arquivo de destino escolhido ... exportar o contedo do JTextArea:
    final String content = textArea.getText();
    final ExportStringTask task =
      new ExportStringTask(destination, content, window);
    // OBS: janela de erro j  mostrada de dentro da task ...
    task.execute(window, getString("exportAction.title"),
      getString("exportAction.msg"));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final protected boolean handleError(final Exception exception) {
    final String appName = getName();
    final JFrame mainFrame = getApplicationFrame();
    final String msg = getString("internal_error.stack.details");
    if (StandardDialogs.showYesNoDialog(mainFrame, appName, msg) == 0) {
      showException(getName(), exception);
    }
    return true;
  }

  /**
   * Indica que o arquivo est modificado.
   */
  private void indicateModification() {
    if (currentFile == null) {
      setTitle(null);
    }
    else {
      setTitle(currentFile.getName());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public void killApplication() {
    try {
      reloadController.cancelReloadSchedule();
      getApplicationFrame().getStatusBar().shutdownTimer();
      if (currentFile != null) {
        currentFile.close(true);
      }
    }
    catch (final Exception e) {
      handleError(e);
    }
  }

  /**
   * Carrega o contedo do arquivo.
   * 
   * @param file arquivo a ser carregado.
   */
  private void loadFile(final ClientProjectFile file) {
    final ApplicationFrame frame = getApplicationFrame();
    try {
      final String content =
        TextReader.readAllWithTask(frame, file, currentCharset);
      if (content == null) {
        resetFile();
        return;
      }
      currentFile = file;
      textArea.setText(content);
      needSave = false;
      setTitle(currentFile.getName());
      textArea.setCaretPosition(0);
    }
    catch (final OutOfMemoryError o) {
      showError(String.format(getString("msg.error.no.memory"), file.getName()));
      resetFile();
    }
    catch (Exception e) {
      resetFile();
    }
  }

  /**
   * Aumento do tamanho do fonte
   * 
   * @param addSize incremento
   */
  protected void modifyFontSize(final int addSize) {
    if (fontSizePercent + addSize > 200) {
      fontSizePercent = 200;
    }
    else if (fontSizePercent + addSize < 75) {
      fontSizePercent = 75;
    }
    else {
      fontSizePercent += addSize;
    }
    final Font font = textArea.getFont();
    final float newFontSize =
      Notepad.ORIGINAL_FONT_SIZE * (fontSizePercent / 100f);
    textArea.setFont(font.deriveFont(newFontSize));
  }

  /**
   * Modificao do fonte
   * 
   * @param fixed indicativo de fixed
   */
  protected void modifyFontType(final boolean fixed) {
    final float fontSize =
      Notepad.ORIGINAL_FONT_SIZE * (fontSizePercent / 100f);
    if (fixed) {
      textArea.setFont(Notepad.FIXED_SIZE_FONT.deriveFont(fontSize));
    }
    else {
      textArea.setFont(Notepad.NORMAL_FONT.deriveFont(fontSize));
    }
  }

  /**
   * Abertura de arquivo.
   * 
   * @param file arquivo a ser aberto.
   * @param readOnly true se o arquivo deve ser aberto apenas para leitura
   */
  private void openFile(final ClientProjectFile file, final boolean readOnly) {
    if (file == null) {
      showError(getString("msg.error.open.null"));
      return;
    }
    if (file.isDirectory()) {
      showError(getString("msg.error.open.directory"));
      return;
    }
    readOnlyFile = readOnly;
    CommonClientProject project = DesktopFrame.getInstance().getProject();
    if (!project.userHasAccessRW(User.getLoggedUser().getId())) {
      readOnlyFile = true;
    }
    if (!readOnlyFile) {
      acquireExclusiveLock(file);
      if (clientFileLock == null
        || clientFileLock.getLockStatus() != LockStatus.LOCK_EXCLUSIVE) {
        final int option =
          StandardDialogs.showYesNoDialog(getApplicationFrame(), getName(),
            String.format(getString("Notepad.open.readonly.msg"), file
              .getName()));
        if (option == JOptionPane.YES_NO_OPTION) {
          removeLock();
          openFile(file, true);
        }
        else {
          resetFile();
        }
        return;
      }
    }
    setReadOnlyMode(readOnlyFile);
    loadFile(file);
  }

  /**
   * Remove o lock do arquivo.
   */
  private void removeLock() {
    if (clientFileLock == null) {
      return;
    }
    clientFileLock.releaseLock(getApplicationFrame());
    clientFileLock = null;
  }

  /**
   * Mtodo para resetar o editor.
   */
  private void resetFile() {
    reloadController.cancelReloadSchedule();
    textArea.setText("");
    setReadOnlyMode(false);
    readOnlyFile = false;
    currentFile = null;
    needSave = false;
    setTitle(null);
    clientFileLock = null;
  }

  /**
   * Salvamento de arquivo.
   * 
   * @param file arquivo a ser salvo.
   */
  private void saveFileAs(final ClientProjectFile file) {
    if (file == null) {
      showError(getString("msg.error.save.null"));
      return;
    }
    if (file.isDirectory()) {
      showError(getString("msg.error.save.directory"));
      return;
    }
    acquireExclusiveLock(file);

    final LockStatus lockStatus =
      (clientFileLock == null) ? null : clientFileLock.getLockStatus();
    if (lockStatus != LockStatus.LOCK_EXCLUSIVE) {
      showInformation(String.format(getString("Notepad.no.lock.msg"), file
        .getName()));
      return;
    }

    final String content = textArea.getText();
    final ApplicationFrame frame = getApplicationFrame();
    final Task<Void> task = new Task<Void>() {
      @Override
      protected void handleError(final Exception error) {
        if (error instanceof FileLockedException) {
          StandardDialogs.showErrorDialog(frame, getName(), String.format(
            getString("Notepad.file.locked.error"), file.getName()));
        }
        else {
          super.handleError(error);
        }
      }

      @Override
      public void performTask() throws Exception {
        final TextDealerStatusInterface statusDealer =
          new TextDealerStatusInterface() {
            @Override
            public void setTextOperationPercentage(int perc) {
              setProgressStatus(perc);
            }

            @Override
            public boolean isTextOperationInterrupted() {
              return wasCancelled();
            }
          };
        setProgressStatus(10);
        TextWriter.writeAllWithoutTask(file, content, currentCharset,
          statusDealer);
        currentFile = file;

        // Anexa a descrio ao arquivo
        final String date = Utilities.getFormattedDate((new Date()).getTime());
        final String userName = User.getLoggedUser().getName();
        final String userLogin = User.getLoggedUser().getLogin();
        final MessageFormat formatter =
          new MessageFormat(getString("msg.saved.description"));
        final String appName = getName();
        final Object[] arg = { date, appName, userLogin, userName };
        final String message = formatter.format(arg);
        file.appendDescription(message + "\n");
      }
    };

    final String saveMsg = getString("msg.saving.file");
    if (task.execute(frame, getName(), saveMsg, true, false)) {
      needSave = false;
      setTitle(file.getName());
      getApplicationFrame().getStatusBar().setInfo(
        getString("msg.success.save"), SUCCESS_MSG_TIMEOUT);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void sendMessage(final String name, final Object value, String senderId) {
    if (value == null) {
      return;
    }
    if (name.equals(ApplicationProject.PROJECT_FILE_MESSAGE)) {
      final ClientProjectFile file = (ClientProjectFile) value;
      final String fileType = file.getType();
      final boolean logFile = looksLikeLogFile(fileType);
      openFile(file, logFile);
      if (logFile) {
        reloadController.startReloadSchedule();
        reloadController.toggleButton.setSelected(true);
      }
    }
  }

  /**
   * Indica se um cdigo de tipo parece ser de log.
   * 
   * @param fileType tipo (cdigo) de arquivo
   * @return indicativo de ele se assemelha a um tipo de representa log.
   */
  private boolean looksLikeLogFile(final String fileType) {
    if (fileType == null) {
      return false;
    }
    final boolean logFile = fileType.equals("LOG");
    return logFile;
  }

  /**
   * Ativa / desativa modo leitura-somente.
   * 
   * @param activate Se true, ativa o modo de leitura-somente
   */
  private void setReadOnlyMode(final boolean activate) {
    if (readOnlyFile || activate) {
      textArea.setEditable(false);
      saveButton.setEnabled(false);
      saveItem.setEnabled(false);
      textArea.setBackground(Color.lightGray);
    }
    else {
      textArea.setEditable(true);
      saveButton.setEnabled(true);
      saveItem.setEnabled(true);
      textArea.setBackground(Color.white);
    }
  }

  /**
   * Atualiza o ttulo da janela, inserindo o nome do arquivo atual.
   * 
   * @param title - se NULL, consideramos "nenhum arquivo aberto". Seno, o nome
   *        do arquivo aberto  inserido no ttulo.
   */
  private void setTitle(final String title) {
    String filename = title;
    getApplicationFrame().getStatusBar().clearStatus();
    if (filename == null) {
      filename = getName(); // apenas o nome da aplicao
      // ["Notepad"]
    }
    else {
      filename = getName() + " - " + filename;
    }

    if (needSave) {
      filename += "*";
    }
    else if (!textArea.isEditable()) {
      filename += " [" + getString("title.read.only") + "]";
    }

    getApplicationFrame().setTitle(filename);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public boolean userCanKillApplication() {
    return canCloseFile();
  }

  /**
   * Construtor padro.
   * 
   * @param id o identificador da aplicao.
   */
  public Notepad(final String id) {
    super(id);
    final Client client = Client.getInstance();
    this.currentCharset = client.getSystemDefaultCharset();
    this.reloadController = new ReloadController();
    buildFrame();
    configHotKeys();
  }

}
