/*
 * $Id: ProjectFileNameAndTypeFilter.java 112481 2010-11-10 20:30:22Z clinio $
 */
package csbase.client.desktop;

import java.util.regex.Pattern;

import csbase.logic.ClientProjectFile;
import csbase.logic.ProjectFileFilter;

/**
 * Filtro para arquivos de projeto, incluindo nome E tipo de arquivo. Este
 * filtro  <i>Case-Insensitive</i>.
 * <p>
 * Os filtros digitados pelo usurio, que so argumentos para o construtor, no
 * so expresses regulares mas uma simplificao convencionada para permitir
 * flexibilidade sem adicionar a complexidade das expresses regulares. Alm do
 * caracteres alfanumricos, os caracteres especiais reconhecidos pelo filtro
 * so:
 * <ol>
 * <li>'<b>?</b>': Aceita um caracter alfanumrico qualquer;</li>
 * <li>'<b>*</b>': Aceita qualquer nmero de caracteres alfanumricos;</li>
 * <li>'<b>$</b>': Marca o fim do arquivo; por <i>default</i>, a classe
 * considera que o filtro entrado  sempre o <u>incio</u> do filtro, isto ,
 * aceita qualquer nmero de caracteres alfanumricos aps aqueles digitados
 * pelo usurio. Se o caracter '$' for entrado, o filtro no aceitar nenhum
 * caracter alm daqueles digitados pelo usurio.</li>
 * </ol>
 * 
 * @author Tecgraf
 */
public class ProjectFileNameAndTypeFilter implements ProjectFileFilter {

  /** Expresso regular a ser comparada com o nome do arquivo. */
  private final Pattern namePattern;
  /** Tipo de arquivos permitidos pelo filtro. */
  private final String fileType;
  /**
   * No caso de um diretrio, verifica se algum dos arquivos filhos so aceitos
   * pelo filtro tambm.
   */
  private boolean checkDescendant = true;

  /**
   * Cria o filtro.
   * 
   * @param name filtro de nome definido pelo usurio. Nulos no so permitidos.
   *        Se um valor vazio for passado, o filtro permitir quaisquer nomes de
   *        arquivos.
   * @param typeCode tipo de arquivo aceito, tambm escolhido pelo usurio. Se
   *        um valor nulo for passado, o filtro permitir qualquer tipo de
   *        arquivo.
   */
  public ProjectFileNameAndTypeFilter(final String name, final String typeCode) {
    this(name, typeCode, true);
  }

  /**
   * Cria o filtro.
   * 
   * @param theName filtro de nome definido pelo usurio. Nulos no so
   *        permitidos. Se um valor vazio for passado, o filtro permitir
   *        quaisquer nomes de arquivos.
   * @param typeCode tipo de arquivo aceito, tambm escolhido pelo usurio. Se
   *        um valor nulo for passado, o filtro permitir qualquer tipo de
   *        arquivo.
   * @param checkDescendant se esse valor for true e o arquivo  um diretrio,
   *        verifica-se se algum dos arquivos filhos  aceito pelo filtro
   *        tambm. Se for false, apenas o arquivo corrente  verificado. O
   *        valor default  true.
   */
  public ProjectFileNameAndTypeFilter(final String theName,
    final String typeCode, final boolean checkDescendant) {
    if (theName == null) {
      throw new IllegalArgumentException("name == null");
    }
    String name = theName;
    name = name.replaceAll("\\.", "\\\\.");
    name = name.replaceAll("\\?", "\\.");
    name = name.replaceAll("\\*", "\\.\\*");
    if ((name.isEmpty() || name.charAt(name.length() - 1) != '*')
      && (name.indexOf('$') == -1)) {
      name += ".*";
    }
    namePattern = Pattern.compile(name, Pattern.CASE_INSENSITIVE);
    fileType = typeCode;
    this.checkDescendant = checkDescendant;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean accept(final ClientProjectFile file) {
    if (nameMatches(file) && typeMatches(file)) {
      return true;
    }
    if (checkDescendant && file.isDirectory()) {
      final ClientProjectFile[] children = file.getLocalChildren();
      if (children == null) {
        /*
         * a lista de filhos ainda no foi obtida do servidor
         */
        return false;
      }
      for (final ClientProjectFile child : children) {
        if (accept(child)) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * Verifica se o nome do arquivo  aceito pelo filtro.
   * 
   * @param file arquivo sendo verificado.
   * 
   * @return true se o nome do arquivo for aceito pelo filtro.
   */
  private boolean nameMatches(final ClientProjectFile file) {
    return namePattern.matcher(file.getName()).matches();
  }

  /**
   * Verifica se o tipo do arquivo  aceito pelo filtro.
   * 
   * @param file arquivo sendo verificado.
   * 
   * @return true se o tipo do arquivo for aceito pelo filtro. *
   */
  private boolean typeMatches(final ClientProjectFile file) {
    return fileType == null || file.getType().equals(fileType);
  }
}