package csbase.client.util.table;

import java.awt.Color;
import java.util.LinkedList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.UIManager;
import javax.swing.border.Border;

/**
 * <p>
 * Viso de Clula Abstrata: esta classe contm implementaes para alguns
 * mtodos da interface {@link CellView}, portanto ela facilita a implementao
 * da interface {@link CellView}.
 * </p>
 */
public abstract class AbstractCellView implements CellView {
  /**
   * A cor da borda de erro.
   */
  private static final Color ERROR_BORDER_COLOR = Color.RED;

  /**
   * A espessura da borda de erro.
   */
  private static final int ERROR_BORDER_THICKNESS = 2;

  /**
   * Indica se est em modo de edio.
   */
  boolean isEditing;

  /**
   * Os observadores.
   */
  private List<CellViewListener> listeners;

  /**
   * Cria a viso de clula abstrata.
   */
  protected AbstractCellView() {
    listeners = new LinkedList<CellViewListener>();
    addCellViewListener(new CellViewListener() {
    
      public void valueWasChanged(CellView cellView) {
        cellView.validate();
      }
    });
  }

  public final void addCellViewListener(CellViewListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    listeners.add(listener);
  }

  public String getTip() {
    return null;
  }

  /**
   * Verifica se est no modo de edio.
   * 
   * @return .
   */
  public final boolean isEditing() {
    return isEditing;
  }

  public final boolean removeCellViewListener(CellViewListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    return listeners.remove(listener);
  }

  public void setBackgroundColor(Color backgroundColor) {
    if (backgroundColor == null) {
      throw new IllegalArgumentException(
        "O parmetro backgroundColor est nulo.");
    }
    getView().setBackground(backgroundColor);
  }

  public void setFocus(boolean hasFocus) {
    if (isValid()) {
      Border border;

      if (hasFocus) {
        border = UIManager.getBorder("Table.focusCellHighlightBorder");
      }
      else {
        border = null;
      }
      setBorder(border);
    }
  }

  public void setForegroundColor(Color foregroundColor) {
    if (foregroundColor == null) {
      throw new IllegalArgumentException(
        "O parmetro foregroundColor est nulo.");
    }
    getView().setForeground(foregroundColor);
  }

  public final boolean setValue(Object value) {
    if ((value == null) && (getValue() == null)) {
      return false;
    }
    if ((value != null) && value.equals(getValue())) {
      return false;
    }
    keepValue(value);
    fireValueWasChanged();
    return true;
  }

  public void startEditing() {
    this.isEditing = true;
  }

  public void stopEditing() {
    this.isEditing = false;
  }

  public final boolean validate() {
    String validationErrorMessage = getValidationErrorMessage();
    if (validationErrorMessage != null) {
      getView().setToolTipText(validationErrorMessage);
      Border border = BorderFactory.createLineBorder(ERROR_BORDER_COLOR,
        ERROR_BORDER_THICKNESS);
      setBorder(border);
      return false;
    }
    getView().setToolTipText(getTip());
    setBorder(null);
    return true;
  }

  /**
   * Dispara o evento {@link CellViewListener#valueWasChanged(CellView)}.
   */
  protected final void fireValueWasChanged() {
    for (CellViewListener listener : listeners) {
      listener.valueWasChanged(this);
    }
  }

  /**
   * Obtm a mensagem de erro de validao.
   * 
   * @return A mensagem de erro ou {@code null} se no houver erro de validao.
   */
  protected String getValidationErrorMessage() {
    return null;
  }
    
  /**
   * <p>
   * Armazena o valor no modelo.
   * </p>
   * 
   * <p>
   * No gera evento.
   * </p>
   * 
   * @param value O valor (Aceita {@code null}).
   */
  protected abstract void keepValue(Object value);

  /**
   * Modifica a borda da clula.
   * 
   * @param border A borda da clula ({@code null} remove a borda atual).
   */
  protected void setBorder(Border border) {
    getView().setBorder(border);
  }

  /**
   * Verifica se uma viso de clula est vlida.
   * 
   * @return .
   */
  private boolean isValid() {
    return getValidationErrorMessage() == null;
  }
}
