/*
 * $Id: ClientLauncher.java 138860 2013-03-12 17:42:57Z ururahy $
 */
package csbase.client;

import java.awt.Toolkit;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.Proxy;
import java.net.ProxySelector;
import java.net.SocketAddress;
import java.net.URI;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.ConsoleHandler;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.exception.ConfigurationException;
import csbase.logic.ServerURI;
import csbase.logic.url.URLParameters;
import csbase.util.edt.TimedEventQueue;

/**
 * Representa uma classe que dispara um cliente.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ClientLauncher {
  /** Valor do estado de erro. */
  private static final int ERROR_STATUS = 1;

  /** Uma expresso regular para espaos em branco. */
  private static final String WHITESPACE_REGEX = "\\s";

  /** Nome da propriedade que representa o nome do sistema. */
  private static final String SYSTEM_NAME_PROPERTY = "system.name";

  /** O nome da propriedade que contm a classe do cliente. */
  private static final String CLIENT_CLASS_PROPERTY = "client.class";

  /** O nome padro para o sistema. */
  private static final String DEFAULT_SYSTEM_NAME = "CSBase";

  /**
   * O nome do parmetro que define o prefixo para obteno das propriedades de
   * extenso.
   */
  private static final String EXTENSION_PREFIX_PARAMETER = "extension_prefix";

  /**
   * Inicia o lanador do cliente.
   * 
   * @param args Um array com os argumentos para a execuo do lanador do
   *        cliente. Deve conter, pelo menos, o argumento
   *        {@link URLParameters#SERVER_URL_PARAMETER}.
   */
  public static void main(String[] args) {

    Map<String, String> parameterMap = createParameterMap(args);

    /*
     * Configura o log do java.
     */
    Level level = Level.OFF;
    String logLevel = parameterMap.get("log_level");
    if (logLevel != null) {
      try {
        level = Level.parse(logLevel);
      }
      catch (Exception e) {
        e.printStackTrace();
      }
    }
    String logRoot = parameterMap.get("log_root");
    if (logRoot == null) {
      logRoot = "csbase.client";
    }
    else {
      logRoot = "csbase.client." + logRoot.trim();
    }
    Logger rootLogger = Logger.getLogger(logRoot.trim());
    rootLogger.setLevel(level);
    ConsoleHandler handler = new ConsoleHandler();
    handler.setLevel(level);
    rootLogger.addHandler(handler);

    /*
     * Configura o log da edt.
     */
    String edtLog = parameterMap.get(URLParameters.LOG_EDT_PARAMETER);
    if (Boolean.parseBoolean(edtLog)) {
      Toolkit.getDefaultToolkit().getSystemEventQueue().push(
        new TimedEventQueue());
    }

    String serverUrl = parameterMap.get(URLParameters.SERVER_URL_PARAMETER);
    if (serverUrl == null) {
      StandardDialogs.showErrorDialog(null, "Erro ao executar o sistema",
        MessageFormat.format("O argumento {0} no foi encontrado.",
          URLParameters.SERVER_URL_PARAMETER));
      System.exit(ERROR_STATUS);
    }
    if (!ServerURI.isValid(serverUrl)) {
      StandardDialogs.showErrorDialog(null, "Erro ao executar o sistema",
        MessageFormat.format("A URI {0} no  vlida", serverUrl));
      System.exit(ERROR_STATUS);
    }
    System.out.println("Servidor: " + serverUrl);
    String extensionPrefix = parameterMap.get(EXTENSION_PREFIX_PARAMETER);
    if (extensionPrefix == null) {
      ConfigurationManager.createInstance();
    }
    else {
      ConfigurationManager.createInstance(extensionPrefix);
    }
    Configuration configuration = null;
    try {
      configuration =
        ConfigurationManager.getInstance().getConfiguration(
          ClientLauncher.class);
    }
    catch (ConfigurationManagerException e) {
      e.printStackTrace();
      StandardDialogs
        .showErrorDialog(
          null,
          null,
          "Falha na instalao do sistema. Por favor, entre em contato com o administrador.");
      System.exit(ERROR_STATUS);
      return;
    }

    /*
     * configuramos o bypass do proxy baseado na propriedade client.useProxy
     * (default = true)
     */
    configureNetworkAccess(configuration);

    String systemName =
      configuration.getOptionalProperty(SYSTEM_NAME_PROPERTY,
        DEFAULT_SYSTEM_NAME);
    try {
      Client client = createClient(systemName, parameterMap);
      if (!client.execute()) {
        System.exit(ERROR_STATUS);
      }
    }
    catch (ConfigurationException e) {
      e.printStackTrace();
      StandardDialogs
        .showErrorDialog(
          null,
          systemName,
          "Falha na instalao do sistema. Por favor, entre em contato com o administrador.");
      System.exit(ERROR_STATUS);
    }
  }

  /**
   * Configura o acesso  rede, dependendo do valor da propriedade opcional
   * <code>client.bypassProxy</code>.
   * <p>
   * Se seu valor default  <code>false</code>, i.e. a configurao padro para
   * acesso  rede (definida pelo <i>Java Control Panel</i>)  usada. Se o valor
   * for <code>true</code> configuramos o sistema para acesso direto  rede
   * (bypass do proxy).
   * 
   * @param configuration configurao do ClientLauncher
   */
  private static void configureNetworkAccess(Configuration configuration) {
    Boolean bypassProxy =
      configuration.getOptionalBooleanProperty("client.bypassProxy", false);
    if (bypassProxy) {
      System.out
        .println("Habilitando conexo direta ao servidor (bypass do proxy).");
      try {
        ProxySelector.setDefault(new ProxySelector() {
          final private List<Proxy> proxyList = Arrays.asList(Proxy.NO_PROXY);

          @Override
          public List<Proxy> select(URI uri) {
            return proxyList;
          }

          @Override
          public void connectFailed(URI uri, SocketAddress sa, IOException ioe) {
            System.err.println("*** Erro de conexo direta (NO_PROXY)");
            System.err.println("*** URI: " + uri);
            System.err.println("*** SocketAddress: " + sa);
            ioe.printStackTrace();
          }
        });
      }
      catch (Exception e) {
        System.err.println("*** Erro ao habilitar bypass do proxy");
        e.printStackTrace();
      }
    }
  }

  /**
   * Cria a classe que representa o cliente.
   * 
   * @param systemName O nome do sistema (instncia).
   * @param parameterMap Um mapa de parmetros que sero usados pelo cliente.
   * 
   * @return O cliente.
   * 
   * @throws ConfigurationException Caso a classe do cliente no seja encontrada
   *         ou esteja inconsistente.
   */
  private static Client createClient(String systemName,
    Map<String, String> parameterMap) {
    Configuration configuration;
    try {
      configuration =
        ConfigurationManager.getInstance().getConfiguration(
          ClientLauncher.class);
    }
    catch (ConfigurationManagerException e) {
      throw new ConfigurationException(e);
    }
    Class<Client> clientClass;
    try {
      clientClass =
        configuration.getOptionalClassProperty(CLIENT_CLASS_PROPERTY,
          Client.class);
    }
    catch (ClassNotFoundException e) {
      throw new ConfigurationException(e);
    }
    Constructor<Client> clientConstructor;
    try {
      clientConstructor =
        clientClass.getDeclaredConstructor(String.class, Map.class);
    }
    catch (NoSuchMethodException e) {
      throw new ConfigurationException(e, MessageFormat.format(
        "O construtor da classe do cliente {0} no existe.", clientClass));
    }
    try {
      return clientConstructor.newInstance(systemName, parameterMap);
    }
    catch (InstantiationException e) {
      throw new ConfigurationException(e, MessageFormat.format(
        "Erro ao criar o cliente {0}.", clientClass));
    }
    catch (IllegalAccessException e) {
      throw new ConfigurationException(e, MessageFormat.format(
        "Erro ao criar o cliente {0}.", clientClass));
    }
    catch (InvocationTargetException e) {
      throw new ConfigurationException(e, MessageFormat.format(
        "Erro ao criar o cliente {0}.", clientClass));
    }
  }

  /**
   * <p>
   * Cria um mapa de parmetros a partir dos argumentos recebidos na linha de
   * comando.
   * </p>
   * <p>
   * Cada argumento deve estar no formato "--nome_do_argumento
   * valor_do_argumento". O "--" no  usado nas chaves do mapa.
   * </p>
   * 
   * @param parameters Um array de argumentos.
   * 
   * @return Um mapa de parmetros.
   * 
   * @throws IllegalArgumentException Caso o argumento esteja num formato
   *         invlido.
   */
  private static Map<String, String> createParameterMap(String[] parameters) {
    Map<String, String> parameterMap = new HashMap<String, String>();
    for (int i = 0; i < parameters.length; i++) {
      String[] parameter = parameters[i].trim().split(WHITESPACE_REGEX);
      if (parameter.length != 2) {
        throw new IllegalArgumentException(MessageFormat.format(
          "O argumento {0} est invlido.", parameters[i]));
      }
      parameterMap.put(parameter[0].substring(2), parameter[1]);
    }
    return parameterMap;
  }
}
