/*
 * $Id$
 */
package csbase.client.algorithms.commands.cache;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;

import csbase.client.util.IFilter;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;

/**
 * Esta classe abstrata representa um filtro genrico de comandos.<br>
 * No intante da checagem, o estado do comando  validado. Caso queira checar
 * algo mais relacionado ao comando basta sobrescrever o mtodo
 * {@link CommandsFilter#acceptCommand}.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class CommandsFilter implements IFilter<CommandInfo> {

  /**
   * Conjunto de estados em que um comando possa estar no momento do teste para
   * que este teste possa ser permitido.<br>
   * Esta  uma coleo imutvel que encapsula um {@link HashSet} utilizado para
   * agilizar a busca por objetos.
   */
  private Collection<CommandStatus> allowedStatuses;

  /**
   * Construtor que cria um filtro que aceita comandos em todos os estados.
   * 
   */
  public CommandsFilter() {
    this(true, CommandStatus.values());
  }

  /**
   * Construtor que cria um filtro que aceita apenas comandos em um dos estados
   * passados.
   * 
   * @param statuses Conjunto de estados em que um comando possa ou no estar no
   *        momento do teste para que este teste possa ser permitido.
   */
  public CommandsFilter(CommandStatus... statuses) {
    this(true, statuses);
  }

  /**
   * Construtor.
   * 
   * @param in Indica a lgica a ser utilizada em relao ao conjunto de estados
   *        passados como parmetro. Se for <tt>true</tt> os comandos a serem
   *        aprovados deveram estar em um dos estados passados como parmetro no
   *        momento da checagem. Se for <tt>false</tt>, os comandos no
   *        poderam estar em nenhum dos estados passados no momento da checagem.
   * @param statuses Conjunto de estados em que um comando possa ou no estar no
   *        momento do teste para que este teste possa ser permitido.
   */
  public CommandsFilter(boolean in, CommandStatus... statuses) {

    if (null == statuses || 0 == statuses.length) {
      throw new IllegalArgumentException(
        "Array de status de comando vazio ou nulo.");
    }

    /*
     * Guarda em um {@link HashSet} para facilitar a busca pelos 
     * {@link CommandStatus} depois.
     */
    HashSet<CommandStatus> allowedStatusesHashSet =
      new HashSet<CommandStatus>();
    
    if (in) {
      for (CommandStatus aStatus : statuses) {
        allowedStatusesHashSet.add(aStatus);
      }
    }
    else {
      // Adiciona todos os estados como permitidos
      for (CommandStatus aStatus : CommandStatus.values()) {
        allowedStatusesHashSet.add(aStatus);
      }
      // em seguida remove os invlidos
      for (CommandStatus aStatus : statuses) {
        allowedStatusesHashSet.remove(aStatus);
      }
    }

    /*
     * Encapsula em uma coleo imutvel para poder repassar atravs do 
     * mtodo {@link CommandsFilter#getAllowedStatus()} sem que haja o 
     * risco de alterao da lista.
     */    
    this.allowedStatuses =
      Collections.unmodifiableCollection(allowedStatusesHashSet);
  }

  /**
   * Obtm uma coleo imutvel dos {@link CommandStatus} em que um comando pode
   * estar ao ser checado por este filtro.
   * 
   * @return uma coleo imutvel dos {@link CommandStatus} em que um comando
   *         pode estar ao ser checado por este filtro.
   */
  public final Collection<CommandStatus> getAllowedStatus() {
    return this.allowedStatuses;
  }

  /**
   * Valida um comando.
   * 
   * @param cmd o comando sendo filtrado.
   * 
   * @return <tt>true</tt> se no momento do teste o estado do comando  um
   *         dentre os estados aceitos pelo filtro e o comando foi aceito pelo
   *         mtodo {@link CommandsFilter#acceptCommand(CommandInfo)}.
   */
  @Override
  public final boolean accept(CommandInfo cmd) {
    return allowedStatuses.contains(cmd.getStatus()) && acceptCommand(cmd);
  }

  /**
   * Valida um comando.<br>
   * A princpio retorna sempre <tt>true</tt>. Caso queira alterar este
   * comportamento, basta sobrescrever este mtodo.
   * 
   * @param cmd o comando sendo filtrado.
   * 
   * @return <tt>true</tt> se o comando  considerado vlido por este filtro.
   *         A princpio retorna sempre <tt>true</tt>. Caso queira alterar
   *         este comportamento, basta sobrescrever este mtodo.
   */
  protected boolean acceptCommand(CommandInfo cmd) {
    return true;
  }
}
