/*
 * $Id:$
 */

package csbase.client.applications.flowapplication;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.awt.geom.AffineTransform;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;
import javax.swing.event.AncestorEvent;
import javax.swing.event.AncestorListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.vix.Filter;
import tecgraf.vix.TypeMessage;
import tecgraf.vix.TypeVO;
import tecgraf.vix.TypeVS;
import csbase.client.algorithms.AlgorithmConfiguratorFactory;
import csbase.client.algorithms.AlgorithmConfiguratorView;
import csbase.client.applications.flowapplication.configurator.FlowAlgorithmConfiguratorView;
import csbase.client.applications.flowapplication.graph.Graph;
import csbase.client.applications.flowapplication.graph.GraphElement;
import csbase.client.applications.flowapplication.graph.GraphFileDescriptor;
import csbase.client.applications.flowapplication.graph.GraphLink;
import csbase.client.applications.flowapplication.graph.GraphListener;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.applications.flowapplication.graph.Grid;
import csbase.client.applications.flowapplication.messages.ChangeCursorMessage;
import csbase.client.applications.flowapplication.messages.ErrorMessage;
import csbase.client.applications.flowapplication.messages.HideHintMessage;
import csbase.client.applications.flowapplication.messages.SelectElementMessage;
import csbase.client.applications.flowapplication.messages.SelectElementsMessage;
import csbase.client.applications.flowapplication.messages.ShowHintMessage;
import csbase.client.applications.flowapplication.messages.ShowPopupMessage;
import csbase.client.applications.flowapplication.zoom.AbstractZoomModel;
import csbase.client.applications.flowapplication.zoom.ZoomModel;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.BugException;
import csbase.exception.ParseException;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.parameters.AbstractFileParameter;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.validation.ValidationMode;

/**
 * <p>
 * rea de trabalho.
 * </p>
 *
 * <p>
 * A rea de trabalho  a regio onde so desenhados os elementos do grafo que
 * representa o fluxo de algoritmos. Ela  responsvel pelo controle de zoom e o
 * controle das barras de rolagem.
 * </p>
 *
 * @author Tecgraf/PUC-Rio
 */
public final class Workspace extends JPanel implements TypeVS {
  /**
   * Largura-padro.
   */
  private static final double DEFAULT_WIDTH = 400;

  /**
   * Altura-padro.
   */
  private static final double DEFAULT_HEIGHT = 300;

  /**
   * Margem.
   */
  private static final int MARGIN = 10;

  /** O grafo. */
  private Graph graph;

  /** O grid. */
  private final Grid grid;

  /** A matriz de transformao. */
  private final AffineTransform matrix;

  /** O vo. */
  private TypeVO vo;

  /** O modelo de zoom. */
  private final ZoomModel zoomModel;

  /**
   * Indica se novas conexes devem ser criadas automaticamente ao adicionar um
   * novo n no fluxo.
   */
  private boolean autoCreateLinks;

  /**
   * Indica se devem ser mostradas as verses dos algoritmos no fluxo.
   */
  private boolean showVersion;

  /**
   * Indica se o nvel de zoom do workspace deve ser ajustado para manter todo o
   * contedo do grafo visvel.
   */
  private boolean fit;

  /**
   * Cria a rea de trabalho.
   *
   * @param window A janela.
   * @param useGrid Indica se o grid deve ser utilizado.
   * @param zoomToFit Indica se o nvel de zoom do workspace deve ser ajustado
   *        para manter todo o contedo do grafo visvel.
   */
  public Workspace(final Window window, final boolean useGrid, boolean zoomToFit) {
    this(new Graph(window), useGrid, zoomToFit);
  }

  /**
   * Cria a rea de trabalho.
   *
   * @param graph grafo.
   * @param useGrid Indica se o grid deve ser utilizado.
   * @param zoomToFit Indica se o nvel de zoom do workspace deve ser ajustado
   *        para manter todo o contedo do grafo visvel.
   */
  public Workspace(final Graph graph, final boolean useGrid,
    final boolean zoomToFit) {
    this.matrix = new AffineTransform();
    this.graph = graph;
    this.zoomModel = new VSZoomModel();
    this.autoCreateLinks = false;
    this.showVersion = false;
    if (useGrid) {
      setBackground(Color.WHITE);
    }
    this.fit = zoomToFit;
    this.grid = new Grid(this);
    this.grid.setEnabled(useGrid);
    this.graph.changeVS(null, this.grid);
    changeVO(null, this.grid);
    this.grid.changeVO(null, this.graph);
    addAncestorListener(new VSAncestorListener());
    addComponentListener(new VSComponentListener());
    addMouseListener(new VSMouseListener());
    addMouseMotionListener(new VSMouseMotionListener());
    addKeyListener(new VSKeyListener());
    this.graph.addGraphListener(new VSGraphListener());
    this.graph.setVersionInfoVisible(showVersion);
    new DropTarget(this, new VSDropTargetListener());
    if (fit) {
      adjustZoomToFit();
    }
  }

  /**
   * Coloca o filtro fornecido como o vo atual da rea de trabalho.
   *
   * @param filter O filtro.
   */
  public void attachFilter(final Filter filter) {
    filter.changeVO(null, this.vo);
    filter.changeVS(null, this);
    this.vo.changeVS(null, filter);
    changeVO(null, filter);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean changeVO(final TypeVO oldvo, final TypeVO newvo) {
    if (oldvo == null) {
      this.vo = newvo;
      return true;
    }
    if (!oldvo.equals(this.vo)) {
      return false;
    }
    this.vo = newvo;
    return true;
  }

  /**
   * Obtm o grafo.
   *
   * @return O grafo.
   */
  public Graph getGraph() {
    return this.graph;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Graphics2D getGraphics2D() {
    final Graphics2D g = (Graphics2D) getGraphics();
    if (g != null) {
      g.transform(this.matrix);
    }
    return g;
  }

  /**
   * Obtm o modelo de zoom.
   *
   * @return O modelo de zoom.
   */
  public ZoomModel getZoomModel() {
    return this.zoomModel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean msgHandlerVS(final TypeMessage msg) {
    if (msg instanceof ErrorMessage) {
      handleErrorMessage((ErrorMessage) msg);
      return true;
    }
    if (msg instanceof ShowPopupMessage) {
      handleShowPopupMessage((ShowPopupMessage) msg);
      return true;
    }
    if (msg instanceof ShowHintMessage) {
      handleShowHintMessage((ShowHintMessage) msg);
      return true;
    }
    if (msg instanceof HideHintMessage) {
      handleHideHintMessage();
      return true;
    }
    if (msg instanceof ChangeCursorMessage) {
      handleChangeCursorMessage((ChangeCursorMessage) msg);
      return true;
    }
    return false;
  }

  /**
   * Ajusta o nvel de zoom do workspace para tornar todo o contedo do grafo
   * visvel.
   */
  public void adjustZoomToFit() {
    double fitValue = zoomModel.getFitValue();
    zoomModel.setValue(fitValue);
  }

  /**
   * <p>
   * Envia uma mensagem ao vo atual desta rea de trabalho.
   * </p>
   * <p>
   * Seno existe um vo atual, ele no faz nada.
   * </p>
   *
   * @param msg A mensagem.
   *
   * @return <code>true</code> se a mensagem foi recebida, ou <code>false</code>
   *         se a mensagem no foi recebida (inclusive se no existir um vo
   *         atual).
   */
  public boolean sendVO(final TypeMessage msg) {
    if (this.vo == null) {
      return false;
    }
    return msg.sendVO(this.vo);
  }

  /**
   * Ajusta o grafo.
   *
   * @param graph O grafo.
   */
  public void setGraph(final Graph graph) {
    graph.addGraphListeners(this.graph.getGraphListenerList());
    this.graph.clearGraphListeners();
    this.graph = graph;
    this.graph.changeVS(null, this.grid);
    this.grid.changeVO(null, this.graph);
    this.graph.nodeValidation(ValidationMode.FULL, true);
    this.graph.notifyChangedWorkspace();
    this.graph.addGraphListener(new VSGraphListener());
    this.graph.setVersionInfoVisible(showVersion);
    updateSize();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void paintChildren(final Graphics g) {
    if (this.vo != null) {
      final Graphics2D g2D = (Graphics2D) g;
      g2D.transform(this.matrix);
      this.vo.callbackRepaint((Graphics2D) g);
    }
  }

  /**
   * Obtm um texto no mecanismo de internacionalizao.
   *
   * @param msgKey A chave da mensagem.
   *
   * @return A mensagem.
   */
  private String getString(final String msgKey) {
    return LNG.get(Workspace.class.getName() + "." + msgKey);
  }

  /**
   * Trata mensagens mudar cursor.
   *
   * @param message A mensagem de mudana de cursor.
   */
  private void handleChangeCursorMessage(final ChangeCursorMessage message) {
    setCursor(message.getCursor());
  }

  /**
   * Trata mensagens erro.
   *
   * @param message A mensagem de erro.
   */
  private void handleErrorMessage(final ErrorMessage message) {
    final String key = message.getKey();
    if (key != null) {
      String errorMessage;
      final Object[] arguments = message.getArguments();
      if (arguments == null) {
        errorMessage = LNG.get(key);
      }
      else {
        errorMessage = LNG.get(key, arguments);
      }
      showWarningDialog(errorMessage);
    }
    else {
      showErrorDialog(message.getException());
    }
  }

  /**
   * Trata mensagens ocultar dica.
   */
  private void handleHideHintMessage() {
    if (getToolTipText() != null) {
      setToolTipText(null);
    }
  }

  /**
   * Trata mensagens exibir dica.
   *
   * @param message A mensagem.
   */
  private void handleShowHintMessage(final ShowHintMessage message) {
    if (!message.getHint().equals(getToolTipText())) {
      setToolTipText(message.getHint());
    }
  }

  /**
   * Trata mensagens exibir menu popup.
   *
   * @param message A mensagem.
   */
  private void handleShowPopupMessage(final ShowPopupMessage message) {
    final JPopupMenu popupMenu = new JPopupMenu();
    final List<JComponent> components = message.getPopupComponentList();
    if (!components.isEmpty()) {
      for (final JComponent component : components) {
        popupMenu.add(component);
      }
      final Point2D point = message.getPoint();
      this.matrix.transform(point, point);
      popupMenu.show(this, (int) point.getX(), (int) point.getY());
    }
  }

  /**
   * Importa configurador de algoritmo.
   *
   * @param view viso do configurador.
   * @param point ponto do workspace onde algoritmo ser colocado.
   */
  private void importAlgorithmConfigurator(
    final AlgorithmConfiguratorView view, final Point point) {
    if (view == null) {
      return;
    }
    if (view instanceof FlowAlgorithmConfiguratorView) {
      final FlowAlgorithmConfiguratorView flowView =
        (FlowAlgorithmConfiguratorView) view;
      importFlowAlgorithmConfigurator(flowView, point);
    }
    else {
      final GraphNode node = this.graph.createGraphNode(view, point, false);
      new SelectElementMessage(node, point).sendVO(this.vo);
      if (autoCreatesLinks()) {
        createLinks(node);
      }
    }
  }

  /**
   * Cria automaticamente ligaes entre as entradas de um determinado n e uma
   * das sadas dos demais ns do fluxo, desde que sejam compatveis. Essa
   * operao s ser efetivada se no houver mais de uma opo possvel para
   * fazer as ligaes. Caso sejam encontradas mltiplas sadas compatveis,
   * nada  feito.
   *
   * @param node o novo n.
   */
  private void createLinks(final GraphNode node) {
    Map<String, GraphFileDescriptor> inputs = getUniqueAvailableInputs(node);
    if (!inputs.isEmpty()) {
      Collection<GraphNode> nodeCollection = graph.getNodeCollection();
      if (nodeCollection.size() > 1) {
        HashMap<String, GraphFileDescriptor> outputs =
          new HashMap<String, GraphFileDescriptor>();
        for (GraphNode graphNode : nodeCollection) {
          if (!graphNode.equals(node)) {
            Map<String, GraphFileDescriptor> nodeOutputs =
              getUniqueAvailableOutputs(graphNode);
            for (Entry<String, GraphFileDescriptor> entry : nodeOutputs
              .entrySet()) {
              GraphFileDescriptor oldValue =
                outputs.put(entry.getKey(), entry.getValue());
              if (oldValue != null) {
                return;
              }
            }
          }
        }
        if (outputs.size() > 0) {
          for (Entry<String, GraphFileDescriptor> inputEntry : inputs
            .entrySet()) {
            String fileType = inputEntry.getKey();
            if (outputs.containsKey(fileType)) {
              List<Point2D> emptyList = Collections.emptyList();
              GraphFileDescriptor output = outputs.get(fileType);
              GraphFileDescriptor input = inputEntry.getValue();
              AbstractFileParameter outputParam = output.getFileParameter();
              AbstractFileParameter inputParam = input.getFileParameter();
              if (outputParam != null && inputParam != null
                && outputParam.getMode() == inputParam.getMode()) {
                graph.createGraphLink(input, output, emptyList);
              }
            }
          }
        }
      }
    }
  }

  /**
   * Obtm as sadas disponveis para se fazer conexes com o n. So
   * consideradas somente as entradas que j no contenham conexes ou cujos
   * parmetros ainda no foram configurados. Se houver mais de uma entrada do
   * mesmo tipo no n, retorna uma coleo vazia pois no poderemos determinar
   * unicamente qual  a maneira correta de fazer a conexo.
   *
   * @param node o n.
   * @return as entradas nicas disponveis para conexes.
   */
  private Map<String, GraphFileDescriptor> getUniqueAvailableOutputs(
    GraphNode node) {
    Map<String, GraphFileDescriptor> outputs =
      new HashMap<String, GraphFileDescriptor>();
    Collection<GraphFileDescriptor> outputFiles =
      node.getOutputFileDescriptorCollection();
    for (GraphFileDescriptor output : outputFiles) {
      AbstractFileParameter fileParameter = output.getFileParameter();
      if (output.getLinkFromCollection().isEmpty()
        && canInferLink(fileParameter)) {
        String fileType;
        if (fileParameter.getMode() == FileParameterMode.DIRECTORY) {
          fileType = FileParameterMode.DIRECTORY.toString();
        }
        else {
          fileType = fileParameter.getFileType();
        }
        GraphFileDescriptor oldValue = outputs.put(fileType, output);
        // Se j existe um parmetro com mesmo tipo, no tem como sugerir um link.
        if (oldValue != null) {
          return Collections.emptyMap();
        }
      }
    }
    return outputs;
  }

  /**
   * Obtm as entradas disponveis para se fazer conexes com o n. So
   * consideradas somente as entradas que j no contenham conexes ou cujos
   * parmetros ainda no foram configurados. Se houver mais de uma sada do
   * mesmo tipo no n, retorna uma coleo vazia pois no poderemos determinar
   * unicamente qual  a maneira correta de fazer a conexo.
   *
   * @param node o n.
   * @return as entradas nicas disponveis para conexes.
   */
  private Map<String, GraphFileDescriptor> getUniqueAvailableInputs(
    final GraphNode node) {
    Map<String, GraphFileDescriptor> inputs =
      new HashMap<String, GraphFileDescriptor>();
    for (GraphFileDescriptor input : node.getInputFileDescriptorCollection()) {
      AbstractFileParameter fileParameter = input.getFileParameter();
      if (input.getLinkTo() == null && canInferLink(fileParameter)) {
        String fileType;
        if (fileParameter.getMode() == FileParameterMode.DIRECTORY) {
          fileType = FileParameterMode.DIRECTORY.toString();
        }
        else {
          fileType = fileParameter.getFileType();
        }
        GraphFileDescriptor oldValue = inputs.put(fileType, input);
        // Se j existe um parmetro com mesmo tipo, no tem como sugerir um link.
        if (oldValue != null) {
          return Collections.emptyMap();
        }
      }
    }
    return inputs;
  }

  /**
   * Indica se um determinado parmetro do tipo arquivo pode ter ligaes
   * inferidas automaticamente.
   *
   * @param fileParameter o parmetro.
   * @return verdadeiro se o parmetro poder ter ligaes inferidas
   *         automaticamente ou falso, se isso no  possvel.
   */
  private boolean canInferLink(AbstractFileParameter fileParameter) {
    return fileParameter != null && fileParameter.getValue() == null
      && fileParameter.usesPipe() != FileParameterPipeAcceptance.FALSE
      && fileParameter.isEnabled() && fileParameter.isVisible();
  }

  /**
   * Importa configurador de fluxo.
   *
   * @param flowAlgorithmConfiguratorView viso do configurador.
   * @param dropPosition ponto do workspace onde algoritmo ser colocado.
   */
  private void importFlowAlgorithmConfigurator(
    final FlowAlgorithmConfiguratorView flowAlgorithmConfiguratorView,
    final Point dropPosition) {
    final Graph flowGraph = flowAlgorithmConfiguratorView.getGraph();
    final Collection<GraphElement> elements = this.graph.merge(flowGraph);
    new SelectElementsMessage(elements).sendVO(this.vo);
    final Rectangle2D bounds = flowGraph.getBounds2D();
    if (bounds != null) {
      final double centerX = bounds.getCenterX();
      final double centerY = bounds.getCenterY();
      double dx = dropPosition.getX() - centerX;
      double dy = dropPosition.getY() - centerY;
      for (final GraphElement element : elements) {
        final Rectangle2D elementBounds = element.getBounds2D();
        if (dx < 0) {
          dx = Math.max(dx, -1 * elementBounds.getMinX());
        }
        if (dy < 0) {
          dy = Math.max(dy, -1 * elementBounds.getMinY());
        }
      }
      for (final GraphElement element : elements) {
        element.drag(dx, dy);
      }
      for (final GraphElement element : elements) {
        element.drop();
      }
      repaint();
    }
  }

  /**
   * <p>
   * Inverte o ponto usando a matriz de transformao.
   * </p>
   * <p>
   * Ou seja, converte um ponto de mundo (grafo) para tela (rea de trabalho).
   * </p>
   *
   * @param pt O ponto.
   */
  private void inverse(final Point2D pt) {
    try {
      Workspace.this.matrix.inverseTransform(pt, pt);
    }
    catch (final NoninvertibleTransformException e) {
      e.printStackTrace();
    }
  }

  /**
   * Atualiza as dimenses do grid.
   */
  private void resizeGrid() {
    if (this.grid != null) {
      final double minX = 0.0;
      final double minY = 0.0;
      double maxX = getBounds().getMaxX();
      double maxY = getBounds().getMaxY();
      final Rectangle2D bounds = this.graph.getBounds2D();
      if (bounds != null) {
        maxX = Math.max(maxX, bounds.getMaxX() + MARGIN);
        maxY = Math.max(maxY, bounds.getMaxY() + MARGIN);
      }
      maxX *= 1 / this.zoomModel.getValue();
      maxY *= 1 / this.zoomModel.getValue();
      this.grid.setBounds2D(minX, minY, maxX, maxY);
      repaint();
    }
  }

  /**
   * Exibe um erro.
   *
   * @param exception A exceo (No aceita {@code null}).
   */
  private void showErrorDialog(final Exception exception) {
    final Window window = SwingUtilities.getWindowAncestor(this);
    StandardErrorDialogs.showErrorDialog(window, getString("error"), exception);
  }

  /**
   * Exibe um aviso.
   *
   * @param message A mensagem.
   */
  private void showWarningDialog(final String message) {
    final Window window = SwingUtilities.getWindowAncestor(this);
    StandardDialogs.showWarningDialog(window, getString("error"), message);
  }

  /**
   * Atualiza o tamanho da rea de trabalho.
   */
  private void updateSize() {
    final Rectangle2D bounds = this.graph.getBounds2D();
    if (bounds != null) {
      final Dimension newPreferredSize = new Dimension();
      if (fit) {
        final double zoomValue = getZoomModel().getValue();
        final double tx = -bounds.getMinX() * zoomValue;
        final double ty = -bounds.getMinY() * zoomValue;
        this.matrix.setTransform(zoomValue, 0.0, 0.0, zoomValue, tx, ty);
        double newWidth = bounds.getWidth() * zoomValue;
        double newHeight = bounds.getHeight() * zoomValue;
        newPreferredSize.setSize(newWidth, newHeight);
      }
      else {
        final double newWidth =
          (bounds.getX() + bounds.getWidth() + MARGIN)
          * this.zoomModel.getValue();
        final double newHeight =
          (bounds.getY() + bounds.getHeight() + MARGIN)
          * this.zoomModel.getValue();
        newPreferredSize.setSize(newWidth, newHeight);
      }
      setPreferredSize(newPreferredSize);
      revalidate();
    }
    resizeGrid();
  }

  /**
   * Adiciona um novo n ao Workspace em posio relativa aos ns
   * pr-existentes.
   *
   * @param algorithmVersionInfo informao do n a ser adicionado.
   */
  public void addNewNode(AlgorithmVersionInfo algorithmVersionInfo) {
    Point position = graph.findPositionForNewNode();
    addNewNode(algorithmVersionInfo, position);
  }

  /**
   * Importa um configurador de algoritmo.
   *
   * @param configurator o configurador.
   */
  public void importConfigurator(AlgorithmConfigurator configurator) {
    if (configurator == null) {
      return;
    }
    Point position = graph.findPositionForNewNode();
    final Window window = SwingUtilities.getWindowAncestor(Workspace.this);
    AlgorithmConfiguratorView view =
      AlgorithmConfiguratorFactory.getInstance().createConfigurationView(
        window, configurator.getAlgorithmVersion(),
        ValidationMode.ALLOW_EMPY_VALUES);
    if (view == null) {
      return;
    }
    Map<Object, Object> parameterValues = configurator.exportValues();
    AlgorithmConfigurator newConfigurator = view.getConfigurator();
    try {
      newConfigurator.importValues(parameterValues);
    }
    catch (ParseException e) {
      /*
       * So valores que estavam no configurador original, portanto no pode
       * ocorrer um erro ao l-los para a nova cpia do configurador.
       */
      throw new BugException(e);
    }
    importAlgorithmConfigurator(view, position);
  }

  /**
   * Adiciona um novo n ao Workspace na posio indicada.
   *
   * @param algorithmVersionInfo informao do n a ser adicionado.
   * @param position local onde o n deve ser posicionado.
   */
  public void addNewNode(AlgorithmVersionInfo algorithmVersionInfo,
    final Point position) {
    if (algorithmVersionInfo == null) {
      return;
    }
    final Window window = SwingUtilities.getWindowAncestor(Workspace.this);
    final AlgorithmConfiguratorView algorithmConfiguratorView =
      AlgorithmConfiguratorFactory.getInstance().createConfigurationView(
        window, algorithmVersionInfo, ValidationMode.ALLOW_EMPY_VALUES);
    if (algorithmConfiguratorView == null) {
      return;
    }
    importAlgorithmConfigurator(algorithmConfiguratorView, position);
  }

  /**
   * Determina se novas conexes devem ser criadas automaticamente ao adicionar
   * um novo n no fluxo.
   *
   * @param autoCreate verdadeiro se novas ligaes devem ser criadas
   *        automaticamente ou falso, caso contrrio.
   */
  public void setAutoCreateLinks(boolean autoCreate) {
    this.autoCreateLinks = autoCreate;
  }

  /**
   * Determina se a informao de verso do algoritmo deve ser mostrada nos ns.
   *
   * @param visible verdadeiro se a informao deve ser mostrada ou falso, caso
   *        contrrio.
   */
  public void setVersionInfoVisible(boolean visible) {
    this.showVersion = visible;
    this.graph.setVersionInfoVisible(visible);
  }

  /**
   * Indica se novas conexes so criadas automaticamente ao adicionar um novo
   * n no fluxo.
   *
   * @return verdadeiro se novas ligaes so criadas automaticamente ou falso,
   *         caso contrrio.
   */
  public boolean autoCreatesLinks() {
    return this.autoCreateLinks;
  }

  /**
   * Listener utilizado para ouvir a notificao de quando o componente 
   * adicionado ao seu pai.
   */
  private final class VSAncestorListener implements AncestorListener {

    /**
     * {@inheritDoc}
     */
    @Override
    public void ancestorAdded(AncestorEvent event) {
      if (fit) {
        adjustZoomToFit();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void ancestorMoved(AncestorEvent event) {
      //Ignora o evento
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void ancestorRemoved(AncestorEvent event) {
      //Ignora o evento
    }

  }

  /**
   * Listener que notifica eventos genricos de componente  rea de trabalho.
   */
  private final class VSComponentListener extends ComponentAdapter {
    /**
     * {@inheritDoc}
     */
    @Override
    public void componentResized(final ComponentEvent e) {
      resizeGrid();
      repaint();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void componentShown(final ComponentEvent e) {
      resizeGrid();
      repaint();
    }
  }

  /**
   * Observador de {@link DropTarget}.
   */
  private final class VSDropTargetListener implements DropTargetListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void dragEnter(final DropTargetDragEvent dtde) {
      // Ignora evento.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void dragExit(final DropTargetEvent dte) {
      // Ignora evento.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void dragOver(final DropTargetDragEvent dtde) {
      final Transferable transferable = dtde.getTransferable();
      if (transferable
        .isDataFlavorSupported(AlgorithmVersionTransferable.DATA_FLAVOR)) {
        dtde.acceptDrag(DnDConstants.ACTION_COPY);
      }
      else {
        dtde.rejectDrag();
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void drop(final DropTargetDropEvent dtde) {
      requestFocusInWindow();
      final Point dropPosition = dtde.getLocation();
      inverse(dropPosition);
      final Transferable transferable = dtde.getTransferable();
      if (!transferable
        .isDataFlavorSupported(AlgorithmVersionTransferable.DATA_FLAVOR)) {
        return;
      }
      AlgorithmVersionInfo algorithmVersionInfo;
      try {
        algorithmVersionInfo =
          (AlgorithmVersionInfo) transferable
          .getTransferData(AlgorithmVersionTransferable.DATA_FLAVOR);
      }
      catch (final UnsupportedFlavorException e) {
        throw new IllegalStateException(e);
      }
      catch (final IOException e) {
        showErrorDialog(e);
        return;
      }
      addNewNode(algorithmVersionInfo, dropPosition);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void dropActionChanged(final DropTargetDragEvent dtde) {
      // Ignora evento.
    }
  }

  /**
   * Observador de grafo.
   */
  private final class VSGraphListener implements GraphListener {

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasChangedWorkspace(final Graph workGraph) {
      updateSize();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementCreated(final Graph workGraph,
      final GraphElement element) {
      updateSize();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementDragged(final Graph workGraph,
      final GraphElement element, final double tx, final double ty) {
      // Ignora mudana.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementDragged(final Graph workGraph,
      final GraphElement element, final Point2D startPoint,
      final Point2D endPoint) {
      // Ignora mudana.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementDropped(final Graph workGraph,
      final GraphElement element, final Point2D point) {
      updateSize();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementParametrized(final Graph workGraph,
      final GraphElement element) {
      // Ignora mudana.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementRemoved(final Graph workGraph,
      final GraphElement element) {
      updateSize();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementSelected(final Graph workGraph,
      final GraphElement element) {
      // Ignora mudana.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasLinkAnchored(final Graph workGraph, final GraphLink link,
      final GraphFileDescriptor fileDescriptor) {
      // Ignora mudana.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasLinkIncreased(final Graph workGraph, final GraphLink link) {
      updateSize();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasLinkStatusChanged(final Graph workGraph, final GraphLink link) {
      // Ignora mudana.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasLinkUnanchored(final Graph workGraph, final GraphLink link,
      final GraphFileDescriptor fileDescriptor) {
      // Ignora mudana.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasNodeResized(final Graph workGraph, final GraphNode node) {
      updateSize();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasParameterSetEnabled(final Graph workGraph,
      final GraphNode node, final String parameterName, final boolean isEnabled) {
      // Ignora mudana.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasReseted(final Graph workGraph) {
      // Ignora mudana.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasParameterSetVisible(final Graph workGraph,
      final GraphNode node, final String parameterName, final boolean isVisible) {
      // Ignora mudana.
    }
  }

  /**
   * Listener que notifica eventos de teclas  rea de trabalho.
   */
  private final class VSKeyListener implements KeyListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void keyPressed(final KeyEvent ev) {
      if (Workspace.this.vo != null) {
        Workspace.this.vo.callbackKey(ev);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void keyReleased(final KeyEvent ev) {
      if (Workspace.this.vo != null) {
        Workspace.this.vo.callbackKey(ev);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void keyTyped(final KeyEvent ev) {
      if (Workspace.this.vo != null) {
        Workspace.this.vo.callbackKey(ev);
      }
    }
  }

  /**
   * Listener que notifica eventos de mouse  rea de trabalho.
   */
  private final class VSMouseListener extends MouseAdapter {
    /**
     * {@inheritDoc}
     */
    @Override
    public void mouseClicked(final MouseEvent ev) {
      handleMouseButtonEvent(ev);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void mouseEntered(final MouseEvent ev) {
      // Ignora evento.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void mousePressed(final MouseEvent ev) {
      handleMouseButtonEvent(ev);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void mouseReleased(final MouseEvent ev) {
      handleMouseButtonEvent(ev);
    }

    /**
     * Trata eventos de boto do mouse.
     *
     * @param ev O evento (No aceita {@code null}).
     */
    private void handleMouseButtonEvent(final MouseEvent ev) {
      requestFocusInWindow();
      if (Workspace.this.vo != null) {
        final Point2D pt = new Point2D.Float(ev.getX(), ev.getY());
        inverse(pt);
        Workspace.this.vo.callbackButton(pt, ev);
      }
    }
  }

  /**
   * Listener que notifica eventos de movimento de mouse  rea de trabalho.
   */
  private final class VSMouseMotionListener implements MouseMotionListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void mouseDragged(final MouseEvent ev) {
      final boolean isLeft = SwingUtilities.isLeftMouseButton(ev);
      if (isLeft && Workspace.this.vo != null) {
        final Point2D pt = new Point2D.Double(ev.getX(), ev.getY());
        inverse(pt);
        Workspace.this.vo.callbackDrag(pt, ev);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void mouseMoved(final MouseEvent ev) {
      if (Workspace.this.vo != null) {
        final Point2D pt = new Point2D.Double(ev.getX(), ev.getY());
        inverse(pt);
        Workspace.this.vo.callbackMove(pt, ev);
      }
    }
  }

  /**
   * Modelo de zoom para a rea de trabalho.
   */
  private final class VSZoomModel extends AbstractZoomModel {

    /**
     * Valor de zoom default
     */
    private static final double STANDARD_ZOOM_VALUE = 1.0;

    /**
     * Valor de zoom mximo
     */
    private static final double MAX_ZOOM_VALUE = 4.0;

    /**
     * Valor de zoom mximo
     */
    private static final double MIN_ZOOM_VALUE = 0.20;

    /**
     * A escala atual.
     */
    private double zoom;

    /**
     * Cria um modelo de zoom.
     */
    public VSZoomModel() {
      super(MIN_ZOOM_VALUE, MAX_ZOOM_VALUE, STANDARD_ZOOM_VALUE, 0.01, 0.10);
      this.zoom = STANDARD_ZOOM_VALUE;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public double getFitValue() {
      Rectangle workspaceBounds = getVisibleRect();
      if (workspaceBounds.isEmpty()) {
        return getStandardValue();
      }
      double workspaceWidth = workspaceBounds.getWidth();
      double workspaceHeight = workspaceBounds.getHeight();

      Rectangle2D graphBounds = graph.getBounds2D();
      double graphWidth, graphHeight;
      if (graphBounds == null) {
        graphWidth = DEFAULT_WIDTH;
        graphHeight = DEFAULT_HEIGHT;
      }
      else {
        graphWidth = graphBounds.getWidth() + MARGIN;
        graphHeight = graphBounds.getHeight() + MARGIN;
      }
      double widthRatio = workspaceWidth / graphWidth;
      double heightRatio = workspaceHeight / graphHeight;
      double ratio = 1;
      if (widthRatio < 1 || heightRatio < 1) {
        ratio = Math.min(widthRatio, heightRatio);
        /*
         * Arredonda o zoom para baixo para garantir que o grafo ser mostrado
         * por completo sem a criao de barras de rolagem.
         */
        ratio = Math.floor(ratio * 100) / 100.0;
      }
      return ratio;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public double getValue() {
      return this.zoom;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void changeValue(final double value) {
      Workspace.this.matrix.setToIdentity();
      Workspace.this.matrix.scale(value, value);
      this.zoom = value;
      updateSize();
      repaint();
    }
  }
}
