package csbase.client.ias;

import java.awt.Color;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.desktop.RemoteTask;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.client.remote.srvproxies.UserGroupProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.User;
import csbase.logic.UserGroup;
import csbase.logic.UserInfo;

/**
 * Extenso do dilogo de modificao de dados de usurios, especfico para o
 * acesso do administrador.
 * 
 * <p>
 * NOTA 1: Caso o mtodo de autenticao do servidor no seja validao local, o
 * dilogo exibe um <i>checkbox</i> permitindo uma autenticao local "forada".
 * Isto  necessrio devido a uma circunstncia atual na Petrobras de
 * descentralizao na administrao de servidores LDAP. Como existem muitas
 * configuraes diferentes, o sistema tem dificuldade em conectar-se a alguns
 * desses servidores, o que impossibilita usurios cadastrados nesses servidores
 * de acessarem o sistema. Para contornar esse problema, foi criada essa
 * possibilidade de login local forado para o administrador poder atribuir a
 * usurios enquadrados nessa situao, enquanto uma soluo mais geral no for
 * encontrada.
 * </p>
 * 
 * <p>
 * NOTA 2: a sigla "DTO" usada na documentao refere-se ao padro <i>Data
 * Transfer Object</i>, utilizado por um objeto que centraliza as informaes do
 * usurio (ver {@link UserInfoDialog#userInfo}).
 * </p>
 * 
 * @author Leonardo Barros
 */
public class AdminModifyUserInfoDialog extends ModifyUserInfoDialog {
  /** Combo de grupo de usurios */
  private JComboBox userGroupComboBox;

  /** Checkbox de login local forado */
  private JCheckBox forceLocalLoginCheckBox;

  /** Boto de perfis do usurio */
  private JButton roleButton;

  /** Boto de permisses do usurio */
  private JButton permissionButton;

  /**
   * Cria o dilogo de modificao de dados para uso exclusivo do administrador.
   * 
   * @param owner janela que criou este dilogo.
   * @param userId identificador do usurio sendo modificado.
   */
  public AdminModifyUserInfoDialog(Window owner, Object userId) {
    super(owner, userId);
    if (!User.getLoggedUser().isAdmin()) {
      throw new IllegalStateException("loggedUser != admin");
    }
  }

  /**
   * {@inheritDoc}<br>
   * Se o boto de senhas ainda no tiver sido adicioaonado, adiciona ao painel.
   * O boto s  habilitado se o administrador estiver editando seus prprio
   * dados. Adiciona botes de seleo de perfis e permisses ao painel.
   */
  @Override
  protected JPanel makeMiddlePanel() {
    JPanel panel = super.makeMiddlePanel();
    boolean forceLocalLogin = false;
    Object attribute = user.getAttribute(User.FORCE_LOCAL_LOGIN);
    if (attribute != null) {
      forceLocalLogin = ((Boolean) attribute).booleanValue();
    }

    // #TODO Revisar a incluso do boto de senhas na superclasse e neste
    // mtodo. A implementao atual, embora correta, est confusa.
    if (!ClientRemoteMonitor.getInstance().canChangePasswords()
      && !forceLocalLogin) {
      passwordButton = makePasswordButton();
      passwordButton.setEnabled(user.isAdmin());
      panel.add(passwordButton);
    }
    roleButton = makeRoleButton();
    panel.add(roleButton);
    permissionButton = makePermissionButton();
    panel.add(permissionButton);

    final JComponent[] buttons =
      new JComponent[] { roleButton, permissionButton, passwordButton };
    GUIUtils.matchPreferredSizes(buttons);
    return panel;
  }

  /**
   * Cria o boto de seleo de perfis, atribuindo-lhe como ao a criao do
   * dilogo de seleo de perfis, alm da habilitao do boto de persistncia
   * das alteraes aps a seleo, caso os requisitos estejam preenchidos.
   * 
   * @return boto de seleo de perfis.
   */
  private JButton makeRoleButton() {
    JButton button = new JButton(LNG.get("IAS_USER_ROLES_BUTTON"));
    button.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        new UserRoleDefinitionDialog(userInfoDialog, userInfo, user, false)
          .showDialog();
        enableModifyUserButtonIfAllowed();
      }
    });
    return button;
  }

  /**
   * Cria o boto de seleo de permisses, atribuindo-lhe como ao a criao
   * do dilogo de seleo de permisses, alm da habilitao do boto de
   * persistncia das alteraes aps a seleo, caso os requisitos estejam
   * preenchidos.
   * 
   * @return boto de seleo de permisses.
   */
  private JButton makePermissionButton() {
    JButton button = new JButton(LNG.get("IAS_USER_PERMISSIONS_BUTTON"));
    button.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        new UserPermissionDefinitionDialog(userInfoDialog, userInfo, user,
          false).showDialog();
        enableModifyUserButtonIfAllowed();
      }
    });
    return button;
  }

  /**
   * {@inheritDoc}<br>
   * Verifica tambm se a seleo de grupos de usurio mudou, ou se o estado do
   * checkbox de login local forado mudou (os dois s vo estar disponveis se
   * o administrador estiver editando os dados de um usurio comum, e mesmo
   * assim o ltimo s ser visvel quando o servidor no estiver autenticando
   * localmente).
   */
  @Override
  protected boolean unsavedDataInput() {
    boolean userGroupChanged = false;
    boolean forceLocalLoginChanged = false;
    if (!user.isAdmin()) {
      userGroupChanged = userGroupChanged();
      if (!ClientRemoteMonitor.getInstance().isLocalLogin()) {
        forceLocalLoginChanged = forceLocalLoginChanged();
      }
    }
    return super.unsavedDataInput() || userGroupChanged
      || forceLocalLoginChanged || roleSelectionChanged()
      || permissionSelectionChanged();
  }

  /**
   * Indica se houve mudana na seleo do checkbox de login local forado.
   * 
   * @return true se a seleo de login local forado  diferente do respectivo
   *         atributo do usurio, false caso contrrio.
   */
  private boolean forceLocalLoginChanged() {
    boolean selectedForceLocalLogin = forceLocalLoginCheckBox.isSelected();
    Object forceLocalLoginObj = user.getAttribute(User.FORCE_LOCAL_LOGIN);
    boolean userForceLocalLogin = false;
    if (forceLocalLoginObj != null) {
      userForceLocalLogin = ((Boolean) forceLocalLoginObj).booleanValue();
    }
    return (selectedForceLocalLogin != userForceLocalLogin);
  }

  /**
   * Indica se houve mudana na seleo do grupo de usurios.
   * 
   * @return true se o grupo selecionado  diferente do grupo em que o usurio
   *         est atualmente alocado, false caso contrrio.
   */
  private boolean userGroupChanged() {
    // Pega o grupo de usurios selecionado
    UserGroupName selectedGroupName =
      (UserGroupName) userGroupComboBox.getSelectedItem();

    // Pega o grupo de usurios ao qual o usurio pertence
    UserGroup userGroup;
    try {
      userGroup =
        UserGroup.getUserGroup(user.getAttribute(UserGroup.USERGROUP_ID));
    }
    catch (Exception e) {
      StandardErrorDialogs.showErrorDialog(userInfoDialog, LNG.get("ERRO")
        + " - " + dialogTitle, e);
      // Erro na captura de informaes.  melhor evitar ativar o boto de
      // modificao.
      return false;
    }
    UserGroupName userGroupName = new UserGroupName(userGroup);

    // Compara ambos
    return (!selectedGroupName.equals(userGroupName));
  }

  /**
   * Indica se a seleo de permisses mudou.
   * 
   * @return true se houve mudana na seleo de permisses, false caso
   *         contrrio.
   */
  private boolean permissionSelectionChanged() {
    Object[] selectedPermissionIds =
      (Object[]) userInfo.getAttribute(User.PERMISSION_IDS);
    Arrays.sort(selectedPermissionIds);
    Object[] userPermissionIds =
      (Object[]) user.getAttribute(User.PERMISSION_IDS);
    Arrays.sort(userPermissionIds);
    return !Arrays.equals(selectedPermissionIds, userPermissionIds);
  }

  /**
   * Indica se a seleo de perfis mudou.
   * 
   * @return true se houve mudana na seleo de perfis, false caso contrrio.
   */
  private boolean roleSelectionChanged() {
    Object[] selectedRoleIds = (Object[]) userInfo.getAttribute(User.ROLE_IDS);
    Arrays.sort(selectedRoleIds);
    Object[] userRoleIds = (Object[]) user.getAttribute(User.ROLE_IDS);
    Arrays.sort(userRoleIds);
    return !Arrays.equals(selectedRoleIds, userRoleIds);
  }

  /**
   * {@inheritDoc} Acrescenta ao DTO a seleo de grupo de usurio e do checkbox
   * de login local forado, quando estes campos estiverem disponveis.
   */
  @Override
  protected void updateUserInfo() {
    super.updateUserInfo();
    if (!user.isAdmin()) {
      UserGroupName userGroupName =
        (UserGroupName) userGroupComboBox.getSelectedItem();
      UserGroup userGroup = userGroupName.getUserGroup();
      userInfo.setAttribute(UserGroup.USERGROUP_ID, userGroup.getId());
      if (!ClientRemoteMonitor.getInstance().isLocalLogin()) {
        Boolean forceLocalLogin =
          new Boolean(forceLocalLoginCheckBox.isSelected());
        userInfo.setAttribute(User.FORCE_LOCAL_LOGIN, forceLocalLogin);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<JComponent[]> createDataComponents() {
    List<JComponent[]> components = super.createDataComponents();
    if (!user.isAdmin()) {
      JLabel userGroupLabel = new JLabel(LNG.get("IAS_USER_USERGROUP"));
      userGroupComboBox = makeUserGroupComboBox();
      components.add(new JComponent[] { userGroupLabel, userGroupComboBox });
      if (!ClientRemoteMonitor.getInstance().isLocalLogin()) {
        JLabel forceLoginLabel = new JLabel(LNG.get("IAS_USER_AUTHENTICATION"));
        forceLocalLoginCheckBox = makeForceLocalLoginCheckBox();
        components.add(new JComponent[] { forceLoginLabel,
            forceLocalLoginCheckBox });
      }
    }
    return components;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void addDataComponentsListeners() {
    super.addDataComponentsListeners();
    ItemListener itemListener = new ItemListener() {
      @Override
      public void itemStateChanged(ItemEvent e) {
        fireOnDataChanged();
      }
    };
    if (userGroupComboBox != null) {
      userGroupComboBox.addItemListener(itemListener);
    }
    if (forceLocalLoginCheckBox != null) {
      forceLocalLoginCheckBox.addItemListener(itemListener);
    }
  }

  /**
   * Cria um checkbox para permitir ao administrador configurar o usurio sendo
   * editado para sempre autenticar-se localmente, independentemente do mtodo
   * de autenticao em uso no servidor. Esse checkbox s  exibido quando duas
   * condies forem satisfeitas: o administrador tem de estar editando um
   * usurio comum e o servidor tem de estar utilizando um mtodo de
   * autenticao que no-local.<br>
   * Sempre que esse checkbox for selecionado, ir habilitar/desabilitar o boto
   * de alterao de senha (estar habilitado quando o checkbox estiver marcado,
   * desabilitado quando este estiver desmarcado).
   * 
   * @return checkbox de login local forado.
   */
  private JCheckBox makeForceLocalLoginCheckBox() {
    final JCheckBox checkBox =
      new JCheckBox(LNG.get("IAS_USER_FORCE_LOCAL_LOGIN"));
    checkBox.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        passwordButton.setEnabled(checkBox.isSelected());
      }
    });
    return checkBox;
  }

  /**
   * {@inheritDoc}<br>
   * Acrescenta ao DTO as selees de grupo de usurio e atributo
   * "login local forado", quando estes estiverem disponveis. <br>
   * OBS: Os dilogos de informaes complementares (perfis, permisses e
   * senhas) se encarregam de atualizar o DTO, portanto no  necessrio
   * atualizar essas informaes aqui.
   */
  @Override
  protected void fillFieldsWithUserData() {
    super.fillFieldsWithUserData();
    // Obtm os dados do usurio (a obteno dos dados  realizada dentro de
    // uma RemoteTask para possibilitar o repaint da tela, pois o processo
    // pode ser demorado).
    RemoteTask<UserInfo> task = new RemoteTask<UserInfo>() {
      @Override
      public void performTask() throws Exception {
        if (!user.isAdmin()) {
          userInfo.setAttribute(UserGroup.USERGROUP_ID,
            user.getAttribute(UserGroup.USERGROUP_ID));
          userInfo.setAttribute(User.ROLE_IDS, user.getRoleIds());
          userInfo.setAttribute(User.PERMISSION_IDS, user.getPermissionIds());
          if (!ClientRemoteMonitor.getInstance().isLocalLogin()) {
            userInfo.setAttribute(User.FORCE_LOCAL_LOGIN,
              user.getAttribute(User.FORCE_LOCAL_LOGIN));
          }
        }
        setResult(userInfo);
      }
    };
    if (!task.execute(userInfoDialog, dialogTitle,
      LNG.get("IAS_WAITING_USER_INFO"))) {
      return;
    }
    userInfo = task.getResult();
    // Os dados so ento copiados para a interface
    if (!user.isAdmin()) {
      try {
        userGroupComboBox.setSelectedItem(new UserGroupName(UserGroup
          .getUserGroup(userInfo.getAttribute(UserGroup.USERGROUP_ID))));
      }
      catch (Exception e) {
        StandardErrorDialogs.showErrorDialog(userInfoDialog,
          LNG.get("IAS_USER_USERGROUP_SELECTION_ERROR"), e);
      }
      if (!ClientRemoteMonitor.getInstance().isLocalLogin()) {
        Object forceLocalLogin = userInfo.getAttribute(User.FORCE_LOCAL_LOGIN);
        if (forceLocalLogin != null && forceLocalLogin.equals(Boolean.TRUE)) {
          forceLocalLoginCheckBox.setSelected(true);
        }
      }
    }
  }

  /**
   * Constri o combo de grupos de usurios disponveis, ordenados
   * alfabeticamente.
   * 
   * @return o combo de grupos de usurios disponveis.
   * 
   * @throws IllegalStateException lanada quando no foi possvel obter os
   *         grupos de usurios.
   */
  private JComboBox makeUserGroupComboBox() {
    // #TODO Esse e outros mtodos e campos so idnticos aos da classe
    // IncludeUserInfoDialog: pensar em forma de evitar duplicao sem
    // comprometer arquitetura.
    JComboBox combo = new JComboBox();
    Vector<UserGroup> userGroups =
      UserGroupProxy.getAllUserGroups(userInfoDialog, dialogTitle,
        LNG.get("IAS_WAITING_ALL_USERGROUPS"));
    if (userGroups == null) {
      throw new IllegalStateException(LNG.get("ias.usergroup.retrieval_error"));
    }
    Collections.sort(userGroups, UserGroup.getNameComparator());
    int userGroupSize = userGroups.size();
    for (int i = 0; i < userGroupSize; i++) {
      combo.addItem(new UserGroupName(userGroups.get(i)));
    }
    if (userGroupSize > 0) {
      combo.setSelectedIndex(0);
    }
    combo.setBackground(Color.white);
    return combo;
  }
}
