/*
 * $Id$
 */
package csbase.client.algorithms.commands.cache.events;

import csbase.client.util.event.IEvent;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;

/**
 * Evento genrico para comandos um comando na cache.
 * 
 * @author Tecgraf / PUC-Rio
 * 
 * @see csbase.client.algorithms.commands.cache.CommandsCache
 */
public class CommandUpdatedEvent implements IEvent {

  /**
   * Tipo do evento.
   */
  public enum Type {
        /**
     * Exceo lanada pela ao de atualizao de comandos.
     */
    exception,
        /**
     * Comando criado.
     */
    created,
        /**
     * Comando atualizado.
     */
    updated,
    /**
     * Falha tentando recuperar as informaes do comando.
     */
    system_failure,
        /**
     * Comando terminou (no se sabe como).
     */
    end,
        /**
     * Comando terminou (com erro).
     */
    error,
        /**
     * Comando terminou (com sucesso).
     */
    success,
        /**
     * Comando teve falha de inicializao.
     */
    failed,
        /**
     * Comando cancelado.
     */
    killed,
        /**
     * Comando perdido.<br>
     * O SGA foi reiniciado e no meio tempo o comando terminou de forma no
     * usual.
     */
    lost,
        /**
     * Comando removido.
     */
    removed,
        /**
     * Comando terminou (no foi possvel obter o cdigo de sada).
     */
    no_code;

    /**
     * Retorna o tipo do evento a partir do tipo de finalizao do comando.
     * 
     * @param finalizationType O tipo de finalizao do comando.
     * @return O tipo do evento.
     */
    public static Type valueOf(CommandFinalizationType finalizationType) {
      switch (finalizationType) {
        case END:
          return Type.end;
        case SUCCESS:
          return Type.success;
        case EXECUTION_ERROR:
          return Type.error;
        case FAILED:
          return Type.failed;
        case KILLED:
          return Type.killed;
        case LOST:
          return Type.lost;
        case NO_EXIT_CODE:
          return Type.no_code;
        case NOT_FINISHED: // O comando ainda no terminou.
          return null;
        default: // UNKNOWN
          return Type.end;
      }
    }
  }

  /**
   * Tipo do evento gerado.
   */
  private Type type;
  /**
   * Exceo que pode ser gerada durante o processo de atualizao da cache.
   */
  private Exception exception;
  /**
   * A descrio de como a exceo foi gerada.
   */
  private String exceptionDescription;
  /**
   * Comando que foi atualizado.
   */
  private CommandInfo command;

  /**
   * Construtor utilizado para criar um evento indicando que uma exceo foi
   * lanada durante a atualizao dos comandos.
   * 
   * @param e a exceo que interrompeu a atualizaao dos comandos.
   * @param description a descrio de como a exceo foi gerada.
   */
  public CommandUpdatedEvent(Exception e, String description) {
    this.type = Type.exception;
    this.exception = e;
    this.exceptionDescription = description;
    this.command = null;
  }

  /**
   * Construtor.
   * 
   * @param command o comando que gerou o evento.
   * @param type tipo do evento.
   */
  public CommandUpdatedEvent(CommandInfo command, Type type) {
    if (Type.exception.equals(type)) {
      throw new IllegalArgumentException(
        "Tipo de evento invlido. O tipo exception s deve ser utilizado quando for lanada uma exceo durante o processo de atualizao dos comandos.");
    }

    this.command = command;
    this.type = type;
  }

  /**
   * Obtm a exceo que interrompeu a atualizaao dos comandos.
   * 
   * @return a exceo que interrompeu a atualizaao dos comandos ou
   *         <tt>null</tt> caso a atualizao dos comandos tenha ocorrido com
   *         sucesso.
   */
  public Exception getException() {
    return exception;
  }

  /**
   * Obtm a descrio de como a exceo foi gerada.
   * 
   * @return a descrio de como a exceo foi gerada.
   */
  public String getExceptionDescription() {
    return exceptionDescription;
  }

  /**
   * Obtm o comando que gerou o evento.
   * 
   * @return o comando que gerou o evento ou <tt>null</tt> caso uma exceo
   *         tenha sido lanada durante a atualizao dos comandos.
   */
  public CommandInfo getCommand() {
    return command;
  }

  /**
   * Obtm o tipo do evento gerado.
   * 
   * @return o tipo do evento gerado.
   */
  public Type getType() {
    return type;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((command == null) ? 0 : command.hashCode());
    result =
      prime * result
      + ((exception == null) ? 0 : exception.getClass().hashCode());
    result =
      prime
      * result
      + ((exceptionDescription == null) ? 0 : exceptionDescription.hashCode());
    result = prime * result + ((type == null) ? 0 : type.hashCode());
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    final CommandUpdatedEvent other = (CommandUpdatedEvent) obj;
    if (command == null) {
      if (other.command != null)
        return false;
    }
    else if (!command.equals(other.command))
      return false;
    if (exception == null) {
      if (other.exception != null)
        return false;
    }
    else if (!exception.getClass().equals(other.exception.getClass()))
      return false;
    if (exceptionDescription == null) {
      if (other.exceptionDescription != null)
        return false;
    }
    else if (!exceptionDescription.equals(other.exceptionDescription))
      return false;
    if (type == null) {
      if (other.type != null)
        return false;
    }
    else if (!type.equals(other.type))
      return false;
    return true;
  }
}