package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.Hashtable;
import java.util.List;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.Application;
import csbase.client.applications.algorithmsmanager.AlgorithmsManager;
import csbase.client.applications.algorithmsmanager.versiontree.AlgorithmPropertiesTableListener;
import csbase.client.applications.algorithmsmanager.versiontree.AlgorithmPropertiesTableModel;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.table.EditableCellRenderer;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmProperty;
import csbase.remote.AlgorithmServiceInterface;
import csbase.remote.ClientRemoteLocator;

/**
 * Classe que apresenta ao usurio uma tabela para o gerenciamento de
 * propriedades dos algoritmos. Essa tabela s pode ser vista e editada pelo
 * administrador do sistema.
 * 
 * @author valeria
 * 
 */
public class AlgorithmsPropertiesDialog extends DesktopComponentFrame implements
  AlgorithmPropertiesTableListener {
  /** Aplicao que criou a janela */
  private AlgorithmsManager application;

  /** Boto para cancelar a edio das propriedades */
  private JButton cancelButton;

  /** Boto para aplicar as alteraes das propriedades */
  private JButton applyButton;

  private JTable algoTable;
  private String[] columnNames = null;
  private Object[][] data = null;
  private List<AlgorithmProperty> properties = null;
  private AlgorithmInfo[] algos = null;
  private AlgorithmInfo[] sortedAlgos = null;
  private AlgorithmPropertiesTableModel model;
  private boolean askForConfirmation = false;
  private static int YES_OPTION = 0;
  private static int NO_OPTION = 1;

  /**
   * Exibe janela contendo a tabela de propriedades de algoritmos.
   * 
   * @param index identificador da janela
   * @param application aplicao que criou a janela
   */
  private AlgorithmsPropertiesDialog(Object index, AlgorithmsManager application) {
    super(index, application.getApplicationFrame(), LNG
      .get("ias.algorithm.AlgoManagerFrame.title.properties_management"));
    this.application = application;
    addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent ev) {
        showSaveDialog();
      }
    });
    getContentPane().setLayout(new GridBagLayout());

    JLabel description =
      new JLabel(getApplication().getString(
        "AlgorithmsPropertiesDialog.description"));

    // Cria a tabela com as propriedades
    algoTable = makeAlgoTable();
    JScrollPane tablePane = new JScrollPane(algoTable);
    tablePane
      .setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);

    getContentPane().add(description,
      new GBC(0, 0).none().west().insets(12, 12, 5, 12));
    getContentPane().add(tablePane,
      new GBC(0, 1).both().center().weights(1, 1).insets(12, 12, 12, 12));
    getContentPane().add(createButtonsPanel(),
      new GBC(0, 2).both().center().weights(0, 0).insets(0, 0, 5, 0));

    setPreferredSize(new Dimension(650, 350));
    pack();

    if (DesktopFrame.getInstance() == null) {
      setLocationRelativeTo(null);
    }
    else {
      center(DesktopFrame.getInstance().getDesktopFrame());
    }
    ((AlgorithmPropertiesTableModel) algoTable.getModel()).addListener(this);
    askForConfirmation = false;
  }

  /**
   * Constri a janela de gerenciamento de propriedades algoritmos.
   * 
   * @param application aplicao que criou a janela
   * @return a janela de gerenciamento de propriedade de algoritmos
   */
  public static AlgorithmsPropertiesDialog getFrame(
    AlgorithmsManager application) {
    String index = AlgorithmsPropertiesDialog.class.getName();
    DesktopComponentFrame frame = getDesktopComponentFrame(index);
    if (frame == null) {
      frame = new AlgorithmsPropertiesDialog(index, application);
    }
    return (AlgorithmsPropertiesDialog) frame;
  }

  /**
   * Monta a tabela de propriedade dos algoritmos
   * 
   * @return a tabela de propriedade dos algoritmos
   */
  private JTable makeAlgoTable() {
    final AlgorithmServiceInterface algoService =
      ClientRemoteLocator.algorithmService;
    // Obtm as chaves das propriedades dos algoritmos
    RemoteTask<List<AlgorithmProperty>> task =
      new RemoteTask<List<AlgorithmProperty>>() {
        @Override
        public void performTask() throws Exception {
          setResult(algoService.getAlgorithmProperties());
        }
      };
    boolean execute =
      task.execute(null, LNG.get("SCHEDULER_ACCESSING_TITLE"),
        LNG.get("SCHEDULER_ACCESSING_MESSAGE"));
    if (execute) {
      properties = task.getResult();
      columnNames = new String[properties.size() + 1];
      columnNames[0] =
        LNG.get("ias.algorithm.AlgoManagerFrame.label.algorithm");
      for (int i = 0; i < properties.size(); i++) {
        columnNames[i + 1] = properties.get(i).getLabel();
      }
    }
    // Obtm os valores das propriedades dos algoritmos
    algos =
      AlgorithmManagementProxy.getAllAlgorithmInfos(getApplication()
        .getApplicationFrame(),
        AlgorithmManagementProxy.AlgorithmOperation.ADMIN_ALGORITHM);
    Vector<AlgorithmInfo> sortedVectorAlgos = new Vector<AlgorithmInfo>();
    for (int i = 0; i < algos.length; i++) {
      sortedVectorAlgos.add(algos[i]);
    }
    Collections.sort(sortedVectorAlgos);

    sortedAlgos =
      sortedVectorAlgos.toArray(new AlgorithmInfo[sortedVectorAlgos.size()]);
    data = new String[sortedAlgos.length][columnNames.length];
    for (int i = 0; i < sortedAlgos.length; i++) {
      data[i][0] = sortedAlgos[i].getName();
    }
    for (int i = 1; i < columnNames.length; i++) {
      for (int j = 0; j < sortedAlgos.length; j++) {
        String propKey = properties.get(i - 1).getKey();
        String propValue = sortedAlgos[j].getPropertyValue(propKey);
        data[j][i] = (propValue == null) ? "" : propValue;
      }
    }

    model = new AlgorithmPropertiesTableModel(data, columnNames);
    JTable algoPropertiesTable = new JTable(model);
    algoPropertiesTable.setDefaultRenderer(Object.class,
      new EditableCellRenderer());
    algoPropertiesTable.putClientProperty("terminateEditOnFocusLost",
      Boolean.TRUE);
    return algoPropertiesTable;
  }

  /**
   * Cria o painel de botes.
   * 
   * @return painel de botes
   */
  private JPanel createButtonsPanel() {
    JPanel buttonPanel = new JPanel(new FlowLayout());
    cancelButton =
      new JButton(getApplication().getString(
        "AlgorithmsPropertiesDialog.button.cancel"));
    cancelButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent ev) {
        showSaveDialog();
      }
    });
    applyButton =
      new JButton(getApplication().getString(
        "AlgorithmsPropertiesDialog.button.apply"));
    applyButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent ev) {
        saveChanges();
      }
    });
    buttonPanel.add(applyButton);
    buttonPanel.add(cancelButton);
    ClientUtilities.adjustEqualSizes(cancelButton, applyButton);
    return buttonPanel;
  }

  /**
   * Obtm a aplicao.
   * 
   * @return a aplicao
   */
  private Application getApplication() {
    return application;
  }

  /**
   * Valida as propriedades dos algoritmos. Retorna true quando as propriedades
   * esto no formato vlido e false caso contrrio.
   * 
   * @return true quando as propriedades esto no formato vlido e false caso
   *         contrrio.
   */
  private boolean validateProperties() {
    boolean isValid = true;
    StringBuilder invalidFields = new StringBuilder();
    for (int i = 0; i < algoTable.getRowCount(); i++) {
      AlgorithmInfo algoInfo = sortedAlgos[i];
      for (int j = 0; j < properties.size(); j++) {
        AlgorithmProperty algoProp = properties.get(j);
        String value = (String) algoTable.getValueAt(i, j + 1);
        if (!algoProp.isValidValue(value)) {
          invalidFields.append("\n  - " + algoInfo.getName() + ": "
            + algoProp.getLabel());
          isValid = false;
        }
      }
    }
    // H variveis invlidas
    if (!isValid) {
      String dialogTitle = LNG.get("algomanager.title.algo_properties");
      String title = LNG.get("algomanager.title.error") + " - " + dialogTitle;
      StringBuilder msg = new StringBuilder();
      msg.append(LNG.get("algomanager.error.validation.properties.1"));
      msg.append("\n");
      msg.append(invalidFields);
      msg.append("\n\n");
      msg.append(LNG.get("algomanager.error.validation.properties.2"));
      StandardDialogs.showErrorDialog(this, title, msg.toString());
    }
    return isValid;
  }

  /**
   * Chama o mtodo remoto para atualizao das propriedades dos algoritmos se
   * elas tiverem sido alteradas.
   */
  private void fireTableRowUpdated() {
    boolean reloadAlgorithms = false;
    for (int i = 0; i < algoTable.getRowCount(); i++) {
      Hashtable<String, String> algoProperties =
        sortedAlgos[i].getPropertyValues();
      Hashtable<String, String> newProperties = new Hashtable<String, String>();
      String propKey = null;
      String propValue = null;
      boolean update = false;
      for (int j = 0; j < properties.size(); j++) {
        AlgorithmProperty algoProp = properties.get(j);
        String newPropValue = ((String) algoTable.getValueAt(i, j + 1)).trim();
        propKey = algoProp.getKey();
        propValue = algoProperties.get(propKey);
        if (newPropValue.length() == 0 && propValue == null) {
          continue;
        }
        if (!newPropValue.equals(propValue)) {
          update = true;
        }
        newProperties.put(propKey, newPropValue);
      }
      if (update) {
        reloadAlgorithms = true;

        String algoName = sortedAlgos[i].getName();
        sortedAlgos[i] =
          AlgorithmManagementProxy.changeAlgorithmProperties(algoName,
            newProperties, this);
        String dialogTitle = LNG.get("algomanager.title.algo_properties");
        if (sortedAlgos[i] == null) {
          String title =
            LNG.get("algomanager.title.error") + " - " + dialogTitle;
          StandardErrorDialogs.showErrorDialog(this, title, MessageFormat
            .format(LNG.get("algomanager.error.update"),
              new Object[] { algoName }));
        }
      }
    }
    askForConfirmation = false;

    if (reloadAlgorithms) {
      /*
       * Recarrega os algoritmos para que as alteraes se reflitam nos outros
       * usurios e no SchedulerService.
       */
      boolean reloaded = AlgorithmManagementProxy.reloadAlgorithms(this);
      if (reloaded) {
        StandardDialogs.showInfoDialog(this, LNG.get("algorithm.title.reload"),
          LNG.get("algorithm.msg.reload"));
      }

    }
  }

  /**
   * Salva as alteraes da tabela e fecha a janela.
   */
  private void saveChanges() {
    if (validateProperties()) {
      fireTableRowUpdated();
      close();
    }
  }

  /**
   * Apresenta um dilogo de confirmao se o usurio deseja salvar alteraes
   * feitas nas propriedades dos algoritmos.
   */
  private void showSaveDialog() {
    int n = NO_OPTION;
    if (askForConfirmation) {
      Object[] options = { LNG.get("IAS_YES"), LNG.get("IAS_NO") };
      n =
        JOptionPane.showOptionDialog(this,
          LNG.get("ias.algorithm.AlgoManagerFrame.question.properties_change"),
          LNG.get("algomanager.title.properties_change"),
          JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, null,
          options, options[0]);
    }
    if (n != YES_OPTION) {
      close();
    }
    else {
      saveChanges();
    }
  }

  /**
   * @see csbase.client.applications.algorithmsmanager.versiontree.AlgorithmPropertiesTableListener#wasModified()
   */
  public void wasModified() {
    askForConfirmation = true;
  }
}
