/*
 * $Id: DesktopFrameOpenBusEventHandler.java 109138 2010-08-16 23:17:25Z clinio
 * $
 */
package csbase.client.desktop;

import java.rmi.RemoteException;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.Client;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applicationmanager.ApplicationType;
import csbase.client.openbus.OpenBusApplicationInstanceData;
import csbase.client.openbus.OpenBusEventHandler;
import csbase.client.openbus.OpenBusStandardEvents;
import csbase.logic.CommonClientProject;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Handler padro de um desktop CSBASE.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DesktopFrameOpenBusEventHandler extends OpenBusEventHandler {

  /**
   * O desktop.
   */
  final private DesktopFrame desktop;

  /**
   * Flag que desktop recebeu pedido de shutdown.
   */
  private Boolean isOnShutdownByRequest = false;

  /**
   * Envia mensagens do tipo
   * {@link OpenBusStandardEvents#DESKTOP_VISIBILITY_INFO}.
   * 
   * @param shown indicativo
   */
  final protected void fireDesktopVisibilityInfo(final boolean shown) {
    final String fatherId = getFatherClientInstanceId();
    final OpenBusStandardEvents ev =
      OpenBusStandardEvents.DESKTOP_VISIBILITY_INFO;
    ev.sendEvent(this, fatherId, shown);
  }

  /**
   * Executa aplicativos e envia mensagens do tipo:
   * {@link OpenBusStandardEvents#APP_EXECUTION_SUCCESS_RSP} ou
   * {@link OpenBusStandardEvents#APP_EXECUTION_FAILURE_RSP}.
   * 
   * @param sourceId id do cliente originrio da mensagem
   * @param appId id da aplicao
   * @return indicativo de tratamnto do evento.
   */
  final protected boolean fireInsideApplication(final String sourceId,
    final String appId) {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    final ApplicationRegistry reg = appManager.getApplicationRegistry(appId);
    if (reg == null) {
      return false;
    }

    try {
      final ApplicationType appInstance = appManager.runApplication(appId);
      final OpenBusStandardEvents okType =
        OpenBusStandardEvents.APP_EXECUTION_SUCCESS_RSP;
      final OpenBusApplicationInstanceData data =
        new OpenBusApplicationInstanceData(appId, appInstance);
      final String stringValue = data.getEventValue();
      okType.sendEvent(this, sourceId, stringValue);
      return true;
    }
    catch (final ApplicationException e) {
      final OpenBusStandardEvents okType =
        OpenBusStandardEvents.APP_EXECUTION_FAILURE_RSP;
      okType.sendEvent(this, sourceId, appId);
      return false;
    }
  }

  /**
   * Dispara evento de falha de abertura de projeto.
   * 
   * @param destId id do destino.
   * @param stringValue projeto em questo
   * @return indicativo de envio.
   */
  final protected boolean firePrjOpenFailedRsp(final String destId,
    final String stringValue) {
    final String errMsg = getString("project.remote.open.error");
    final String failedMsg = errMsg + ": " + stringValue;
    desktop.displayTrayErrorMessage(failedMsg);
    final OpenBusStandardEvents ev = OpenBusStandardEvents.PRJ_OPEN_FAILED_RSP;
    return ev.sendEvent(this, destId, stringValue);
  }

  /**
   * @param destId id do destino
   * @param stringValue projeto.
   * @return indicativo de envio.
   */
  final protected boolean firePrjOpenSuccessRsp(final String destId,
    final String stringValue) {
    final OpenBusStandardEvents ev = OpenBusStandardEvents.PRJ_OPEN_SUCCESS_RSP;
    return ev.sendEvent(this, destId, stringValue);
  }

  /**
   * Monta um texto de internacionalizao.
   * 
   * @param tag a tag.
   * @return o texto
   */
  final private String getString(final String tag) {
    final String className = this.getClass().getSimpleName();
    final String formattedTag = className + "." + tag;
    return LNG.get(formattedTag);
  }

  /**
   * Tratador de: {@link OpenBusStandardEvents#DESKTOP_SHUTDOWN_REQ}
   * 
   * @param sourceId id da origem
   * @return indicativo de tratamento.
   */
  private boolean handleDesktopShutdownReq(final String sourceId) {
    final Client client = Client.getInstance();
    final String fatherId = client.getFatherClientInstanceId();

    if (sourceId.equals(fatherId)) {
      setShutDownByRequest();
      desktop.shutdownDesktop();
    }
    return true;
  }

  /**
   * Tratador do evento: {@link OpenBusStandardEvents#DESKTOP_VISIBILITY_REQ}.
   * 
   * @param flag indicativo
   * @return indicativo de evento processado.
   */
  private boolean handleDesktopVisibilityReq(final boolean flag) {
    desktop.setDesktopVisible(flag);
    if (!flag) {
      final Client client = Client.getInstance();
      desktop.displayTrayInfoMessage(client.getSystemName());
    }
    return true;
  }

  /**
   * @param sourceId origem
   * @return indica o tratamento.
   */
  private boolean handleLauncherEndInfo(final String sourceId) {
    final Client client = Client.getInstance();
    final String fatherId = client.getFatherClientInstanceId();
    if (sourceId.equals(fatherId)) {
      System.out.println("Father died!");
    }
    return true;
  }

  /**
   * Tratador de: {@link OpenBusStandardEvents#PRJ_OPEN_REQ}.
   * 
   * @param sourceId identificador da origem do evento.
   * @param stringValue id do projeto a ser aberto.
   * @return indicativo de tratamento do evento.
   */
  private boolean handlePrjOpenReq(final String sourceId,
    final String stringValue) {
    final String[] splitted = stringValue.split("::");
    if (splitted == null || splitted.length != 2) {
      return firePrjOpenFailedRsp(sourceId, stringValue);
    }
    final String userId = splitted[0];
    final String projectName = splitted[1];

    // ATENO!
    // Isso vai ser apagado quando o DataKey levar o identificador do projeto:
    Object projectId = userId + "/" + projectName;

    CommonClientProject prj;
    try {
      prj = CommonClientProject.openProject(projectId, true);
      final boolean changed = setCurrentProject(prj);
      if (!changed) {
        return firePrjOpenFailedRsp(sourceId, stringValue);
      }
    }
    catch (final RemoteException e) {
      return firePrjOpenFailedRsp(sourceId, stringValue);
    }
    if (prj == null) {
      return firePrjOpenFailedRsp(sourceId, stringValue);
    }
    return firePrjOpenSuccessRsp(sourceId, stringValue);
  }

  /**
   * Consulta o atributo (ver documentao de {@link #isOnShutdownByRequest}).
   * 
   * @return o atributo.
   */
  private final Boolean isOnShutdownByRequest() {
    synchronized (this.isOnShutdownByRequest) {
      return this.isOnShutdownByRequest;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean receiveEventWithBooleanValue(final String sourceId,
    final String destId, final String type, final boolean booleanValue) {
    if (!isEventFromFatherAndToMe(sourceId, destId)) {
      return false;
    }

    if (OpenBusStandardEvents.DESKTOP_VISIBILITY_REQ.isMyType(type)) {
      return handleDesktopVisibilityReq(booleanValue);
    }

    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean receiveEventWithNoValue(final String sourceId,
    final String destId, final String type) {
    if (!isEventFromFatherAndToMe(sourceId, destId)) {
      return false;
    }

    final OpenBusStandardEvents launcherEndType =
      OpenBusStandardEvents.DSKLAUNCHER_END_INFO;
    if (launcherEndType.isMyType(type)) {
      return handleLauncherEndInfo(sourceId);
    }

    final OpenBusStandardEvents shutdownType =
      OpenBusStandardEvents.DESKTOP_SHUTDOWN_REQ;
    if (shutdownType.isMyType(type)) {
      return handleDesktopShutdownReq(sourceId);
    }

    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean receiveEventWithStringValue(final String sourceId,
    final String destId, final String type, final String stringValue) {
    if (!isEventFromFatherAndToMe(sourceId, destId)) {
      return false;
    }

    if (OpenBusStandardEvents.APP_EXECUTION_REQ.isMyType(type)) {
      final String appId = stringValue;
      return fireInsideApplication(sourceId, appId);
    }

    if (OpenBusStandardEvents.PRJ_OPEN_REQ.isMyType(type)) {
      return handlePrjOpenReq(sourceId, stringValue);
    }

    return false;
  }

  /**
   * Ajuste do projeto corrente.
   * 
   * @param prj projeto
   * @return indicativo
   */
  private boolean setCurrentProject(final CommonClientProject prj) {
    try {
      SwingThreadDispatcher.invokeAndWait(new Runnable() {
        @Override
        public void run() {
          desktop.setCurrentProject(prj);
        }
      });
    }
    catch (final Exception e) {
      return false;
    }
    return true;
  }

  /**
   * ajuste o atributo para {@code true} (ver documentao de
   * {@link #isOnShutdownByRequest}).
   */
  private final void setShutDownByRequest() {
    synchronized (this.isOnShutdownByRequest) {
      this.isOnShutdownByRequest = true;
    }
  }

  /**
   * Mtodo de finalizao; envia mensagens do tipo:
   * {@link OpenBusStandardEvents#DESKTOP_END_INFO}.
   */
  public void shutdown() {
    if (!isOnShutdownByRequest()) {
      final OpenBusStandardEvents type = OpenBusStandardEvents.DESKTOP_END_INFO;
      final String fatherId = getFatherClientInstanceId();
      type.sendEvent(this, fatherId, null);
    }
  }

  /**
   * Construtor
   * 
   * @param desktop o desktop
   */
  public DesktopFrameOpenBusEventHandler(final DesktopFrame desktop) {
    this.desktop = desktop;
    final OpenBusStandardEvents type = OpenBusStandardEvents.DESKTOP_START_INFO;
    final String fatherId = getFatherClientInstanceId();
    type.sendEvent(this, fatherId, null);
  }
}
