package csbase.client.applications.flowapplication.graph;

import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

/**
 * Define o posicionamento dos ns do grafo de forma a criar uma nica coluna
 * centralizada.
 */
public class SingleColumnGraphLayout implements GraphLayout {

  /**
   * {@inheritDoc}
   */
  @Override
  public void doLayout(Graph graph) {
    Collection<GraphNode> nodeCollection = graph.getNodeCollection();
    GraphNode topNode = findRootNode(nodeCollection);
    if (topNode == null) {
      // O layout no se aplica pois no h uma raiz.
      return;
    }
    Point2D position = findRootPosition(graph);
    // Segue o grafo atravs das ligaes, atribuindo as novas posies aos ns.
    for (int i = 0; i < nodeCollection.size(); i++) {
      topNode.setLocation(position);
      Collection<GraphFileDescriptor> outputs =
        topNode.getOutputFileDescriptorCollection();
      for (GraphFileDescriptor output : outputs) {
        Collection<GraphLink> links = output.getLinkFromCollection();
        if (!links.isEmpty()) {
          GraphLink link = links.iterator().next();
          if (link != null && link.isWellFormed()) {
            Rectangle2D bounds = topNode.getBounds2D();
            Rectangle2D marginBounds = Grid.includeMargin(bounds);
            double newY = marginBounds.getMaxY() + Graph.NODE_INSET;
            position.setLocation(position.getX(), newY);
            topNode = link.getEndNode();
          }
        }
      }
    }
    // Aplica o layout de ligaes para que fiquem todas retas, sem pontos intermedirios.
    OrthogonalLinkGraphLayout layout = new OrthogonalLinkGraphLayout();
    layout.doLayout(graph);
  }

  /**
   * Determina a posio da raiz do grafo.
   * 
   * @param graph o grafo.
   * @return a posio da raiz do grafo.
   */
  private Point2D findRootPosition(Graph graph) {
    double rootNodeX = Graph.INITIAL_POINT.getX();
    for (GraphNode node : graph.getNodeCollection()) {
      // Determina o centro da coluna nica.
      rootNodeX = Math.max(rootNodeX, node.getBounds2D().getCenterX());
    }

    double rootNodeY = Graph.INITIAL_POINT.getY();

    Point2D position = new Point2D.Double(rootNodeX, rootNodeY);
    return position;
  }

  /**
   * Obtm o n raiz do grafo, ou seja, o n que no deve ser colocado no topo
   * do grafo.
   * 
   * @param nodeCollection o conjunto de ns.
   * @return o n raiz ou falso se no puder se determinar um nico n raiz.
   */
  private GraphNode findRootNode(Collection<GraphNode> nodeCollection) {
    GraphNode topNode = null;
    for (GraphNode node : nodeCollection) {
      if (hasMultipleOutputLinks(node)) {
        return null;
      }
      boolean hasParent = false;
      Collection<GraphFileDescriptor> inputs =
        node.getInputFileDescriptorCollection();
      for (GraphFileDescriptor input : inputs) {
        GraphLink link = input.getLinkTo();
        if (link != null && link.isWellFormed()) {
          hasParent = true;
        }
      }
      if (!hasParent) {
        if (topNode == null) {
          topNode = node;
        }
        else {
          // Existem duas razes no grafo, no tem como fazer uma nica coluna.
          return null;
        }
      }
    }
    return topNode;
  }

  /**
   * Indica se um n contm mais de uma ligao de sada.
   * 
   * @param node o n a ser testado.
   * @return verdadeiro se o n contm mais de uma ligao de sada ou falso,
   *         caso contrrio.
   */
  private boolean hasMultipleOutputLinks(GraphNode node) {
    Collection<GraphFileDescriptor> outputs =
      node.getOutputFileDescriptorCollection();
    for (GraphFileDescriptor output : outputs) {
      Collection<GraphLink> linkFromCollection = output.getLinkFromCollection();
      if (linkFromCollection.size() > 1) {
        // O n tem mais de um filho, no tem como fazer uma nica coluna.
        return true;
      }
    }
    return false;
  }

  /**
   * Determina se o grafo  conexo.
   * 
   * @param graph o grafo.
   * @return verdadeiro se o grafo  conexo ou falso, caso contrrio.
   */
  private boolean isFullyConnected(Graph graph) {
    Collection<GraphLink> linkCollection = graph.getLinkCollection();
    Set<GraphNode> connectedNodes = new HashSet<GraphNode>();
    for (GraphLink link : linkCollection) {
      if (link.isWellFormed()) {
        connectedNodes.add(link.getStartNode());
        connectedNodes.add(link.getEndNode());
      }
    }
    Collection<GraphNode> nodes = graph.getNodeCollection();
    return (nodes.size() == connectedNodes.size());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canApplyLayout(Graph graph) {
    return isFullyConnected(graph)
      && findRootNode(graph.getNodeCollection()) != null;
  }

}
