/*
 * $Id$
 */
package csbase.client.project.tasks;

import csbase.logic.ClientProjectFile;

/**
 * Classe que implementa a tarefa de consultar os filhos de um diretrio em um
 * projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class GetChildrenTask extends CoreTask<ClientProjectFile[]> {

  /**
   * Diretrio.
   */
  private ClientProjectFile dir;
  /**
   * Flag que indica se a subrvore deve ser obtida recursivamente.
   */
  private final boolean recursive;
  /**
   * Flag que indica se o cache local deve ser descartado.
   */
  private final boolean discardLocal;

  /**
   * Construtor. O cache local ser utilizado, e os ns obtidos do servidor
   * <b>no</b> estaro com seus filhos preenchidos.
   * 
   * @param dir diretrio
   */
  public GetChildrenTask(final ClientProjectFile dir) {
    this(dir, false, false);
  }

  /**
   * Construtor.
   * 
   * @param dir diretrio
   * @param discardLocal <code>true</code> se devemos descartar a cache local e
   *        obter novamente os dados do servidor
   * @param recursive <code>true</code> se toda a subrvore deve ser obtida
   *        recursivamente
   */
  public GetChildrenTask(ClientProjectFile dir, boolean discardLocal,
    boolean recursive) {
    this.dir = dir;
    this.discardLocal = discardLocal;
    this.recursive = recursive;
  }

  /**
   * {@inheritDoc}
   * 
   * @see ClientProjectFile#getChildren()
   */
  @Override
  protected void performTask() throws Exception {
    ClientProjectFile[] children = dir.getChildren(discardLocal, recursive);
    setResult(children);
  }

  /**
   * Cria uma instncia da task e a executa.
   * 
   * @param dir n para o qual se deseja obter os filhos
   * @return array com os filhos do diretrio corrente
   */
  static final public ClientProjectFile[] runTask(final ClientProjectFile dir) {
    return runTask(dir, false, false);
  }

  /**
   * Cria uma instncia da task e a executa.
   * 
   * @param dir diretrio
   * @param discardLocal <code>true</code> se devemos descartar a cache local e
   *        obter novamente os dados do servidor
   * @param recursive <code>true</code> se toda a subrvore deve ser obtida
   *        recursivamente
   * @return array com os filhos do diretrio corrente
   */
  static final public ClientProjectFile[] runTask(final ClientProjectFile dir,
    boolean discardLocal, boolean recursive) {
    final GetChildrenTask task =
      new GetChildrenTask(dir, discardLocal, recursive);
    final String[] args = new String[] { dir.toString() };
    final boolean hasResult = task.execute(args);
    if (!hasResult) {
      return null;
    }

    final ClientProjectFile[] children = task.getResult();
    return children;
  }
}
