/**
 * $Id$
 */
package csbase.client.project.tasks;

import java.awt.Dialog.ModalityType;

import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.ClientServerManager;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.ProjectTree;
import csbase.client.remote.ClientProjectObserver;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.CommonClientProject;
import csbase.logic.UserProjectInfo;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Abre um projeto no servidor e atualiza numa rvore de projetos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class OpenProjectOnTreeRemoteTask extends OpenProjectRemoteTask {
  /**
   * A rvore de projetos.
   */
  private ProjectTree tree;

  /**
   * Abre um projeto no servidor e atualiza numa rvore de projetos. A task 
   * criada no modo de bloqueio de janelas {@link ModalityType#DOCUMENT_MODAL}.
   * 
   * @param projectId Identificador do projeto.
   * @param tree A rvore de projetos.
   */
  public OpenProjectOnTreeRemoteTask(Object projectId, ProjectTree tree) {
    super(projectId, ModalityType.DOCUMENT_MODAL);
    this.tree = tree;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected final void performTask() throws Exception {
    CommonClientProject project = this.tree.getProject();
    if (project != null) {
      throw new IllegalArgumentException("J existe projeto aberto.");
    }
    super.performTask();
    project = this.getResult();
    ClientRemoteLocator.server.setServerSessionProperty(ClientServerManager
      .getInstance().getSession().getKey(),
      ProjectServiceInterface.CURRENT_PROJECT_SESSION_PROPERTY,
      new UserProjectInfo(project.getId(), project.getName(), project
        .getUserId()));
    try {
      project.installObserver(new ClientProjectObserver(project));
    }
    catch (Exception e) {
      DesktopFrame desktop = DesktopFrame.getInstance();
      StandardErrorDialogs.showErrorDialog(desktop.getDesktopFrame(), desktop
        .getTitle(), e);
    }
    SwingThreadDispatcher.invokeLater(new TreeUpdater(this.tree, project));
  }

  /**
   * Atualiza uma rvore de projetos com um projeto.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private static class TreeUpdater implements Runnable {
    /**
     * A rvore de projetos.
     */
    private ProjectTree tree;
    /**
     * O projeto.
     */
    private CommonClientProject project;

    /**
     * Atualiza uma rvore de projetos com um projeto.
     * 
     * @param tree A rvore de projetos.
     * @param project O projeto.
     */
    TreeUpdater(ProjectTree tree, CommonClientProject project) {
      this.tree = tree;
      this.project = project;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      this.tree.setProject(this.project);
    }
  }
}
