/*
 * $Id$
 */

package csbase.client.applications.commandsmonitor.actions;

import java.awt.event.ActionEvent;

import csbase.client.applications.AbstractSimpleApplicationAction;
import csbase.client.applications.Application;
import csbase.client.desktop.RemoteTask;
import csbase.client.remote.srvproxies.messageservice.MessageProxy;
import csbase.client.util.event.EventListener;
import csbase.client.util.event.EventManager;
import csbase.client.util.event.IEvent;
import csbase.logic.SchedulerStateChangedEvent;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;

/**
 * Liga/desliga o agendador de comandos.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class SchedulerSwitchAction extends
  AbstractSimpleApplicationAction<Application> {

  /**
   * Gerente de eventos da aplicao que detm esta ao.<br>
   * Permite lanar um evento indicando mudanas de estado - (des)bloqueado - do
   * agendador de comandos.
   */
  private EventManager eventManager = new EventManager();
  /**
   * <tt>true</tt> indica que o agendador de comandos est bloqueado.
   */
  private boolean blocked;

  /**
   * Construtor.
   * 
   * @param application Aplicao que detm essa ao.
   */
  public SchedulerSwitchAction(Application application) {
    super(application);

    setEnabled(User.getLoggedUser().isAdmin());
    registerListetener();
    updateState();
  }

  /**
   * Registra o ouvinte de mudana de estado do Scheduler.
   */
  private void registerListetener() {
    MessageProxy.addListener(new IMessageListener() {
      @Override
      public void onMessagesReceived(Message... messages) throws Exception {
        for (Message aMessage : messages) {
          SchedulerStateChangedEvent event =
            (SchedulerStateChangedEvent) aMessage.getBody();
          setSchedulerBlocked(event.isBlocked());
        }
      }
    }, SchedulerStateChangedEvent.class);
  }

  /**
   * Checa se o agendador de comandos est bloqueado.
   * 
   * @return <tt>true</tt> se o agendador de comandos est bloqueado.
   */
  public boolean isSchedulerBlocked() {
    return blocked;
  }

  /**
   * Adiciona um ouvinte que ir receber e tratar eventos de mudana de estado -
   * (des)bloqueado - do agendador de comandos.
   * 
   * @param listener ouvinte de eventos de mudana de estado do agendador de
   *        comandos.
   */
  public void addStateChangedListener(EventListener<StateChangedEvent> listener) {
    eventManager.addEventListener(listener, StateChangedEvent.class);
  }

  /**
   * Alterna o estado - (des)bloqueado - do agendador de comandos. {@inheritDoc}
   */
  @Override
  protected void handleActionPerformed(ActionEvent ae) throws Exception {
    final boolean newState = !blocked;
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        ClientRemoteLocator.schedulerService.setBlocked(newState);
      }
    };
    String title =
      getApplication().getString("task.scheduler.state.update.title");
    String msg =
      getApplication().getString("task.scheduler.state.update.message");
    task.execute(getApplication().getApplicationFrame(), title, msg);
  }

  /**
   * Altera o valor, persistido em memria, do estado do agendador, e lana o
   * evento de alterao de estado.
   * 
   * @param blocked <tt>true</tt> indica que o agendador de comandos est
   *        bloqueado.
   */
  private void setSchedulerBlocked(boolean blocked) {
    this.blocked = blocked;
    this.eventManager.fireEvent(new StateChangedEvent(blocked));
  }

  /**
   * Atualiza o status da fila: bloqueada ou desbloqueada.
   */
  private void updateState() {
    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        setResult(ClientRemoteLocator.schedulerService.isBlocked());
      }
    };
    String title =
      getApplication().getString("task.scheduler.state.retrieve.title");
    String msg =
      getApplication().getString("task.scheduler.state.retrieve.message");
    if (task.execute(getApplication().getApplicationFrame(), title, msg)) {
      setSchedulerBlocked(task.getResult());
    }
  }

  /**
   * Evento lanado quando h mudana no estado do agendador de comandos.
   * 
   * @author Tecgraf / PUC-Rio
   */
  public class StateChangedEvent implements IEvent {
    /**
     * <tt>true</tt> indica que o agendador bloqueado e <tt>false</tt> indica
     * que foi desbloqueado no momento em que o evendo foi lanado.
     */
    private final boolean blocked;

    /**
     * Construtor.
     * 
     * @param blocked <tt>true</tt> indica que o agendador bloqueado e
     *        <tt>false</tt> indica que foi desbloqueado.
     */
    private StateChangedEvent(boolean blocked) {
      this.blocked = blocked;
    }

    /**
     * Verifica o estado do agendador no momento em que este evento foi criado.
     * 
     * @return <tt>true</tt> indica que o agendador bloqueado e <tt>false</tt>
     *         indica que foi desbloqueado.
     */
    public boolean isSchedulerBlocked() {
      return blocked;
    }
  }
}
