package csbase.client.applications.filetransfer;

import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JLabel;
import javax.swing.JScrollPane;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.AbstractSimpleApplicationPanel;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.kernel.ClientException;
import csbase.client.project.ProjectTree;
import csbase.client.project.ProjectTreeFactory;
import csbase.logic.CommonClientProject;
import csbase.logic.UserProjectInfo;
import csbase.logic.filters.ProjectFileCompositeFilter;
import csbase.logic.filters.ProjectFileCompositeOrFilter;
import csbase.logic.filters.ProjectFileDirectoryFilter;
import csbase.logic.filters.ProjectFileTypeFilter;

/**
 * Classe que desenha um painel com uma combo com os projetos disponveis para
 * aquele usurio e mostra uma rvore com os respectivos arquivos do projeto
 * selecionado.
 * 
 * @author Tecgraf
 */
public class ProjectFileSelectorPanel extends
  AbstractSimpleApplicationPanel<Application> {

  /** A rvore de diretrios do projeto */
  protected ProjectTree projectTree;

  /** Combo de Projeto */
  protected ProjectComboBox projectCombo;

  /** Projeto */
  private CommonClientProject project;

  /** Tarefa responsvel pela leitura dos projetos */
  private ReadProjectTask readProjecTask;

  /** Rtulo do combo. */
  private String comboLabel;

  /** Scrollpane para rvore */
  protected JScrollPane projectPanel;

  /**
   * Construtor.
   * 
   * @param application aplicao.
   * @param selectOpenProject indica que o projeto aberto, se existir, deve ser
   *        selecionado na combobox.
   * @param comboLabel rtulo da combobox.
   * @param singleSelection define se a seleo  mltipla ou no.
   * @param fileTypes filtros de tipo de arquivo. apenas esses tipos e os
   *        diretrios aparecero na rvore. Se null, aceita todos os tipos.
   */
  public ProjectFileSelectorPanel(final Application application,
    boolean selectOpenProject, String comboLabel, boolean singleSelection,
    String... fileTypes) {
    super(application);
    setLayout(new GridBagLayout());

    this.comboLabel = comboLabel;

    // Criando combo de Projeto
    projectCombo = new ProjectComboBox(application);
    projectCombo.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ae) {
        project =
          openProjectAction(application, project, projectCombo, projectTree);
      }
    });

    try {
      // Criando a rvore de projeto   
      projectTree = createProjectTree(project, singleSelection, fileTypes);
      projectPanel = new JScrollPane(projectTree.getTree());

    }
    catch (Exception e) {
      e.printStackTrace();
    }

    buildPanel();

    //  O Projeto Origem  o mesmo do projeto que est aberto 
    // (ou null, se no houver projeto aberto)
    if (selectOpenProject) {
      CommonClientProject currentProject =
        DesktopFrame.getInstance().getProject();
      /* Se houver projeto aberto, ficar selecionado na combo */
      if (currentProject != null) {
        projectCombo.setSelectedItem(new UserProjectInfo(
          currentProject.getId(), currentProject.getName(), currentProject
            .getUserId()));
      }
    }
  }

  /**
   * Retorna a rvore do projeto
   * 
   * @return projectTree
   */
  public ProjectTree getProjectTree() {
    return projectTree;
  }

  /**
   * Retorna a combo do projeto
   * 
   * @return projectCombo
   */
  public ProjectComboBox getProjectCombo() {
    return projectCombo;
  }

  /**
   * Retorna o projeto
   * 
   * @return project
   */
  public CommonClientProject getProject() {
    return project;
  }

  /** Desinstala/remove observadores da combo e do projeto. */
  public void removeObservers() {
    try {
      // Desinstalando o observador do projeto
      if (project != null) {
        project.uninstallObserver();
      }
      // Removendo a combo de ProjectAdminObserver
      projectCombo.deleteObserver();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void buildPanel() {
    add(new JLabel(comboLabel), new GBC(0, 0).northwest());
    add(projectCombo, new GBC(1, 0).northwest().insets(0, 5, 10, 0));

    projectPanel.setPreferredSize(new Dimension(200, 465));

    add(projectPanel, new GBC(0, 1).both().northwest().insets(0, 0, 0, 0)
      .width(2));
  }

  /**
   * Cria a rvore de projeto de acordo com tipos de arquivos.
   * 
   * @param project projeto.
   * @param singleSelection define se a seleo  mltipla ou no.
   * @param fileTypes tipos de arquivos a serem filtrados. Se null, mostra
   *        todos.
   * 
   * @return rvore de projetos.
   * 
   * @throws ClientException em caso de erro.
   */
  protected ProjectTree createProjectTree(CommonClientProject project,
    boolean singleSelection, String... fileTypes) throws ClientException {
    ProjectTree projectTree;
    ProjectFileCompositeFilter filter = null;
    // Cria um filtro com todos os diretrios ou arquivos com uma determinada extenso
    if (fileTypes.length != 0) {
      ProjectFileTypeFilter fileFilter = new ProjectFileTypeFilter(fileTypes);
      ProjectFileDirectoryFilter dirFilter =
        ProjectFileDirectoryFilter.getInstance();
      filter = new ProjectFileCompositeOrFilter();
      filter.addChild(fileFilter);
      filter.addChild(dirFilter);
    }

    ApplicationFrame frame = getApplication().getApplicationFrame();

    projectTree =
      ProjectTreeFactory.getInstance().makeProjectTree(frame, frame.getTitle(),
        project, filter);

    projectTree.setSingleSelectionMode(singleSelection);
    projectTree.enableRenameAction(false);
    projectTree.enableDeleteAction(false);
    /* Habilitando menu popup */
    projectTree.addMouseListener();

    return projectTree;
  }

  /**
   * Abre o projeto selecionado na combo e atualiza a rvore.
   * 
   * @param application referncia para a aplicao
   * @param project Projeto atualmente aberto na rvore.
   * @param projectCombo Combo onde foi selecionado o projeto que ser aberto.
   * @param projectTree rvore de diretrios que exibe o contedo do projeto.
   * 
   * @return o projeto aberto ou null em caso de erro.
   */
  private CommonClientProject openProjectAction(Application application,
    CommonClientProject project, ProjectComboBox projectCombo,
    ProjectTree projectTree) {

    if (projectCombo.getSelectedItem() == null) {
      return null;
    }
    // Instancia a tarefa uma nica vez
    if (readProjecTask == null) {
      readProjecTask = new ReadProjectTask(projectTree, projectCombo, project);
    }

    if (readProjecTask.execute(application.getApplicationFrame(), application
      .getName(), getString("ProjectFileSelectorPanel.wait.message"))) {
      return readProjecTask.getResult();
    }
    return null;
  }
}