/*
 * $Id$
 */
package csbase.client.ias;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JPanel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.selector.ContainerSelection;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.remote.srvproxies.RoleProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.Role;
import csbase.logic.User;
import csbase.logic.UserInfo;

/**
 * A classe <code>UserRoleDefinitionDialog</code>  responsvel pela seleo de
 * perfis para o usurio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class UserRoleDefinitionDialog {
  /**
   * Referncia para o objeto que concentra os dados do usurio para facilitar a
   * persistncia
   */
  private UserInfo userInfo;

  /**
   * Janela da classe "pai"
   */
  private DesktopComponentDialog ownerDialog;

  /**
   * Janela desta classe, responsvel pela seleo de perfis
   */
  private DesktopComponentDialog mainDialog;

  /**
   * Ttulo da janela
   */
  private String dialogTitle;

  /**
   * Componente que permite a seleo de perfis do usurio
   */
  private ContainerSelection<Role> rolesSelection;

  /**
   * Obtm os perfis do usurio.
   * 
   * @return Um array contendo os perfis do usurio, ou um array vazio, caso
   *         este no possua nenhum perfil.
   */
  private Vector<Role> getUserRoles() {
    Object[] roleIds = (Object[]) userInfo.getAttribute(User.ROLE_IDS);
    if (roleIds == null) {
      return new Vector<Role>();
    }
    Vector<Role> roles = new Vector<Role>(roleIds.length);
    try {
      for (int i = 0; i < roleIds.length; i++) {
        roles.add(i, Role.getRole(roleIds[i]));
      }
    }
    catch (Exception e) {
      StandardErrorDialogs.showErrorDialog(mainDialog, dialogTitle,
        LNG.get("IAS_USER_ROLES_RETRIEVAL_ERROR"));
    }
    return roles;
  }

  /**
   * Cria o dilogo de seleo de perfis.
   */
  public void showDialog() {
    this.dialogTitle = LNG.get("IAS_USER_ROLES_TITLE");

    // Cria o dilogo
    mainDialog = new DesktopComponentDialog(ownerDialog, dialogTitle);

    // Pega os perfis j selecionados (lado direito do componente de seleo) e
    // subtrai dos perfis disponveis para seleo (lado esquerdo do componente
    // de seleo)
    Vector<Role> roles =
      RoleProxy.getAllRoles(mainDialog, dialogTitle,
        LNG.get("IAS_WAITING_ALL_ROLES"));
    if (roles == null) {
      StandardErrorDialogs.showErrorDialog(mainDialog, dialogTitle,
        LNG.get("IAS_USER_ROLES_RETRIEVAL_ERROR"));
      return;
    }

    Container cp = mainDialog.getContentPane();
    JPanel selectionPanel = createSelectionPanel(roles, getUserRoles());
    cp.add(selectionPanel, BorderLayout.CENTER);
    JButton confirmButton = new JButton(LNG.get("IAS_CONFIRM"));
    confirmButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        updateUserRoles();
        mainDialog.close();
      }
    });

    JButton cancelButton = new JButton(LNG.get("IAS_CANCEL"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        mainDialog.close();
      }
    });
    ClientUtilities.adjustEqualSizes(confirmButton, cancelButton);
    JPanel buttonsPanel = new JPanel();
    buttonsPanel.add(confirmButton);
    buttonsPanel.add(cancelButton);
    cp.add(buttonsPanel, BorderLayout.SOUTH);
    mainDialog.pack();
    mainDialog.center(ownerDialog);
    mainDialog.setVisible(true);
  }

  /**
   * Cria o painel para seleo de perfis.
   * 
   * @param selectableRoles perfis disponveis para seleo
   * @param userRoles perfis do usurio
   * 
   * @return o painel criado
   */
  private JPanel createSelectionPanel(Vector<Role> selectableRoles,
    Vector<Role> userRoles) {
    ObjectTableProvider selectableFormat = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { LNG.get("IAS_AVAILABLE_ROLES") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { String.class };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object arg0) {
        Role role = (Role) arg0;
        return (role == null) ? null : new String[] { role.getName() };
      }
    };
    ObjectTableProvider selectedFormat = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { LNG.get("IAS_SELECTED_ROLES") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { String.class };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object arg0) {
        Role role = (Role) arg0;
        return (role == null) ? null : new String[] { role.getName() };
      }
    };
    rolesSelection =
      new ContainerSelection<Role>(selectableFormat, selectedFormat, true,
        false);
    rolesSelection.loadItems(selectableRoles, userRoles);
    rolesSelection.adjustTableColumns();
    return rolesSelection.getPanel();
  }

  /**
   * Atualiza o objeto que concentra as informaes do usurio com os perfis
   * selecionados.
   */
  void updateUserRoles() {
    if (rolesSelection == null) {
      userInfo.setAttribute(User.ROLE_IDS, new Object[0]);
      return;
    }
    List<Role> roles = rolesSelection.getSelectedItems();
    Object[] roleIds = new Object[roles.size()];
    for (int i = 0; i < roles.size(); i++) {
      roleIds[i] = roles.get(i).getId();
    }
    userInfo.setAttribute(User.ROLE_IDS, roleIds);
  }

  /**
   * Limpa a seleo atual de perfis.
   */
  public void clearSelected() {
    if (rolesSelection != null) {
      rolesSelection.clearSelectedItems();
    }
    updateUserRoles();
  }

  /**
   * Cria o dilogo de seleo de perfis de um usurio.
   * 
   * @param ownerDialog janela que criou este dilogo.
   * @param userInfo objeto que centraliza dados de um usurio. Usado para
   *        transportar a seleo de perfis realizada no dilogo.
   * @param user usurio sendo editado, permite comparar a seleo atual de
   *        perfis com a seleo antiga.
   * @param isNew indica se o dilogo est sendo chamado durante uma incluso de
   *        usurio (true) ou alterao (false).
   */
  public UserRoleDefinitionDialog(DesktopComponentDialog ownerDialog,
    UserInfo userInfo, User user, boolean isNew) {
    // #TODO Rever a instanciao da classe para evitar a necessidade dos
    // parmetros isNew e user.
    this.ownerDialog = ownerDialog;
    this.userInfo = userInfo;
    if (user == null && !isNew) {
      throw new IllegalArgumentException("user == null");
    }
    if (!isNew) {
      this.userInfo.setAttribute(User.ROLE_IDS,
        user.getAttribute(User.ROLE_IDS));
    }
  }

  /**
   * Construtor alternativo para novo usurio (sem perfis).
   * 
   * @param ownerDialog janela que criou este dilogo.
   * @param userInfo objeto que centraliza dados de um usurio. Usado para
   *        transportar a seleo de perfis realizada no dilogo.
   * @param isNew indica se o dilogo est sendo chamado durante uma incluso de
   *        usurio (true) ou alterao (false).
   */
  public UserRoleDefinitionDialog(DesktopComponentDialog ownerDialog,
    UserInfo userInfo, boolean isNew) {
    this(ownerDialog, userInfo, null, isNew);
  }
}
