package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.awt.Window;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JTree;
import javax.swing.tree.TreePath;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.algorithmsmanager.DataPropertiesListener;
import csbase.client.applications.algorithmsmanager.actions.VersionCreateAction;
import csbase.client.applications.algorithmsmanager.actions.VersionDuplicateAction;
import csbase.client.applications.algorithmsmanager.actions.VersionExportAction;
import csbase.client.applications.algorithmsmanager.actions.VersionImportAction;
import csbase.client.applications.algorithmsmanager.actions.VersionRemoveAction;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.client.applications.algorithmsmanager.versiontree.VersionNode;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTreeNode;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionInfo;

/**
 * Painel com as informaes das categorias do algoritmo selecionado para
 * edio.
 * 
 */
public class AlgorithmVersionInfoPanel extends CommonInfoEditPanel implements
  DataPropertiesListener {
  /** Painel com as informaes de uma verso do algoritmo */
  private VersionPanel versionEditPanel;

  /** rvore de verses do algoritmo */
  private VersionTreeView versionTreeView;

  /** Painel com a rvore de verses */
  private JPanel versionTreePanel;

  /** Painel principal */
  private JPanel mainPanel;

  /** Painel com as aes a serem realizadas sobre a verso selecionada */
  private JPanel actionsPanel;

  /** Painel com as informaes estticas do algoritmo */
  private AlgorithmStaticInfoPanel algorithmHeaderPanel;

  /** Ao de importao de uma verso */
  private VersionImportAction importVersionAction;

  /** Ao de criao de uma verso */
  private VersionCreateAction createVersionAction;

  /** Ao de criao de duplica uma verso */
  private VersionDuplicateAction duplicateVersionAction;

  /** Ao de remoo de uma verso */
  private VersionRemoveAction removeVersionAction;

  /** Ao de exportao de uma verso */
  private VersionExportAction exportVersionAction;

  /** Indica que est sendo feita uma edio */
  private boolean isEditing;

  /**
   * Constri o painel com as informaes de categorias do algoritmo.
   * 
   * @param algoPane painel com os dados de atualizao de um algoritmo
   */
  public AlgorithmVersionInfoPanel(AlgorithmEditDataPanel algoPane) {
    super(algoPane);
  }

  /**
   * Obtm o painel global de edio de dados de algoritmo.
   * 
   * @return o painel global de edio de dados de algoritmo
   */
  private AlgorithmEditDataPanel getAlgorithmEditDataPanel() {
    return (AlgorithmEditDataPanel) getEditPanel();
  }

  /**
   * Obtm o algoritmo selecionado para edio.
   * 
   * @return o algoritmo selecionado para edio
   */
  public AlgorithmListItem getSelectedAlgorithmItem() {
    return getAlgorithmEditDataPanel().getSelectedAlgorithm();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initializeData() {
    if (getSelectedAlgorithmItem() == null) {
      return;
    }
    if (!isEditing) {
      algorithmHeaderPanel.setSelectedAlgorithm(getSelectedAlgorithmItem());
      updateVersionTreePanel();
      versionEditPanel.initialize(getSelectedAlgorithmItem(), null);
      versionTreeView.setCurrentNode();
    }
    changeOperationsState(isEditing);
  }

  /**
   * Altera o painel com a rvore da verso para o algoritmo selecionado.
   * 
   */
  private void updateVersionTreePanel() {
    mainPanel.remove(versionTreePanel);
    mainPanel.remove(actionsPanel);
    versionTreePanel = getVersionTreePanel();
    actionsPanel = getVersionButtonPanel();
    mainPanel.add(versionTreePanel, new GBC(0, 1).both().west().insets(0, 0, 0,
      0));
    mainPanel.add(actionsPanel, new GBC(1, 1).none().west().insets(0, 0, 0, 0));
    if (getSelectedAlgorithmItem() != null) {
      getVersionTreeView().initializeVersionTree(
        getSelectedAlgorithmItem().getItem());
    }
    mainPanel.revalidate();
    mainPanel.repaint();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JPanel buildMainInfoPanel() {
    if (mainPanel == null) {
      mainPanel = new JPanel(new GridBagLayout());

      algorithmHeaderPanel = new AlgorithmStaticInfoPanel(getApplication());
      versionTreePanel = getVersionTreePanel();
      actionsPanel = getVersionButtonPanel();
      versionEditPanel = getVersionInfoPanel();

      mainPanel.add(algorithmHeaderPanel, new GBC(0, 0).horizontal().gridwidth(
        2).west().insets(0, 0, 0, 0));
      mainPanel.add(versionTreePanel, new GBC(0, 1).both().west().insets(0, 0,
        0, 0));
      mainPanel.add(actionsPanel, new GBC(1, 1).none().west()
        .insets(0, 0, 0, 0));
      mainPanel.add(versionEditPanel, new GBC(0, 2).horizontal().gridwidth(2)
        .west().insets(0, 0, 0, 0));
    }
    return mainPanel;
  }

  /**
   * Cria o painel com os botes referentes s aes que podem ser realizadas
   * sobre verses de algoritmos.
   * 
   * @return o painel criado
   */
  private JPanel getVersionButtonPanel() {
    if (getSelectedAlgorithmItem() == null) {
      return new JPanel();
    }
    actionsPanel = new JPanel(new GridBagLayout());

    // Cria as aes
    createVersionAction = new VersionCreateAction(this, null);
    duplicateVersionAction = new VersionDuplicateAction(this, null);
    removeVersionAction = new VersionRemoveAction(this, null);
    importVersionAction = new VersionImportAction(this, null);
    exportVersionAction = new VersionExportAction(this, null);

    JButton addButton = new JButton(createVersionAction);
    JButton duplicateButton = new JButton(duplicateVersionAction);
    JButton removeButton = new JButton(removeVersionAction);
    JButton importButton = new JButton(importVersionAction);
    JButton exportButton = new JButton(exportVersionAction);

    actionsPanel.add(addButton, new GBC(0, 0).horizontal().west().insets(5, 15,
      5, 15));
    actionsPanel.add(importButton, new GBC(0, 1).horizontal().west().insets(5,
      15, 5, 15));
    actionsPanel.add(exportButton, new GBC(0, 2).horizontal().west().insets(5,
      15, 5, 15));
    actionsPanel.add(duplicateButton, new GBC(0, 3).horizontal().west().insets(
      5, 15, 5, 15));
    actionsPanel.add(removeButton, new GBC(0, 4).horizontal().west().insets(5,
      15, 5, 15));
    ClientUtilities.adjustEqualSizes(addButton, duplicateButton, removeButton,
      importButton);
    return actionsPanel;
  }

  /**
   * Cria o painel com as informaes da verso.
   * 
   * @return o painel com as informaes da verso
   */
  private VersionPanel getVersionInfoPanel() {
    if (versionEditPanel == null) {
      versionEditPanel =
        new VersionPanel(getApplication(), getSelectedAlgorithmItem());
      versionEditPanel.addVersionListener(this);
      versionEditPanel.setEnabled(false);
      versionEditPanel.enableEdition(true);
    }
    return versionEditPanel;
  }

  /**
   * Cria o painel com filtro da rvore de categorias.
   * 
   * @return o painel com filtro criado
   */
  private JPanel getVersionTreePanel() {
    if (getSelectedAlgorithmItem() == null) {
      return new JPanel();
    }
    versionTreePanel = getVersionTreeView().getVersionTreePanel();
    return versionTreePanel;
  }

  /**
   * Obtm a rvore de verses do algoritmo.
   * 
   * @return a rvore de verses do algoritmo
   */
  private VersionTreeView getVersionTreeView() {
    if (versionTreeView == null && getSelectedAlgorithmItem() != null) {
      versionTreeView =
        new VersionTreeView(this, getSelectedAlgorithmItem().getItem());
    }
    return versionTreeView;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void apply() {
    isEditing = false;
    if (!saveInfo()) {
      return;
    }
    changeOperationsState(false);
  }

  /**
   * Salva as informaes de verso.
   * 
   * @return retorna true, se salvou com sucesso, caso contrrio, retorna false
   */
  private boolean saveInfo() {
    if (versionEditPanel.validateProperties()) {
      updateVersion();
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void cancel() {
    if (confirmCancelling()) {
      initializeData();
      changeOperationsState(false);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setDataChanged() {
    isEditing = false;
    changeOperationsState(isEditing);
  }

  /**
   * Confirma com o usurio se a operao de cancelamento deve ser efetivada, j
   * que perder todas as alteraes feitas sobre o dado.
   * 
   * @return retorna true, se a alterao do dado deve ser cancelada, caso
   *         contrrio, retorna false
   */
  private boolean confirmCancelling() {
    int confirm =
      getApplication().showOptionDialog(
        getString("AlgorithmVersionInfoPanel.msg.cancel.confirm"),
        new String[] {
            getString("AlgorithmVersionInfoPanel.msg.cancel.confirm.yes"),
            getString("AlgorithmVersionInfoPanel.msg.cancel.confirm.no") });
    if (confirm == 0) {
      isEditing = false;
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean wasModified() {
    if (versionEditPanel.wasModified()) {
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void propertiesChanged(boolean wasModified) {
    isEditing = wasModified;
    changeOperationsState(wasModified);
  }

  /**
   * Atualiza a verso que foi alterada no servidor.
   */
  private void updateVersion() {
    if (getSelectedAlgorithmItem() == null) {
      return;
    }
    AlgorithmInfo algoInfo = getSelectedAlgorithmItem().getItem();
    String name = algoInfo.getName();
    AlgorithmVersionInfo versionInfo = getSelectedVersion();
    algoInfo =
      AlgorithmManagementProxy.changeVersionProperties(name, versionInfo
        .getId(), versionEditPanel.getProperties(), getWindow());
    String dialogTitle =
      getString("AlgorithmVersionInfoPanel.title.algo_properties");
    if (algoInfo == null) {
      String title =
        MessageFormat.format(
          getString("AlgorithmVersionInfoPanel.title.error"),
          new Object[] { dialogTitle });
      StandardErrorDialogs.showErrorDialog(getWindow(), title, MessageFormat
        .format(getString("AlgorithmVersionInfoPanel.error.update"),
          new Object[] { name }));
      return;
    }
  }

  /**
   * Obtm a verso selecionada na rvore de verses.
   * 
   * @return a verso selecionada
   */
  public AlgorithmVersionInfo getSelectedVersion() {
    List<AlgorithmVersionInfo> versions = getSelectedVersions();
    if (versions.isEmpty()) {
      return null;
    }
    return versions.get(0);
  }

  /**
   * Retorna a lista de verses selecionadas na rvore de verses.
   * 
   * @return lista com as verses selecionadas.
   */
  public List<AlgorithmVersionInfo> getSelectedVersions() {
    List<AlgorithmVersionInfo> versions = new ArrayList<>();
    TreePath[] selectionPaths =
      versionTreeView.getVersionTree().getSelectionPaths();
    if (selectionPaths != null) {
      for (TreePath path : selectionPaths) {
        Object node = path.getLastPathComponent();
        if (node instanceof VersionNode) {
          VersionNode versionNode = (VersionNode) node;
          AlgorithmVersionInfo info =
            (AlgorithmVersionInfo) versionNode.getUserObject();
          versions.add(info);
        }
      }
    }
    return versions;
  }

  /**
   * Obtm a janela da aplicao.
   * 
   * @return a janela da aplicao
   */
  private Window getWindow() {
    return getAlgorithmEditDataPanel().getApplication().getApplicationFrame();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getTitle() {
    return getString("AlgorithmVersionInfoPanel.tab.versions");
  }

  /**
   * Verifica e modifica o estado das aes sobre uma ou mais categorias de
   * acordo com as selees feitas pelo usurio na rvore de categorias.
   * 
   */
  public void verifyAndChangeButtonsState() {
    // ao de Criar verso
    enableAction(createVersionAction, true);

    // ao de Duplicar verso
    enableAction(duplicateVersionAction, getSelectedVersions().size() == 1);

    // ao de Remover verso
    enableAction(removeVersionAction, !getSelectedVersions().isEmpty());

    // ao de Importar pacote de verso de algoritmo
    enableAction(importVersionAction, true);

    // ao de Exportar pacote de verso de algoritmo
    enableAction(exportVersionAction, getSelectedVersions().size() == 1);
  }

  /**
   * Habilita ou desabilita as aes dos botes que representam as operaes
   * sobre os dados selecionados.
   * 
   * @param opAction ao da operao que vai modificar seu estado
   * @param state se true, habilita as aes, caso contrrio, desabilita as
   *        aes
   */
  protected void enableAction(Action opAction, boolean state) {
    if (opAction != null) {
      opAction.setEnabled(state);
    }
  }

  /**
   * Obtm o n selecionado na rvore de verses.
   * 
   * @return o n selecionado na rvore de dados
   */
  public int getSelectedRow() {
    JTree dataTree = versionTreeView.getVersionTree();
    int[] selectionRows = dataTree.getSelectionRows();
    if (selectionRows != null && selectionRows.length > 0) {
      return selectionRows[0];
    }
    return 0;
  }

  /**
   * Habilita ou no o painel de edio de uma verso.
   * 
   * @param state se true, habilita o painel de edio de uma verso, caso
   *        contrrio, desabilita o painel de edio de uma verso
   */
  public void enableVersionEditPanel(boolean state) {
    versionEditPanel.enableEdition(state);
  }

  /**
   * Confirma se pode mudar a seleo para uma nova verso.
   * 
   * @return retorna true, se o usurio confirmou a mudana para a nova verso,
   *         caso contrrio, retorna false
   */
  public boolean confirmSelectionChanged() {
    return !(isEditing && !confirmCancelling());
  }

  /**
   * Inicializa a edio para o n de verso especificado.
   * 
   * @param versionNode novo n de verso a ser editado
   */
  public void initVersionNodeEdition(VersionTreeNode versionNode) {
    isEditing = false;
    versionEditPanel.initialize(getSelectedAlgorithmItem(), versionNode);
    verifyAndChangeButtonsState();
  }

  /**
   * Realiza uma ao quando um algoritmo  alterado na aplicao Gerenciador de
   * Algoritmos.
   * 
   * @param algoInfo informaes do algoritmo alterado
   */
  public void handleAlgorithmUpdated(AlgorithmInfo algoInfo) {
    if (algoInfo != null) {
      getVersionTreeView().handleAlgorithmUpdated(algoInfo);
    }
  }
}
