package csbase.client.applications.projectsmanager.dialogs;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;

import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.dialogs.core.ProjectsManagerDialog;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.panels.ProjectSharingInfoPanel;
import csbase.client.applications.projectsmanager.proxy.AddPersonalCommentTask;

/**
 * Dilogo de incluso de comentrios no histrico de projetos.
 * 
 * @author jnlopes
 */
public class ProjectCommentsDialog extends ProjectsManagerDialog {

  /**
   * Lista de projetos selecionados.
   */
  private List<ProjectsManagerData> projectsManagerData;

  /**
   * rea de texto onde o comentrio deve ser redigido.
   */
  private final JTextArea commentArea = new JTextArea();

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao.
   * @param projectsManagerData Lista de projetos selecionados.
   */
  public ProjectCommentsDialog(final ProjectsManager projectsManager,
    final List<ProjectsManagerData> projectsManagerData) {
    super(projectsManager);
    this.projectsManagerData = projectsManagerData;
    buildPanel();
  }

  /**
   * Monta o painel com os botes.
   * 
   * @return o painel.
   */
  private JPanel buildButtonsPanel() {
    JButton writeButton = new JButton();
    JButton cancelButton = new JButton();
    final ProjectCommentsDialog phd = this;
    writeButton.setAction(new AbstractAction() {
      /**
       * {@inheritDoc}
       */
      @Override
      public void actionPerformed(ActionEvent e) {
        ProjectsManager projectsManager = getProjectsManager();
        AddPersonalCommentTask aphbt =
          new AddPersonalCommentTask(projectsManager, projectsManagerData,
            commentArea.getText());
        ApplicationFrame appFrame = projectsManager.getApplicationFrame();
        String taskTitle = getString("ProjectCommentsDialog.running.title");
        String taskMsg = getString("ProjectCommentsDialog.running.message");
        aphbt.execute(appFrame, taskTitle, taskMsg);
        if (aphbt.wasCancelled()) {
          String err = getString("ProjectCommentsDialog.cancelled.message");
          aphbt.showError(err);
          return;
        }
        if (aphbt.getStatus() != true) {
          Exception exception = aphbt.getError();
          String err =
            getString("ProjectCommentsDialog.comment.inclusion.error.message");
          String msg = err + "\n" + exception.getMessage();
          StandardDialogs.showErrorDialog(appFrame, "", msg);
        }
        /*
         * Executa a tarefa e obtm seu retorno. O retorno  uma lista dos
         * projetos nos quais, por algum problema, no foi possvel incluir o
         * comentrio.
         */
        List<ProjectsManagerData> ret = aphbt.getResult();
        // Verifica se houve falha para algum projeto. 
        if (ret != null && ret.size() > 0) {
          StringBuilder text = new StringBuilder();
          for (ProjectsManagerData prj : ret) {
            text.append(" - " + prj.getProjectName() + "\n");
          }
          String err =
            getString("ProjectCommentsDialog.failure.inserting.some.comment.message");
          String msg = err + "\n" + text.toString();
          StandardDialogs.showWarningDialog(appFrame, "", msg);
        }
        phd.dispose();
      }
    });
    writeButton.setText(getString("ProjectCommentsDialog.insert.text"));
    cancelButton.setAction(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        phd.dispose();
      }
    });
    cancelButton.setText(getString("ProjectCommentsDialog.cancel.text"));
    JComponent[] buttons = { writeButton, cancelButton };
    GUIUtils.matchPreferredSizes(buttons);
    JPanel buttonsPanel = new JPanel();
    buttonsPanel.add(writeButton);
    buttonsPanel.add(cancelButton);
    return buttonsPanel;
  }

  /**
   * Monta o componente de insero do texto.
   * 
   * @return o componente
   */
  private JScrollPane buildTextPane() {
    commentArea.setRows(5);
    commentArea.setColumns(40);
    commentArea.setLineWrap(true);

    final JScrollPane pane = new JScrollPane(commentArea);
    BorderUtil.setTitledBorder(pane,
      getString("ProjectCommentsDialog.comment.area.title"));
    return pane;
  }

  /**
   * Monta o componente dos projetos.
   * 
   * @return o componente
   */
  private ProjectSharingInfoPanel buildProjectPanel() {
    final ProjectsManager projectsManager = getProjectsManager();
    final ProjectSharingInfoPanel projectsPanel =
      new ProjectSharingInfoPanel(projectsManager, projectsManagerData, false);
    BorderUtil.setTitledBorder(projectsPanel,
      getString("ProjectCommentsDialog.projects.title"));
    return projectsPanel;
  }

  /**
   * Monta o painel.
   */
  private void buildPanel() {
    final JPanel buttonsPanel = buildButtonsPanel();
    final JScrollPane pane = buildTextPane();
    final ProjectSharingInfoPanel projectsPanel = buildProjectPanel();

    setLayout(new GridBagLayout());
    add(projectsPanel, new GBC(0, 0).horizontal());
    add(pane, new GBC(0, 1).both());
    add(buttonsPanel, new GBC(0, 2).horizontal());
  }
}
