package csbase.client.facilities.configurabletable.UI;

import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import tecgraf.javautils.gui.GBC;
import csbase.client.facilities.configurabletable.table.ConfigurableTable;

/**
 * Painel especializado em exibir tabelas configurveis.
 * 
 * @see ConfigurableTable UIFactory
 * 
 * @author Tecgraf
 */
public class Panel4Tables extends JPanel implements UI4Tables {

  /**
   * Identificador do painel.
   */
  private String id;

  /**
   * Orientao das tabelas no painel.
   */
  private ORIENTATION orientation;

  /**
   * Lista de todas as informaes da tabelas que sero exibidas na interface.
   */
  private List<TableInfo> tables;

  /**
   * Construtor default.
   * 
   * @param id - identificador do painel.
   * @param orientation - orientao das tabelas.
   */
  public Panel4Tables(String id, ORIENTATION orientation) {
    super(new GridBagLayout());

    if (id == null) {
      throw new IllegalArgumentException("id no pode ser nulo.");
    }

    if (orientation == null) {
      throw new IllegalArgumentException("orientation no pode ser nulo.");
    }

    this.id = id;
    this.orientation = orientation;
    this.tables = new ArrayList<TableInfo>();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getId() {
    return id;
  }

  /**
   * Adiciona uma tabela no painel.
   * 
   * @param <T> - tipo dos dados exibidos pela tabela.
   * @param table - tabela a ser adicionada no painel.
   * @param label - Rtulo que aparecer acima da tabela.
   * @param dataType - tipo dos dados exibidos pela tabela.
   */
  public <T> void addTable(ConfigurableTable<?> table, String label,
    Class<T> dataType) {

    if (table == null) {
      throw new IllegalArgumentException("table no pode ser nulo.");
    }

    if (label == null) {
      throw new IllegalArgumentException("label no pode ser nulo.");
    }

    if (dataType == null) {
      throw new IllegalArgumentException("dataType no pode ser nulo.");
    }

    tables.add(new TableInfo(table, label, dataType));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void buildUI() {
    removeAll();

    int xTableIndex = 0;
    int yTableIndex = 0;

    int xLabelIndex = 0;
    int yLabelIndex = 0;

    for (TableInfo info : tables) {

      add(new JLabel(info.label), new GBC(xLabelIndex, yLabelIndex).west().top(
        10).left(10).bottom(0));

      yTableIndex = yLabelIndex + 1;

      JScrollPane scroolPane = new JScrollPane(info.table);

      // ajustando o tamanho da tabela
      Dimension dimension = info.table.getPreferredSize();
      info.table.setPreferredSize(null);
      scroolPane.setPreferredSize(dimension);

      add(scroolPane, new GBC(xTableIndex, yTableIndex).both().insets(10, 10,
        10, 10));

      switch (orientation) {
        case HORIZONTAL:
        case Horizontal:
        case horizontal:
          xTableIndex++;
          xLabelIndex++;
          break;

        default:
        case VERTICAL:
        case Vertical:
        case vertical:
          yLabelIndex += 2;
          yTableIndex++;
          break;
      }
    }

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConfigurableTable<?> getTableById(String tableId) {

    for (TableInfo info : tables) {
      if (info.table.getId().equals(tableId)) {
        return info.table;
      }
    }

    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  @SuppressWarnings("unchecked")
  public <T> ConfigurableTable<T> getTableById(String tableId, Class<T> dataType) {

    for (TableInfo info : tables) {
      if (info.table.getId().equals(tableId) && info.clazz.equals(dataType)) {
        return (ConfigurableTable<T>) info.table;
      }
    }

    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ConfigurableTable<?>> getAllTables() {

    List<ConfigurableTable<?>> allTables =
      new ArrayList<ConfigurableTable<?>>();

    for (TableInfo info : tables) {
      allTables.add(info.table);
    }

    return allTables;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  @SuppressWarnings("unchecked")
  public <T> List<ConfigurableTable<T>> getAllTables(Class<T> dataType) {
    List<ConfigurableTable<T>> allTables =
      new ArrayList<ConfigurableTable<T>>();

    for (TableInfo info : tables) {
      if (info.clazz.equals(dataType)) {
        allTables.add((ConfigurableTable<T>) info.table);
      }
    }

    return allTables;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ConfigurableTable<?>> getTablesFromSelectedComponent() {
    return getAllTables();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <T> List<ConfigurableTable<T>> getTablesFromSelectedComponent(
    Class<T> dataType) {
    return getAllTables(dataType);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getTableLabel(String tableId) {
    for (TableInfo info : this.tables) {
      if (tableId != null && tableId.equals(info.table.getId())) {
        return info.label;
      }
    }
    return null;
  }

  /**
   * Classe auxiliar que armazena as informaes de uma tabela necessrias para
   * posiciona-las na interface.
   * 
   * @author Tecgraf
   */
  private class TableInfo {

    /**
     * Referncia para tabela.
     */
    ConfigurableTable<?> table;

    /**
     * Rtulo da tabela.
     */
    String label;

    /**
     * Tipo de dado que a tabela exibe.
     */
    Class<?> clazz;

    /**
     * Construtor padro.
     * 
     * @param table - referncia para tabela.
     * @param label - rtulo da tabela.
     * @param clazz - tipo de dado que a tabela exibe.
     */
    public TableInfo(ConfigurableTable<?> table, String label, Class<?> clazz) {
      this.table = table;
      this.label = label;
      this.clazz = clazz;
    }

  }

}
