package csbase.client.facilities.configurabletable.table;

import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.AbstractAction;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.SortOrder;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableModel;

import csbase.client.facilities.configurabletable.ConfigurableTableFactory;
import csbase.client.facilities.configurabletable.UI.Panel4Tables;
import csbase.client.facilities.configurabletable.UI.TabbedPane4Tables;
import csbase.client.facilities.configurabletable.column.IConfigurableColumn;
import tecgraf.javautils.core.filter.IFilter;
import tecgraf.javautils.gui.table.SortableTable;

/**
 * Esta tabela utiliza o conceito de que cada instncia possui um identificador
 * nico. Este identificador  muito til nos componentes que utilizam essa
 * tabela, por exemplo, {@link TabbedPane4Tables}, {@link Panel4Tables} e
 * {@link ConfigurableTableFactory}.
 * 
 * Essa tabela usa o modelo de tabelas definido pela classe
 * {@link ConfigurableTableModel}. Para adiconarmos/modificarmos linhas desta
 * tabela basta passarmos uma lista de objetos do tipo {@code T}.
 * 
 * Essa tabela tambm permite definir um filtro para suas linhas. Para isso, o
 * filtro deve implementar a interface {@link IFilter}.
 * 
 * Para criar uma tabela configurvel  necessrio uma lista de colunas que
 * implementam a interface {@link IConfigurableColumn}, uma lista de objetos do
 * tipo {@code T} e, opcionalmente, um filtro {@link IFilter}. Veja abaixo:
 * 
 * {@code
 * List<IConfigurableColumn<Person>> columns 
 *             = new ArrayList<IConfigurableColumn<Person>>();
 * ...
 * List<Person> rows = new ArrayList<Person>();
 * rows.add(new Person(...));
 * rows.add(new Person(...));
 * 
 * IFilter<Person> filter = new PersonSingleFilter<Person>();
 * 
 * ConfigurableTable<Person> table 
 *    = new ConfigurableTable<Person>(columns, filter, rows);
 * 
 * }
 * 
 * Para atualizar as linhas dessa tabela basta fazermos:
 * 
 * {@code
 * table.setRows(newRows);
 * table.updateRows();
 * }
 * 
 * Para atualizar as linhas aps setar um filtro basta fazermos:
 * 
 * {@code
 * table.setFilter(newFilter);
 * table.updateRows();
 * }
 * 
 * Esta tabela tambm permite que o usurio defina, atravs de um dialogo da
 * tabela, qual as colunas que ele deseja ver. Para isso, basta clicar com o
 * boto direito sobre a tabela.
 * 
 * @see IConfigurableColumn IFilter ConfigurableTableFactory
 * 
 * @param <T> - tipo do objeto que corresponde as linhas da tabela.
 * @author Tecgraf
 */
public class ConfigurableTable<T> extends SortableTable {

  /**
   * Identificador da tabela.
   */
  private String id;

  /**
   * Modelo de objetos da tabela.
   */
  private ConfigurableTableModel<T> model;

  /**
   * Aes usadas pela interface para atualizar a visibilidade das colunas.
   */
  private Map<String, ColumnAction> columnsActions;

  /**
   * Lista que armazena todos os ouvintes que sero acionados sempre que a
   * visibilidade de uma coluna for alterada.
   */
  private List<ColumnVisibilityListener> columnVisibilityListeners;

  /**
   * Objeto encarregado de obter uma chave (String) a partir de uma linha
   * genrica.  usado quando se deseja manter as mesmas linhas selecionadas
   * aps uma atualizao das linhas.
   */
  private RowToKey<T> rowToKey;

  /**
   * Construtor.
   * 
   * @param id - identificador da tabela.
   * @param columns - colunas da tabela.
   * @param filter - filtro que  utilizado para filtrar as linhas da tabela.
   * @param rows - lista de objetos que definem as linhas da tabela.
   */
  public ConfigurableTable(String id, List<IConfigurableColumn<T>> columns,
    IFilter<T> filter, List<T> rows) {
    this(id, columns, filter);
    if (rows == null) {
      throw new IllegalArgumentException("rows no pode ser nulo.");
    }
    updateRows(rows);
  }

  /**
   * Construtor padro.
   * 
   * @param id - identificador da tabela.
   * @param columns - implementao das colunas da tabela.
   * @param filter - filtro das linhas da tabela.
   */
  public ConfigurableTable(String id, List<IConfigurableColumn<T>> columns,
    IFilter<T> filter) {

    if (id == null) {
      throw new IllegalArgumentException("id no pode ser nulo.");
    }

    if (columns == null) {
      throw new IllegalArgumentException("columns no pode ser nulo.");
    }

    this.id = id;

    List<T> emptyList = new ArrayList<T>();
    model = new ConfigurableTableModel<T>(columns, filter, emptyList);
    super.setModel(model);

    columnVisibilityListeners = new ArrayList<ColumnVisibilityListener>();

    setColumnsComparators();

    createColumnsActions();
    addHeaderAction();
  }

  /**
   * Identificador nico da tabela.
   * 
   * @return identificador da tabela.
   */
  public String getId() {
    return id;
  }

  /**
   * Obtm a lista de objetos que corresponde as linhas da tabela.
   * 
   * @return lista de objetos que corresponde as linhas da tabela.
   */
  public List<T> getRows() {
    return model.getRows();
  }

  /**
   * Obtm o filtro da tabela.
   * 
   * @return filtro.
   */
  public IFilter<T> getFilter() {
    return this.model.getFilter();
  }

  /**
   * Obtem a lista de objetos selecionados.
   * 
   * @return lista de objetos selecionados.
   */
  public List<T> getSelectedObjects() {
    List<T> result = new ArrayList<T>();

    int[] selectedRows = this.getSelectedRows();
    if (selectedRows != null && selectedRows.length > 0) {
      for (int i = 0; i < selectedRows.length; i++) {
        T selectedObject =
          model.getRow(convertRowIndexToModel(selectedRows[i]));
        result.add(selectedObject);
      }
    }

    return result;
  }

  /**
   * Altera o filtro usado na tabela.
   * 
   * @param filter o filtro.
   */
  public void setFilter(IFilter<T> filter) {
    this.model.setFilter(filter);
  }

  /**
   * Atualiza a visualizao das linhas da tabela.
   * 
   * @param newRows - lista com as novas linhas da tabela.
   */
  public void updateRows(List<T> newRows) {
    if (rowToKey != null) {
      Set<String> keys = preUpdateRows();
      model.setRows(newRows);
      model.fireTableDataChanged();
      posUpdateRows(keys);
    }
    else {
      model.setRows(newRows);
      model.fireTableDataChanged();
    }
  }

  /**
   * Atualiza a visualizao das linhas da tabela.
   */
  public void updateRows() {
    if (rowToKey != null) {
      Set<String> keys = preUpdateRows();
      model.fireTableDataChanged();
      posUpdateRows(keys);
    }
    else {
      model.fireTableDataChanged();
    }
  }

  /**
   * Atualiza a visualizao das colunas da tabela.
   */
  public void updateColumns() {
    // Antes de atualizar a visualizao das colunas,
    //  necessrio armazenar o ndice da coluna que ordenou a tabela.
    int columnIndex = getSortedColIndexView();

    if (columnIndex != -1) {
      IConfigurableColumn<T> column = model.getColumn(columnIndex);
      SortOrder order = getCurrentSortOrder();

      model.fireTableStructureChanged();
      setColumnsComparators();

      if (column.isVisible()) {
        sort(columnIndex, order);
      }
    }
    else {
      model.fireTableStructureChanged();
      setColumnsComparators();
    }

    //  necessrio atualizar todas as actions que controlam 
    // a visibilidade das colunas.
    for (ColumnAction action : columnsActions.values()) {
      action.updateAllCheckBoxes();
    }
  }

  /**
   * Lista com os estados de todas as colunas.
   * 
   * @return lista com os estados de todas as colunas.
   */
  public List<ColumnState> getColumnsState() {
    return model.getColumnsState();
  }

  /**
   * Define os estados de todas as colunas.
   * 
   * @param columnsState - lista com os estados de todas as colunas.
   */
  public void setColumnsState(List<ColumnState> columnsState) {
    model.setColumnsState(columnsState);
  }

  /**
   * Cria uma lista de checkboxes que permite o usurio trocar a visibilidade de
   * cada coluna.
   * 
   * @return - lista de checkboxes.
   */
  public List<JCheckBoxMenuItem> createColumnsCheckBoxes() {
    List<JCheckBoxMenuItem> checkboxes = new ArrayList<JCheckBoxMenuItem>();

    for (IConfigurableColumn<T> column : model.getAllColumns()) {
      ColumnAction action = columnsActions.get(column.getId());

      JCheckBoxMenuItem item = new JCheckBoxMenuItem(action);
      item.setText(column.getColumnName());
      item.setSelected(column.isVisible());

      // registra o novo item na ao
      action.registerJCheckBoxMenuItem(item);

      checkboxes.add(item);
    }

    return checkboxes;
  }

  /**
   * Nmero total de colunas (visveis e ocultas).
   * 
   * @return nmero de colunas (visveis e ocultas).
   */
  public int getTotalColumnCount() {
    return model.getAllColumns().size();
  }

  /**
   * Adiciona um ouvinte que  acionado sempre que a visibilidade de alguma
   * coluna for alterada.
   * 
   * @param listener - ouvinte
   */
  public void addColumnVisibilityListener(ColumnVisibilityListener listener) {
    columnVisibilityListeners.add(listener);
  }

  /**
   * Obtm a lista com todos os ouvintes de visibilidade de colunas.
   * 
   * @return lista de ouvintes.
   */
  public List<ColumnVisibilityListener> getColumnVisibilityListeners() {
    return columnVisibilityListeners;
  }

  /**
   * Define o objeto encarregado de obter uma chave (String) de uma linha
   * genrica.
   * 
   * @param rowToKey - objeto encarregado de obter uma chave (String) de uma
   *        linha genrica.
   */
  public void setRowToKey(RowToKey<T> rowToKey) {
    this.rowToKey = rowToKey;
  }

  // --------------------- mtodos sobrescritos -------------------------

  /**
   * Este mtodo foi sobrescrito para no permitir que o usurio modifique o
   * modelo de tabelas.
   */
  @Override
  public void setModel(TableModel tableModel) {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TableCellRenderer getCellRenderer(int row, int column) {
    TableCellRenderer renderer = super.getCellRenderer(row, column);

    int columnModel = convertColumnIndexToModel(column);

    IConfigurableColumn<T> columnImpl = model.getColumn(columnModel);
    TableCellRenderer columnRenderer = columnImpl.createTableCellRenderer();
    if (columnRenderer != null) {
      renderer = columnRenderer;
    }

    if (renderer != null && renderer instanceof DefaultTableCellRenderer) {
      int align = columnImpl.getAlign();
      ((DefaultTableCellRenderer) renderer).setHorizontalAlignment(align);
    }

    return renderer;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TableCellEditor getCellEditor(int row, int column) {
    TableCellEditor cellEditor = super.getCellEditor(row, column);

    int columnModel = convertColumnIndexToModel(column);
    IConfigurableColumn<T> columnImpl = model.getColumn(columnModel);
    TableCellEditor editor = columnImpl.createTableCellEditor();

    if (editor != null) {
      cellEditor = editor;
    }

    return cellEditor;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((id == null) ? 0 : id.hashCode());
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    ConfigurableTable<?> other = (ConfigurableTable<?>) obj;
    if (id == null) {
      if (other.id != null) {
        return false;
      }
    }
    else if (!id.equals(other.id)) {
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return getId();
  }

  // ------------------------ mtodos protegidos -------------------------

  /**
   * Adiciona uma ao no header da tabela.
   */
  protected void addHeaderAction() {

    final List<JCheckBoxMenuItem> checkboxes = this.createColumnsCheckBoxes();

    getTableHeader().addMouseListener(new MouseAdapter() {
      private JPopupMenu popUp;

      @Override
      public void mouseClicked(MouseEvent e) {
        if (MouseEvent.BUTTON3 == e.getButton()) {
          if (popUp == null) {
            popUp = new JPopupMenu();

            for (JCheckBoxMenuItem item : checkboxes) {
              popUp.add(item);
            }
          }

          popUp.show(e.getComponent(), e.getX(), e.getY());
        }
      }
    });

  }

  // ---------------------- mtodos privados ------------------------------

  /**
   * Obtm um conjunto de chaves que identificam as linhas selecionadas da
   * tabela.
   * 
   * @return conjunto de chaves das linhas selecionadas.
   */
  private Set<String> preUpdateRows() {
    Set<String> keys = new HashSet<String>();

    int[] selectedRows = getSelectedRows();

    for (int i = 0; i < selectedRows.length; i++) {
      int modelIndex = convertRowIndexToModel(selectedRows[i]);
      T line = getRows().get(modelIndex);
      keys.add(rowToKey.getKey(line));
    }
    return keys;
  }

  /**
   * Re-seleciona as linhas da tabela aps a atualizao.
   * 
   * @param keys - chaves dos objetos a serem re-selecionados.
   */
  private void posUpdateRows(Set<String> keys) {
    for (int i = 0; i < getRows().size(); i++) {
      T line = getRows().get(i);
      if (keys.contains(rowToKey.getKey(line))) {
        int index = convertRowIndexToView(i);
        addRowSelectionInterval(index, index);
      }
    }
  }

  /**
   * Atribui na tabela, o comparador de cada coluna.
   */
  private void setColumnsComparators() {
    for (int i = 0; i < model.getColumnCount(); i++) {
      Comparator<?> comparator = model.getColumns().get(i).getComparator();
      if (comparator != null) {
        super.setComparator(i, comparator);
      }
    }
  }

  /**
   * Cria uma aes que sero usadas pela interface do usurio para modificar a
   * visibilidade das colunas.
   */
  private void createColumnsActions() {
    this.columnsActions = new HashMap<String, ColumnAction>();

    for (IConfigurableColumn<T> column : model.getAllColumns()) {
      ColumnAction action = new ColumnAction(column, this);
      this.columnsActions.put(column.getId(), action);
    }
  }

  // ----------------- classe privada -------------------------------

  /**
   * Ao encarregada de trocar a visibilidade das colunas de uma tabela.
   * 
   * @author Tecgraf
   */
  private class ColumnAction extends AbstractAction {

    /**
     * Referncia para a coluna.
     */
    private IConfigurableColumn<?> column;

    /**
     * Referncia para a tabela a que pertence a coluna.
     */
    private ConfigurableTable<?> table;

    /**
     * Lista de checkboxes que utilizam essa ao.
     */
    private List<JCheckBoxMenuItem> checkboxes;

    /**
     * Construtor padro.
     * 
     * @param column - coluna.
     * @param table - tabela a que pertence a coluna.
     */
    public ColumnAction(final IConfigurableColumn<?> column,
      final ConfigurableTable<?> table) {
      this.column = column;
      this.table = table;
      this.checkboxes = new ArrayList<JCheckBoxMenuItem>();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      this.column.setVisible(!column.isVisible());
      table.updateColumns();

      // aciona todos os ouvintes de visibilidade de colunas
      for (ColumnVisibilityListener l : table.getColumnVisibilityListeners()) {
        l.visibilityChanged(column);
      }
    }

    /**
     * Cadastra um novo item de checkBox que deve ser atualizado sempre que essa
     * ao for executada.
     * 
     * @param item - item de checkbox.
     */
    public void registerJCheckBoxMenuItem(JCheckBoxMenuItem item) {
      this.checkboxes.add(item);
    }

    /**
     * Mtodo que atualiza todos os checkboxes que utilizam essa action.
     */
    public void updateAllCheckBoxes() {
      for (JCheckBoxMenuItem item : this.checkboxes) {
        item.setSelected(this.column.isVisible());
      }
    }

  }

}