package csbase.client.facilities.configurabletable.table;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.swing.event.TableModelEvent;
import javax.swing.table.AbstractTableModel;

import csbase.client.facilities.configurabletable.column.IConfigurableColumn;
import tecgraf.javautils.core.filter.IFilter;

/**
 * Modelo de tabelas que trabalha com colunas configurveis. <br>
 * As 3 principais caractersticas deste modelo so: <br>
 * 1 - As linhas da tabela so representados por uma lista de objetos. <br>
 * 2 - As colunas so configurveis ({@link IConfigurableColumn}). <br>
 * 3 - Usa um filtro para as linhas da tabela ({@link IFilter}).
 * 
 * Este modelo trabalha apenas com colunas que implementam a interface
 * {@link IConfigurableColumn} e, dessa forma, s exibe as colunas que esto
 * configuradas para serem visveis.
 * 
 * Alm disso, as linhas definidas por esse modelo so sujeitas a um filtro que
 * implementa a interface {@link IFilter}. Dessa forma, esse modelo exibe apenas
 * as linhas filtradas por este filtro, caso o filtro seja nulo, o modelo exibe
 * todas as linhas.
 * 
 * @param <T> - tipo dos objetos que correspondem as linhas da tabela.
 * 
 * @author Tecgraf
 */
public class ConfigurableTableModel<T> extends AbstractTableModel {

  /** Lista de todos os objetos que representam as linhas da tabela */
  private List<T> rows;

  /** Lista de todas as colunas da tabela na ordem em que devem aparecer. */
  private List<IConfigurableColumn<T>> columns;

  /** Filtro das linhas */
  private IFilter<T> filter;

  /* Listas auxiliares */

  /**
   * Lista de objetos, filtrados por <code>filter</code> que representam as
   * linhas da tabela. Essa lista  mantida para no ser necessrio re-calcular
   * as linhas filtradas.
   */
  private List<T> filterableRows;

  /**
   * Lista de colunas visveis. Essa lista  mantida para no ser necessrio
   * re-calcular todas as colunas visveis.
   */
  private List<IConfigurableColumn<T>> visibleColumns;

  /**
   * Cria o modelo.
   * 
   * @param columns - Modelo das colunas da tabela na ordem em que devem
   *        aparecer.
   * @param filter - filtro das linhas.
   * @param rows - lista de objetos-linha.
   * 
   * @throws IllegalArgumentException se a lista de linhas ou o provedor forem
   *         iguais a null, ou se a lista de nomes das colunas retornada pelo
   *         provedor for null
   */
  public ConfigurableTableModel(List<IConfigurableColumn<T>> columns,
    IFilter<T> filter, List<T> rows) {
    if (columns == null) {
      throw new IllegalArgumentException("columns no pode ser nulo.");
    }

    if (columns.size() == 0) {
      throw new IllegalArgumentException(
        " necessrio que haja pelo menos uma coluna.");
    }

    if (rows == null) {
      throw new IllegalArgumentException("rows no pode ser nulo.");
    }

    this.columns = columns;
    this.filter = filter;
    this.rows = rows;

    this.visibleColumns = new ArrayList<IConfigurableColumn<T>>();
    this.filterableRows = new ArrayList<T>();

    updateFilterableRows();
    updateVisibleColums();
  }

  /**
   * Obtm a lista de colunas visveis na ordem em que elas aparecem.<br>
   * 
   * @return list de colunas visveis na ordem em que elas aparecem.
   */
  public List<IConfigurableColumn<T>> getColumns() {
    return this.visibleColumns;
  }

  /**
   * Obtm todas as colunas do modelo.
   * 
   * @return todas as colunas (visveis ou no)
   */
  public List<IConfigurableColumn<T>> getAllColumns() {
    return this.columns;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getRowCount() {
    return filterableRows.size();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnCount() {
    return visibleColumns.size();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getColumnName(int columnIndex) {
    return visibleColumns.get(columnIndex).getColumnName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<?> getColumnClass(int columnIndex) {
    return visibleColumns.get(columnIndex).getColumnClass();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValueAt(int rowIndex, int columnIndex) {

    if (rowIndex >= filterableRows.size()) {
      throw new RuntimeException("No existe objeto para a linha " + rowIndex);
    }

    if (columnIndex >= visibleColumns.size()) {
      throw new RuntimeException("No existe coluna com o ndice "
        + columnIndex);
    }

    T row = filterableRows.get(rowIndex);
    IConfigurableColumn<T> column = visibleColumns.get(columnIndex);

    return column.getValue(row);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAt(Object newValue, int rowIndex, int columnIndex) {

    if (rowIndex >= filterableRows.size()) {
      throw new RuntimeException("No existe objeto para a linha " + rowIndex);
    }

    if (columnIndex >= visibleColumns.size()) {
      throw new RuntimeException("No existe coluna com o ndice "
        + columnIndex);
    }

    T row = filterableRows.get(rowIndex);

    visibleColumns.get(columnIndex).setValue(row, newValue);
    fireTableCellUpdated(rowIndex, columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isCellEditable(int rowIndex, int columnIndex) {

    if (rowIndex >= filterableRows.size()) {
      throw new RuntimeException("No existe objeto para a linha " + rowIndex);
    }

    if (columnIndex >= visibleColumns.size()) {
      throw new RuntimeException("No existe coluna com o ndice "
        + columnIndex);
    }

    T row = filterableRows.get(rowIndex);
    return visibleColumns.get(columnIndex).isEditable(row);
  }

  /**
   * Obtm a coluna de ndice columnIndex.
   * 
   * @param columnIndex - ndice da coluna.
   * @return coluna de ndice columnIndex.
   */
  public IConfigurableColumn<T> getColumn(int columnIndex) {
    return visibleColumns.get(columnIndex);
  }

  /**
   * Remove todos os elementos.
   */
  public void clear() {
    int size = rows.size();
    this.rows = new ArrayList<T>();
    if (size > 0) {
      fireTableRowsDeleted(0, size - 1);
    }
    updateFilterableRows();
  }

  /**
   * Substitui a lista de objetos-linha.
   * 
   * @param newRows nova lista de objetos-linha.
   * 
   * @throws IllegalArgumentException se a lista de novas linhas for null
   */
  public void setRows(List<T> newRows) {
    if (newRows == null) {
      throw new IllegalArgumentException("newRows no pode ser nulo.");
    }
    this.rows = newRows;
    fireTableChanged(new TableModelEvent(this));
    updateFilterableRows();
  }

  /**
   * Adiciona um objeto-linha  lista. Se j existir um objeto com o mesmo
   * identificador, este  sobrescrito (no permite duplicatas).
   * 
   * @param row objeto-linha a ser adicionado.
   * 
   * @throws IllegalArgumentException se <code>row</code> for nulo.
   */
  public void add(T row) {
    add(row, true);
  }

  /**
   * Adiciona uma coleo de objetos-linha  lista. Se j existir um objeto com
   * o mesmo identificador, este  sobrescrito (no permite duplicatas).
   * 
   * @param rows coleo de objetos-linha a serem adicionados.
   * 
   * @throws IllegalArgumentException se <code>rows</code> for nulo.
   */
  public void addAll(Collection<T> rows) {
    if (rows == null) {
      throw new IllegalArgumentException("rows no pode ser nulo.");
    }
    for (T row : rows) {
      add(row, false);
    }
    fireTableChanged(new TableModelEvent(this));
    updateFilterableRows();
  }

  /**
   * Remove o objeto da tabela.
   * 
   * @param row objeto a ser removido.
   * 
   * @return <code>true</code> se o objeto for removido com sucesso,
   *         <code>false</code> caso contrrio.
   * 
   * @throws IllegalArgumentException se <code>row</code> for nulo.
   */
  public boolean remove(T row) {
    if (row == null) {
      throw new IllegalArgumentException("row no pode ser nulo.");
    }

    if (rows.contains(row)) {
      int index = rows.indexOf(row);
      rows.remove(index);
      updateFilterableRows();
      fireTableRowsDeleted(index, index);
      return true;
    }

    return false;
  }

  /**
   * Remove um objeto da lista pelo seu ndice do modelo.
   * 
   * @param rowIndex ndice do objeto a ser removido.
   * 
   * @return o objetoremovido.
   * 
   * @throws IndexOutOfBoundsException se o ndice for negativo ou se for maior
   *         do que o tamanho da lista de objeto-linha.
   */
  public T remove(int rowIndex) {
    T row = filterableRows.remove(rowIndex);
    rows.remove(row);
    fireTableRowsDeleted(rowIndex, rowIndex);
    return row;
  }

  /**
   * Remove os objetos especificados pelos seus ndices.
   * 
   * @param rowIndexes ndices dos objetos a serem removidos.
   * 
   * @return coleo de objetos removidos.
   * 
   * @throws IllegalArgumentException se <code>rowIndexes</code> for nulo.
   * @throws IndexOutOfBoundsException se algum ndice for negativo ou se for
   *         maior do que o tamanho da lista de objeto-linha.
   * 
   */
  public Collection<T> removeAll(int[] rowIndexes) {
    if (rowIndexes == null) {
      throw new IllegalArgumentException("rowIndexes no pode ser nulo.");
    }
    List<T> removedRows = new ArrayList<T>();
    if (rowIndexes.length > 0) {
      int start = Integer.MAX_VALUE, end = 0;
      for (int rowIndex : rowIndexes) {
        removedRows.add(filterableRows.get(rowIndex));
        start = Math.min(start, rowIndex);
        end = Math.max(end, rowIndex);
      }
      filterableRows.removeAll(removedRows);
      rows.removeAll(removedRows);
      fireTableRowsDeleted(start, end);
    }
    return removedRows;
  }

  /**
   * Obtm a lista de objetos (j filtrada) que correspondem as linhas da
   * tabela.
   * 
   * @return lista de objetos que correspondem as linhas da tabela.
   */
  public List<T> getRows() {
    return filterableRows;
  }

  /**
   * Obtm todos os objetos que correspondem as linhas da tabela.
   * 
   * @return todos os objetos que correspondem as linhas da tabela.
   */
  public List<T> getAllRows() {
    return rows;
  }

  /**
   * Obtm um objeto-linha especfico.
   * 
   * @param rowIndex ndice do objeto-linha
   * 
   * @return o objeto-linha solicitado, ou null
   * 
   * @throws IndexOutOfBoundsException se o ndice for invlido
   */
  public T getRow(int rowIndex) {
    return filterableRows.get(rowIndex);
  }

  /**
   * Filtro das linhas.
   * 
   * @return filtro das linhas.
   */
  public IFilter<T> getFilter() {
    return filter;
  }

  /**
   * Modifica o filtro das linhas.
   * 
   * @param filter - novo filtro.
   */
  public void setFilter(IFilter<T> filter) {
    this.filter = filter;
    updateFilterableRows();
  }

  /**
   * Obtm a coluna dado o seu identificador.
   * 
   * @param id - identificador nico da coluna.
   * @return coluna correspondnte.
   */
  public IConfigurableColumn<T> getColumnById(String id) {

    if (id == null) {
      throw new IllegalArgumentException("id da coluna no pode ser nulo.");
    }

    IConfigurableColumn<T> result = null;

    for (IConfigurableColumn<T> column : columns) {
      if (column.getId().equals(id)) {
        result = column;
        break;
      }
    }

    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void fireTableDataChanged() {
    updateVisibleColums();
    updateFilterableRows();

    fireTableChanged(new TableModelEvent(this));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void fireTableStructureChanged() {
    updateVisibleColums();
    updateFilterableRows();

    fireTableChanged(new TableModelEvent(this, TableModelEvent.HEADER_ROW));
  }

  /**
   * Lista com os estados de todas as colunas.
   * 
   * @return lista com os estados de todas as colunas.
   */
  public List<ColumnState> getColumnsState() {
    List<ColumnState> columnsState = new ArrayList<ColumnState>();

    for (IConfigurableColumn<T> column : getAllColumns()) {
      columnsState.add(new ColumnState(column.getId(), column.isVisible()));
    }

    return columnsState;
  }

  /**
   * Define os estados de todas as colunas.
   * 
   * @param columnsState - lista com os estados de todas as colunas.
   */
  public void setColumnsState(List<ColumnState> columnsState) {
    if (columnsState != null) {
      for (ColumnState state : columnsState) {
        IConfigurableColumn<T> column = getColumnById(state.getColumnId());
        if (column != null) {
          // atualiza a visibilidade da coluna.
          column.setVisible(state.isVisible());
        }
      }
    }
  }

  /**
   * Adiciona uma nova linha  lista, com a possibilidade de notificar ou no a
   * tabela.
   * 
   * @param row nova linha
   * @param notify <code>true</code> se a tabela deve ser notificada da insero
   */
  private void add(T row, boolean notify) {
    if (row == null) {
      throw new IllegalArgumentException("row no pode ser nulo.");
    }
    int index = rows.indexOf(row);
    if (index != -1) {
      rows.set(index, row);
      if (notify) {
        fireTableRowsUpdated(index, index);
      }
    }
    else {
      rows.add(row);
      if (notify) {
        index = rows.size() - 1;
        fireTableRowsInserted(index, index);
      }
    }
    updateFilterableRows();
  }

  /**
   * Atualiza a lista que mantm apenas as linhas filtradas.
   */
  private void updateFilterableRows() {
    filterableRows.clear();

    if (filter != null) {
      for (T row : rows) {
        if (filter.accept(row)) {
          filterableRows.add(row);
        }
      }
    }
    else {
      filterableRows.addAll(rows);
    }
  }

  /**
   * Atualiza a lista que mantm apenas as colunas visveis.
   */
  private void updateVisibleColums() {
    visibleColumns.clear();

    for (IConfigurableColumn<T> column : columns) {
      if (column.isVisible()) {
        visibleColumns.add(column);
      }
    }
  }

}
