/*
 * UserPermissionDefinitionDialog
 * 
 * $Author$ $Revision$ - $Date: 2006-08-15 16:42:08 -0300
 * (Tue, 15 Aug 2006) $
 */
package csbase.client.ias;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JPanel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.selector.ContainerSelection;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.remote.srvproxies.PermissionProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.Permission;
import csbase.logic.User;
import csbase.logic.UserInfo;

/**
 * A classe <code>UserPermissionDefinitionDialog</code>  responsvel pela
 * seleo de permisses avulsas para o usurio.
 * 
 * @author Glaucio e Ana
 * @version $Revision$
 */
public class UserPermissionDefinitionDialog {
  // #TODO Essa classe  praticamente idntica  UserRoleDefinitionDialog.
  // Ambas no poderiam ser uma s?
  /**
   * Referncia para o objeto que concentra os dados do usurio para facilitar a
   * persistncia
   */
  private UserInfo userInfo;

  /** Janela da classe "pai" */
  private DesktopComponentDialog ownerWindow;

  /** Janela desta classe, responsvel pela seleo de perfis */
  private DesktopComponentDialog mainDialog;

  /** Ttulo da janela */
  private String dialogTitle;

  /** Componente que permite a seleo de perfis do usurio */
  private ContainerSelection<Permission> permissionsSelection;

  /**
   * Obtm as permisses do usurio.
   * 
   * @return Um array contendo as permisses do usurio, ou um array vazio, caso
   *         este no possua nenhuma permisso.
   */
  private Vector<Permission> getUserPermissions() {
    Object[] permissionIds =
      (Object[]) userInfo.getAttribute(User.PERMISSION_IDS);
    if (permissionIds == null) {
      return new Vector<Permission>();
    }
    Vector<Permission> permissions =
      new Vector<Permission>(permissionIds.length);
    try {
      for (int i = 0; i < permissionIds.length; i++) {
        permissions.add(i, Permission.getPermission(permissionIds[i]));
      }
    }
    catch (Exception e) {
      StandardErrorDialogs.showErrorDialog(mainDialog, dialogTitle,
        LNG.get("IAS_USER_PERMISSIONS_RETRIEVAL_ERROR"));
    }
    return permissions;
  }

  /**
   * Cria o dilogo de seleo de permisses.
   */
  public void showDialog() {
    this.dialogTitle = LNG.get("IAS_USER_PERMISSIONS_TITLE");

    // Cria o dilogo
    mainDialog = new DesktopComponentDialog(ownerWindow, dialogTitle);

    // Pega as permisses j selecionadas (lado direito do componente de 
    // seleo) e subtrai das permisses disponveis para seleo (lado 
    // esquerdo do componente de seleo)
    Vector<Permission> permissions =
      PermissionProxy.getAllPermissions(mainDialog, dialogTitle,
        LNG.get("IAS_WAITING_ALL_PERMISSIONS"));
    if (permissions == null) {
      StandardErrorDialogs.showErrorDialog(mainDialog, dialogTitle,
        LNG.get("IAS_USER_PERMISSIONS_RETRIEVAL_ERROR"));
      return;
    }

    Container cp = mainDialog.getContentPane();
    JPanel selectionPanel =
      createSelectionPanel(permissions, getUserPermissions());
    cp.add(selectionPanel, BorderLayout.CENTER);
    JButton confirmButton = new JButton(LNG.get("IAS_CONFIRM"));
    confirmButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        updateUserPermissions();
        mainDialog.close();
      }
    });

    JButton cancelButton = new JButton(LNG.get("IAS_CANCEL"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        mainDialog.close();
      }
    });
    ClientUtilities.adjustEqualSizes(confirmButton, cancelButton);
    JPanel buttonsPanel = new JPanel();
    buttonsPanel.add(confirmButton);
    buttonsPanel.add(cancelButton);
    cp.add(buttonsPanel, BorderLayout.SOUTH);
    mainDialog.pack();
    mainDialog.center(ownerWindow);
    mainDialog.setVisible(true);
  }

  /**
   * Cria o painel para seleo de permisses.
   * 
   * @param selectablePermissions permisses disponveis para seleo
   * @param selectedPermissions permisses j selecionadas anteriormente
   * 
   * @return o painel criado
   */
  private JPanel createSelectionPanel(Vector<Permission> selectablePermissions,
    Vector<Permission> selectedPermissions) {
    ObjectTableProvider selectableFormat = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { LNG.get("IAS_AVAILABLE_PERMISSIONS") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { String.class };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object arg0) {
        Permission permission = (Permission) arg0;
        return (permission == null) ? null : new String[] { permission
          .getName() };
      }
    };
    ObjectTableProvider selectedFormat = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { String.class };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { LNG.get("IAS_SELECTED_PERMISSIONS") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object arg0) {
        Permission permission = (Permission) arg0;
        return (permission == null) ? null : new String[] { permission
          .getName() };
      }
    };
    permissionsSelection =
      new ContainerSelection<Permission>(selectableFormat, selectedFormat,
        true, false);
    permissionsSelection.loadItems(selectablePermissions, selectedPermissions);
    permissionsSelection.adjustTableColumns();
    return permissionsSelection.getPanel();
  }

  /**
   * Atualiza o objeto que concentra as informaes do usurio com as permisses
   * selecionadas.
   */
  void updateUserPermissions() {
    if (permissionsSelection == null) {
      userInfo.setAttribute(User.PERMISSION_IDS, new Object[0]);
      return;
    }
    List<Permission> permissions = permissionsSelection.getSelectedItems();
    Object[] permissionIds = new Object[permissions.size()];
    for (int i = 0; i < permissions.size(); i++) {
      permissionIds[i] = permissions.get(i).getId();
    }
    userInfo.setAttribute(User.PERMISSION_IDS, permissionIds);
  }

  /**
   * Limpa a seleo atual de permisses.
   */
  public void clearSelected() {
    if (permissionsSelection != null) {
      permissionsSelection.clearSelectedItems();
    }
    updateUserPermissions();
  }

  /**
   * Cria o dilogo de seleo de permisses de um usurio.
   * 
   * @param ownerWindow janela que criou este dilogo.
   * @param userInfo objeto que centraliza dados de um usurio. Usado para
   *        transportar a seleo de permisses realizada no dilogo.
   * @param user usurio sendo editado, permite comparar a seleo atual de
   *        permisses com a seleo antiga.
   * @param isNew indica se o dilogo est sendo chamado durante uma incluso de
   *        usurio (true) ou alterao (false).
   */
  public UserPermissionDefinitionDialog(DesktopComponentDialog ownerWindow,
    UserInfo userInfo, User user, boolean isNew) {
    // #TODO Rever a instanciao da classe para evitar a necessidade dos
    // parmetros isNew e user.    
    this.ownerWindow = ownerWindow;
    this.userInfo = userInfo;
    if (user == null && !isNew) {
      throw new IllegalArgumentException("user == null");
    }
    if (!isNew) {
      this.userInfo.setAttribute(User.PERMISSION_IDS,
        user.getAttribute(User.PERMISSION_IDS));
    }
  }

  /**
   * Construtor alternativo para novo usurio (sem permisses).
   * 
   * @param ownerDialog janela que criou este dilogo.
   * @param userInfo objeto que centraliza dados de um usurio. Usado para
   *        transportar a seleo de permisses realizada no dilogo.
   * @param isNew indica se o dilogo est sendo chamado durante uma incluso de
   *        usurio (true) ou alterao (false).
   */
  public UserPermissionDefinitionDialog(DesktopComponentDialog ownerDialog,
    UserInfo userInfo, boolean isNew) {
    this(ownerDialog, userInfo, null, isNew);
  }
}
