package csbase.client.remote.srvproxies.messageservice.consumers;

import java.rmi.RemoteException;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;

import tecgraf.javautils.core.filter.IFilter;
import csbase.remote.MessageServiceInterface;
import csbase.util.messages.Message;

/**
 * Implementa uma {@link IMessageConsumer estratgia de recebimento de
 * mensagens} que de tempos em tempos busca por novas mensagens no
 * {@link MessageServiceInterface servio de mensagens}.
 * 
 * @author Tecgraf
 */
public class MessagePollingConsumer implements IMessageConsumer {

  /**
   * Intermedirio entre receptores e produtores de mensagens.
   */
  private MessageServiceInterface service;

  /**
   * De quanto em quantos milisegundos deve buscar novas mensagens no
   * {@link MessageServiceInterface servio de mensagens}.
   */
  private long period;

  /**
   * Tarefa responsvel por fazer polling no {@link MessageServiceInterface
   * servio de mensagens} em busca de novas mensagens.
   */
  private TimerTask task;
  /**
   * Ouvinte que ir tratar as mensagens obtidas pelo timer.
   */
  private IListener listener;
  /**
   * Filtro que determina que mensagens sero consumidas do servio de mensagens
   * e repassadas para o {@link IMessageConsumer.IListener}.
   */
  private IFilter<Message> filter;

  /**
   * Construtor.
   * 
   * @param service broker de mensagens.
   * @param period De quanto em quanto tempo deve buscar novas mensagens no
   *        {@link MessageServiceInterface servio de mensagens}.
   * @param unit Unidade de tempo do parmetro period.
   */
  public MessagePollingConsumer(MessageServiceInterface service, long period,
    TimeUnit unit) {
    if (service == null) {
      throw new IllegalArgumentException("service == null");
    }
    if (period <= 0) {
      throw new IllegalArgumentException("period <= 0");
    }
    if (unit == null) {
      throw new IllegalArgumentException("unit == null");
    }

    this.service = service;
    this.period = unit.toMillis(period);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void setListener(IListener listener,
    IFilter<Message> filter) {

    if (listener == null) {
      throw new IllegalArgumentException("listener == null");
    }
    if (filter == null) {
      throw new IllegalArgumentException("filter == null");
    }

    this.listener = listener;
    this.filter = filter;
    startPolling();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void clearListener() {
    stopPolling();
    this.listener = null;
    this.filter = null;
  }

  /**
   * Comea a buscar novas mensagens no {@link MessageServiceInterface servio
   * de mensagens}.
   */
  public void startPolling() {
    if (task == null) {
      Timer timer = new Timer();
      this.task = new TimerTask() {
        @Override
        public void run() {
          try {
            Message[] messages = service.receive(filter);
            listener.onMessagesReceived(messages);
          }
          catch (RemoteException e) {
            listener.onExceptionThrown(e);
          }
        }
      };
      timer.scheduleAtFixedRate(task, 0, period);
    }
  }

  /**
   * Para de buscar novas mensagens no {@link MessageServiceInterface servio de
   * mensagens}.
   */
  private void stopPolling() {
    if (task != null) {
      task.cancel();
      task = null;
    }
  }
}
