package csbase.client.util.gui.log;

import java.awt.Dimension;

import javax.swing.JButton;
import javax.swing.JTextField;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.SwingConstants;

import csbase.client.util.event.EventListener;
import csbase.client.util.gui.log.LogPanel.PagingEvent;
import csbase.client.util.gui.log.LogPanelReloader.StatusChangedEvent;
import csbase.client.util.gui.log.actions.AutoReloadAction;
import csbase.client.util.gui.log.actions.CloseLogAction;
import csbase.client.util.gui.log.actions.DecPageAction;
import csbase.client.util.gui.log.actions.ExportLogAction;
import csbase.client.util.gui.log.actions.FirstPageAction;
import csbase.client.util.gui.log.actions.IncPageAction;
import csbase.client.util.gui.log.actions.LastPageAction;
import csbase.client.util.gui.log.actions.OpenLogAction;

/**
 * Barra de ferramentas customizvel a ser utilizada pelo painel de logs.
 * 
 * @author pasti
 */
public final class LogPanelToolBar extends JToolBar {

  /**
   * Adiciona a funcionalidade de abrir e fechar arquivos.
   */
  static final int OPEN_CLOSE = 0x01;

  /**
   * Adiciona a funcionalidade de auto-recarga do arquivo corrente.
   */
  static final int RELOAD = 0x02;

  /**
   * Adiciona a funcionalidade de exportao do arquivo de log corrente.
   */
  static final int EXPORT = 0x03;

  /**
   * Adiciona a funcionalidade de paginao (ir para a primeira pgina, pgina
   * anterior, pgina seguinte, ltima pgina).
   */
  static final int PAGING = 0x04;

  /**
   * Construtor para uma barra de ferramentas customizada.
   * 
   * @param logPanel painel de visualizao de logs.
   * @param toolBarFlags uma mscara de bits indicando que ferramentas devem
   *        aparecer na toolbar. As opes so: {@link #OPEN_CLOSE},
   *        {@link #RELOAD} e {@link #PAGING}.
   */
  public LogPanelToolBar(final LogPanel logPanel, final int toolBarFlags) {
    setFloatable(false);
    initializeComponents(logPanel, toolBarFlags);
  }

  /**
   * Faz a inicializao dos componentes.
   * 
   * @param logPanel painel de visualizao de logs.
   * @param toolBarFlags indicativos de botes.
   */
  private void initializeComponents(final LogPanel logPanel,
    final int toolBarFlags) {
    if (has(toolBarFlags, LogPanel.OPEN_CLOSE)) {
      createOpenCloseButtons(logPanel);
      addSeparator();
    }

    if (has(toolBarFlags, LogPanel.RELOAD)) {
      createReloadButton(logPanel);
      addSeparator();
    }

    if (has(toolBarFlags, LogPanel.EXPORT)) {
      createExportLogButton(logPanel);
      addSeparator();
    }

    if (has(toolBarFlags, LogPanel.PAGING)) {
      createPagingButtons(logPanel);
    }
  }

  /**
   * Cria os botes de abrir e fechar arquivos.
   * 
   * @param logPanel painel de visualizao de logs.
   */
  private void createOpenCloseButtons(final LogPanel logPanel) {
    add(new OpenLogAction(logPanel));
    add(new CloseLogAction(logPanel));
  }

  /**
   * Cria o boto de auto-recarga do arquivo corrente.
   * 
   * @param logPanel painel de visualizao de logs.
   */
  private void createReloadButton(final LogPanel logPanel) {
    final JToggleButton autoReloadButton = new JToggleButton();
    autoReloadButton.setAction(new AutoReloadAction(logPanel));
    autoReloadButton.setText(null);

    logPanel.getReloader().addStatusChangedEventListener(
      new EventListener<LogPanelReloader.StatusChangedEvent>() {
        @Override
        public void eventFired(StatusChangedEvent event) {
          autoReloadButton.setSelected(event.isRunning());
        }
      });

    add(autoReloadButton);
  }

  /**
   * Cria o boto para exportar o arquivo de log corrente.
   * 
   * @param logPanel painel de visualizao de logs.
   */
  private void createExportLogButton(final LogPanel logPanel) {
    final JButton exportLogButton = new JButton();
    exportLogButton.setAction(new ExportLogAction(logPanel));
    exportLogButton.setEnabled(false);
    exportLogButton.setText(null);

    add(exportLogButton);
  }

  /**
   * Cria os botes de paginao (ir para a primeira pgina, pgina anterior,
   * pgina seguinte, ltima pgina).
   * 
   * @param logPanel painel de visualizao de logs.
   */
  private void createPagingButtons(final LogPanel logPanel) {
    add(new FirstPageAction(logPanel));
    add(new DecPageAction(logPanel));

    /** Texto com nmero de pginas. */
    final JTextField pageNumberText = new JTextField(10);
    pageNumberText.setEditable(false);
    pageNumberText.setHorizontalAlignment(SwingConstants.CENTER);
    pageNumberText.setMaximumSize(new Dimension(150, 30));
    /** Inicializa o texto. */
    long page = logPanel.getPageNumber();
    long pagesCount = logPanel.countPages();
    setText(pageNumberText, page, pagesCount);
    /** Atualiza o texto em caso de eventos de paginao. */
    logPanel.addPagingEventListener(new EventListener<LogPanel.PagingEvent>() {
      @Override
      public void eventFired(PagingEvent event) {
        long page = event.getPageNumber();
        long pagesCount = event.countPages();
        setText(pageNumberText, page, pagesCount);
      }
    });
    add(pageNumberText);

    add(new IncPageAction(logPanel));
    add(new LastPageAction(logPanel));
  }

  /**
   * Utilizado pela funcionalidade de paginao, atribui ao componente um texto
   * indicando o total de pginas e a pgina corrente.
   * 
   * @param pageNumberText texto do campo que indica o nmero da pgina
   * @param page nmero da pgina corrente
   * @param pagesCount nmero total de pginas
   */
  private void setText(final JTextField pageNumberText, long page,
    long pagesCount) {
    if (0 == pagesCount) {
      pageNumberText.setText("");
    }
    else {
      String pageLabel = getString("paging.label", page, pagesCount);

      pageNumberText.setText(pageLabel);
    }
  }

  /**
   * Verifica se um conjunto de bits est contido em outro.
   * 
   * @param flags mscara de bits a serem testados.
   * @param flag a ser verificado se est contido em flags.
   * @return indicativo de bits contido
   */
  private boolean has(int flags, int flag) {
    return (flags & flag) > 0;
  }

  /**
   * Pede ao {@link LogPanel}, atravs do mtodo
   * {@link LogPanel#getString(String, Object...)}, um texto com a chave formada
   * por "ToolBar."+tag.
   * 
   * @param tag a tag
   * @param args argumento para a formatao de texto estilo
   *        {@link String#format(String, Object...)}
   * 
   * @return um texto sem espaos antes nem depois caso a chave tenha sido
   *         encontrada ou nulo caso contrrio.
   */
  private static final String getString(String tag, Object... args) {
    return LogPanel.getString("ToolBar." + tag, args);
  }
}
