package csbase.client.util.table;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.EventObject;
import java.util.LinkedList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.event.CellEditorListener;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;

/**
 * Tabela: facilita a criao de tabelas com colunas que exibem componentes
 * personalizados.
 * 
 * @param <R> O valor armazenado como linha da tabela.
 */
public final class Table<R> {
  /**
   * Modelo de dados da {@link JTable}.
   */
  private AbstractTableModel abstractTableModel;

  /**
   * Boto Adicionar.
   */
  private JButton addButton;

  /**
   * As colunas.
   */
  private List<TableColumn<R>> columns;

  /**
   * Indica se a tabela est habilitada ou desabilitada.
   */
  private boolean isEnabled;

  /**
   * A {@link javax.swing.JTable}.
   */
  private JTable jtable;

  /**
   * Os observadores.
   */
  private List<TableListener<R>> listeners;

  /**
   * Boto Remover Todos.
   */
  private JButton removeAllButton;

  /**
   * Boto Remover.
   */
  private JButton removeButton;

  /**
   * O modelo das linhas.
   */
  private RowModel<R> rowModel;

  /**
   * A fbrica de valores de linhas.
   */
  private RowValueFactory<R> rowValueFactory;

  /**
   * A viso da tabela.
   */
  private JPanel view;

  /**
   * Indica o nmero de linhas da tabela que devem estar visveis (no caso do
   * nmero de linhas ser varivel).
   */
  private Integer visibleRowCount;

  /**
   * Nmero mnimo de linhas
   */
  private Integer minRowCount;

  /**
   * Nmero mximo de linhas
   */
  private Integer maxRowCount;

  /**
   * Cria uma tabela.
   * 
   * @param rowModel O modelo das linhas (No aceita {@code null}).
   * @param columns As colunas da tabela (No aceita {@code null} e no pode
   *        estar vazio).
   */
  public Table(RowModel<R> rowModel, List<TableColumn<R>> columns) {
    this(null, rowModel, columns, null, null, null);
  }

  /**
   * Cria uma tabela que permite adicionar e remover linhas.
   * 
   * @param rowValueFactory A fbrica de linhas (Aceita {@code null}).
   * @param rowModel O modelo das linhas (No aceita {@code null}).
   * @param columns As colunas da tabela (No aceita {@code null} e no pode
   *        estar vazio).
   * @param visibleRowCount O nmero de linhas da tabela que devem estar
   *        visveis.
   * @param minRowCount O mnimo nmero de linhas da tabela.
   * @param maxRowCount O mnimo nmero de linhas da tabela.
   */
  public Table(RowValueFactory<R> rowValueFactory, RowModel<R> rowModel,
    List<TableColumn<R>> columns, Integer visibleRowCount, Integer minRowCount,
    Integer maxRowCount) {
    listeners = new LinkedList<TableListener<R>>();
    this.rowValueFactory = rowValueFactory;
    this.visibleRowCount = visibleRowCount;
    this.minRowCount = minRowCount;
    this.maxRowCount = maxRowCount;
    setRowModel(rowModel);
    setColumns(columns);
    createAbstractTableModel();
    createCells();
    createCellModelListeners();

    createJTable();

    if (minRowCount != null) {
      int lastRow = getRowCount();
      while (lastRow < minRowCount) {
        addRow(rowValueFactory.create(lastRow));
        lastRow = getRowCount();
      }
    }

    view = new JPanel(new BorderLayout());
    view.add(new JScrollPane(jtable), BorderLayout.CENTER);
    if (rowValueFactory != null) {
      JPanel buttonPanel1 = createButtonPanel();
      view.add(buttonPanel1, BorderLayout.EAST);
      addTableListener(new TableListener<R>() {
        public void rowWasAdded(R newRowValue) {
          updateAddButton();
          updateRemoveButton();
          updateRemoveAllButton();
        }

        public void rowWasRemoved(int rowIndex) {
          updateAddButton();
          updateRemoveButton();
          updateRemoveAllButton();
        }
      });
    }
    setEnabled(true);
  }

  /**
   * Adiciona uma linha.
   * 
   * @param rowValue O valor armazenado na linha (No aceita {@code null}).
   */
  public void addRow(R rowValue) {
    rowModel.addRow(rowValue);
  }

  /**
   * Adiciona um observador de tabela.
   * 
   * @param listener O observador (No aceita {@code null}).
   */
  public void addTableListener(TableListener<R> listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    listeners.add(listener);
  }

  /**
   * Obtm o nmero de colunas.
   * 
   * @return O nmero de colunas.
   */
  public int getColumnCount() {
    return columns.size();
  }

  /**
   * Obtm o nmero de linhas da tabela.
   * 
   * @return O nmero de linhas da tabela.
   */
  public int getRowCount() {
    return rowModel.getRowCount();
  }

  /**
   * Obtm os valores das linhas.
   * 
   * @return Os valores das linhas.
   */
  public List<R> getRowValues() {
    return rowModel.getRows();
  }

  /**
   * Obtm a dica da tabela.
   * 
   * @return A dica ou {@code null} se no houver uma dica.
   */
  public String getToolTipText() {
    return jtable.getToolTipText();
  }

  /**
   * Obtm a viso desta tabela.
   * 
   * @return A viso.
   */
  public JComponent getView() {
    return view;
  }

  /**
   * Indica se a possui linhas.
   * 
   * @return .
   */
  public boolean hasRows() {
    return getRowCount() != 0;
  }

  /**
   * Indica se esta tabela est habilita ou desabilitada.
   * 
   * @return .
   */
  public boolean isEnabled() {
    return isEnabled;
  }

  /**
   * Remove uma linha.
   * 
   * @param rowIndex O ndice da linha (Precisa ser maior ou igual a 0 e ser
   *        menor do que o nmero de linhas).
   */
  public void removeRow(int rowIndex) {
    assertRowIndex(rowIndex);
    rowModel.removeRow(rowIndex);
  }

  /**
   * Remove um observador de tabela.
   * 
   * @param listener O observador (No aceita {@code null}).
   * 
   * @return {@code true} em caso de sucesso ou {@code false} se o observador
   *         no estava cadastrado.
   */
  public boolean removeTableListener(TableListener<R> listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    return listeners.remove(listener);
  }

  /**
   * Seleciona uma clula especfica, removendo as seleo das outras clulas.
   * 
   * @param rowIndex O ndice da linha (O ndice da linha tem que ser maior ou
   *        igual a 0 e menor do que a quantidade de linhas).
   * @param columnIndex O ndice da coluna (O ndice da coluna tem que ser maior
   *        ou igual a 0 e menor do que a quantidade de colunas).
   */
  public void selectCell(int rowIndex, int columnIndex) {
    if (rowIndex < 0 || rowIndex >= getRowCount()) {
      String message =
        String.format("ndice da linha (%d) fora da faixa esperada (0 a %d).",
          rowIndex, getRowCount());
      throw new IllegalArgumentException(message);
    }
    if (columnIndex < 0 || columnIndex >= getColumnCount()) {
      String message =
        String.format("ndice da linha (%d) fora da faixa esperada (0 a %d).",
          columnIndex, getColumnCount() - 1);
      throw new IllegalArgumentException(message);
    }
    jtable.changeSelection(rowIndex, columnIndex, false, false);
    // jtable.editCellAt(rowIndex, columnIndex);
  }

  /**
   * Habilita/Desabilita esta tabela.
   * 
   * @param isEnabled Indica se  para habilitar ou desabilitar a tabela.
   */
  public void setEnabled(boolean isEnabled) {
    this.isEnabled = isEnabled;
    jtable.setEnabled(isEnabled);
    view.setEnabled(isEnabled);
    updateAddButton();
    updateRemoveButton();
    updateRemoveAllButton();
  }

  /**
   * Modifica a dica da tabela.
   * 
   * @param toolTipText A dica (Aceita {@code null}).
   */
  public void setToolTipText(String toolTipText) {
    jtable.setToolTipText(toolTipText);
  }

  /**
   * Ajusta a altura das linhas da tabela.
   */
  private void adjustRowHeight() {
    int rowHeight = 0;
    for (int columnIndex = 0; columnIndex < getColumnCount(); columnIndex++) {
      TableColumn<R> column = getColumn(columnIndex);
      for (int rowIndex = 0; rowIndex < getRowCount(); rowIndex++) {
        CellView cell = column.getCellView(rowIndex);
        JComponent cellView = cell.getView();
        int cellHeight = (int) cellView.getPreferredSize().getHeight();
        rowHeight = Math.max(rowHeight, cellHeight);
      }
    }
    jtable.setRowHeight(rowHeight);
  }

  /**
   * Assegura que o ndice da linha  vlido (De 0 a nmero de linhas -1).
   * 
   * @param rowIndex ndice da linha.
   */
  private void assertRowIndex(int rowIndex) {
    if ((rowIndex < 0) || (getRowCount() <= rowIndex)) {
      throw new IllegalArgumentException(String.format(
        "ndice de linha fora da faixa esperada.\nFaixa esperada 0 a %d.",
        getRowCount() - 1));
    }
  }

  /**
   * Cria o {@link javax.swing.table.TableModel} da {@link JTable}.
   */
  private void createAbstractTableModel() {
    abstractTableModel = new AbstractTableModel() {

      @Override
      public Class<?> getColumnClass(int columnIndex) {
        return Object.class;
      }

      public int getColumnCount() {
        return columns.size();
      }

      @Override
      public String getColumnName(int columnIndex) {
        TableColumn<R> column = getColumn(columnIndex);
        return column.getTitle();
      }

      public int getRowCount() {
        return rowModel.getRowCount();
      }

      public Object getValueAt(int rowIndex, int columnIndex) {
        TableColumn<R> column = getColumn(columnIndex);
        CellModel cellModel = column.getCellModel(rowIndex);
        return cellModel.getValue();
      }

      @Override
      public boolean isCellEditable(int rowIndex, int columnIndex) {
        TableColumn<R> column = getColumn(columnIndex);
        return column.isEditable();
      }

      @Override
      public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
        TableColumn<R> column = getColumn(columnIndex);
        CellModel cellModel = column.getCellModel(rowIndex);
        cellModel.setValue(aValue);
      }
    };
  }

  /**
   * Cria o painel com os botes.
   * 
   * @return .
   */
  private JPanel createButtonPanel() {
    JPanel buttonPanel = new JPanel(new GridBagLayout());

    createAddButton();
    createRemoveButton();
    createRemoveAllButton();

    final int bIns = 3;
    buttonPanel.add(addButton, new GBC(0, 0).center().insets(bIns));
    buttonPanel.add(removeButton, new GBC(0, 1).center().insets(bIns));
    buttonPanel.add(removeAllButton, new GBC(0, 2).center().insets(bIns));
    final JComponent[] array =
      new JComponent[] { addButton, removeButton, removeAllButton };
    GUIUtils.matchPreferredSizes(array);

    final int brd = 5;
    buttonPanel.setBorder(BorderFactory.createEmptyBorder(brd, brd, brd, brd));

    return buttonPanel;
  }

  /**
   * Cria as clulas de uma linha.
   * 
   * @param rowIndex O ndice da linha (Precisa ser maior ou igual a 0 e menor
   *        do que o nmero de linhas).
   * @param rowValue O objeto armazenado na linha (No aceita {@code null}).
   */
  private void createCell(int rowIndex, R rowValue) {
    for (TableColumn<R> column : columns) {
      column.createCell(rowIndex, rowValue);
    }
  }

  /**
   * Cria os observadores dos modelos das clulas para uma linha.
   * 
   * @param rowIndex O ndice da linha.
   */
  private void createCellModelListener(int rowIndex) {
    for (TableColumn<R> column : columns) {
      CellModel cellModel = column.getCellModel(rowIndex);
      cellModel.addCellModelListener(new CellModelListener() {
        public void valueWasChanged(CellModel cm) {
          abstractTableModel.fireTableDataChanged();
        }
      });
    }
  }

  /**
   * Cria os observadores dos modelos das clulas.
   */
  private void createCellModelListeners() {
    for (int i = 0; i < getRowCount(); i++) {
      createCellModelListener(i);
    }
  }

  /**
   * Cria as clulas.
   */
  private void createCells() {
    for (int rowIndex = 0; rowIndex < getRowCount(); rowIndex++) {
      R rowValue = rowModel.getRow(rowIndex);
      createCell(rowIndex, rowValue);
    }
  }

  /**
   * Cria a {@link JTable}.
   */
  private void createJTable() {
    // Correo do BUG 4330950
    jtable = new JTable(abstractTableModel) {
      @Override
      public void editingCanceled(ChangeEvent e) {
        super.removeEditor();
      }

      @Override
      public void editingStopped(ChangeEvent e) {
        TableCellEditor editor = getCellEditor();
        if (editor != null) {
          Object value = editor.getCellEditorValue();
          setValueAt(value, editingRow, editingColumn);
          super.removeEditor();
        }
      }

      @Override
      public void removeEditor() {
        if (isEditing()) {
          getCellEditor().stopCellEditing();
        }
        super.removeEditor();
      }
    };
    jtable.setDefaultRenderer(Object.class, createTableCellRenderer());
    jtable.setDefaultEditor(Object.class, createTableCellEditor());

    // Correo do BUG 4709394 do Swing:
    jtable.putClientProperty("terminateEditOnFocusLost", Boolean.TRUE);

    jtable.setColumnSelectionAllowed(false);
    if (rowValueFactory != null) {
      jtable.setCellSelectionEnabled(false);
      jtable.setRowSelectionAllowed(true);
    }
    else {
      jtable.setRowSelectionAllowed(false);
      jtable.setCellSelectionEnabled(true);
    }

    jtable.getSelectionModel().addListSelectionListener(
      new ListSelectionListener() {
        public void valueChanged(ListSelectionEvent e) {
          updateRemoveButton();
        }
      });

    if (!rowModel.isEmpty()) {
      adjustRowHeight();
    }

    // Compactando a altura da viewport para evitar que aparea uma rea
    // vazia quando a quantidade de linhas for fixa e a quantidade de linhas
    // for pequena.
    if (rowValueFactory == null) {
      Dimension tableSize = jtable.getPreferredSize();
      Dimension viewPortSize = jtable.getPreferredScrollableViewportSize();
      double height = Math.min(tableSize.getHeight(), viewPortSize.getHeight());
      Dimension dimension = new Dimension();
      dimension.setSize(viewPortSize.getWidth(), height);
      jtable.setPreferredScrollableViewportSize(dimension);
    }
    else {
      int rowHeight = jtable.getRowHeight();
      int visibleRows;
      if (visibleRowCount != null) {
        visibleRows = visibleRowCount;
      }
      else {
        visibleRows = 5;
      }
      // Altura de n linhas (descontando o tamanho das bordas)
      int height = (visibleRows * rowHeight) - visibleRows;
      int width = (int) jtable.getPreferredSize().getWidth();
      jtable.setPreferredScrollableViewportSize(new Dimension(width, height));
    }
  }

  /**
   * Cria o Boto Remover Todos.
   */
  private void createRemoveAllButton() {
    final String label = LNG.get("Table.remove.all.button");
    removeAllButton = new JButton(new AbstractAction(label) {
      public void actionPerformed(ActionEvent e) {
        removeAllRows();
      }
    });
    updateRemoveAllButton();
  }

  /**
   * Cria o Boto Remover.
   */
  private void createRemoveButton() {
    final String label = LNG.get("Table.remove.button");
    removeButton = new JButton(new AbstractAction(label) {
      public void actionPerformed(ActionEvent e) {
        removeSelectedRows();
      }
    });
    updateRemoveButton();
  }

  /**
   * Cria o Boto Adicionar.
   */
  private void createAddButton() {
    final String label = LNG.get("Table.add.button");
    addButton = new JButton(new AbstractAction(label) {
      public void actionPerformed(ActionEvent e) {
        addRow(rowValueFactory.create(getRowCount()));
      }
    });
    updateAddButton();
  }

  /**
   * Cria o {@link TableCellEditor} da {@link JTable}.
   * 
   * @return .
   */
  private TableCellEditor createTableCellEditor() {
    return new TableCellEditor() {

      private TableColumn<R> currentColumn = getColumn(0);

      private CellView currentCellView;

      private List<CellEditorListener> editListeners =
        new ArrayList<CellEditorListener>();

      public void addCellEditorListener(CellEditorListener l) {
        editListeners.add(l);
      }

      public void cancelCellEditing() {
        for (int i = 0; i < editListeners.size(); i++) {
          CellEditorListener listener = editListeners.get(i);
          listener.editingCanceled(new ChangeEvent(this));
        }
      }

      public Object getCellEditorValue() {
        return currentCellView.getValue();
      }

      public Component getTableCellEditorComponent(JTable table, Object value,
        boolean isSelected, int rowIndex, int columnIndex) {
        currentColumn = getColumn(columnIndex);
        currentCellView = currentColumn.getCellView(rowIndex);
        updateCellViewColors(currentCellView, false, currentColumn.isEditable());
        currentCellView.setFocus(true);
        currentCellView.startEditing();
        JComponent cellView = currentCellView.getView();
        if (currentCellView.getTip() != null) {
          cellView.setToolTipText(currentCellView.getTip());
        }
        return cellView;
      }

      public boolean isCellEditable(EventObject anEvent) {
        return true;
      }

      public void removeCellEditorListener(CellEditorListener l) {
        editListeners.remove(l);
      }

      public boolean shouldSelectCell(EventObject anEvent) {
        return true;
      }

      public boolean stopCellEditing() {
        if (!currentCellView.validate()) {
          return false;
        }
        currentCellView.stopEditing();
        for (int i = 0; i < editListeners.size(); i++) {
          CellEditorListener listener = editListeners.get(i);
          listener.editingStopped(new ChangeEvent(this));
        }
        return true;
      }
    };
  }

  /**
   * Cria o {@link TableCellRenderer} da {@link JTable}.
   * 
   * @return .
   */
  private TableCellRenderer createTableCellRenderer() {
    return new TableCellRenderer() {

      public Component getTableCellRendererComponent(JTable table,
        Object value, boolean isSelected, boolean hasFocus, int rowIndex,
        int columnIndex) {
        TableColumn<R> column = getColumn(columnIndex);
        CellView cellView = column.getCellView(rowIndex);
        boolean isEditable = column.isEditable();
        updateCellViewColors(cellView, isSelected, isEditable);
        cellView.setFocus(hasFocus);
        if (!cellView.isEditing()) {
          cellView.setValue(value);
        }
        final JComponent vw = cellView.getView();
        return vw;
      }
    };
  }

  /**
   * Obtm uma coluna da tabela.
   * 
   * @param columnIndex O ndice da coluna (No pode ser negativo e no pode ser
   *        maior ou igual ao nmero de colunas).
   * 
   * @return A coluna.
   */
  private TableColumn<R> getColumn(int columnIndex) {
    return columns.get(columnIndex);
  }

  /**
   * Indice se existem linhas selecionadas.
   * 
   * @return .
   */
  private boolean hasSelectedRows() {
    return jtable.getSelectedRowCount() != 0;
  }

  /**
   * <p>
   * Remove todas as linhas.
   * </p>
   */
  private void removeAllRows() {
    while (!rowModel.isEmpty()) {
      removeRow(0);
    }
  }

  /**
   * <p>
   * Remove as linhas selecionadas.
   * </p>
   */
  private void removeSelectedRows() {
    for (int rowIndex = 0; rowIndex < getRowCount();) {
      if (jtable.isRowSelected(rowIndex)) {
        removeRow(rowIndex);
      }
      else {
        rowIndex++;
      }
    }
  }

  /**
   * Seleciona uma linha.
   * 
   * @param rowIndex O ndice da linha (De 0 a nmeros de linhas - 1).
   */
  private void selectRow(int rowIndex) {
    jtable.getSelectionModel().setSelectionInterval(rowIndex, rowIndex);
  }

  /**
   * Atribui as colunas a esta tabela.
   * 
   * @param columns As colunas da tabela (No aceita {@code null} e no pode
   *        estar vazio).
   */
  private void setColumns(List<TableColumn<R>> columns) {
    if (columns == null) {
      throw new IllegalArgumentException("O parmetro columns est nulo.");
    }

    if (columns.isEmpty()) {
      throw new IllegalArgumentException("O parmetro columns est vazio.");
    }
    this.columns = new ArrayList<TableColumn<R>>(columns);
  }

  /**
   * Atribui o modelo das linhas.
   * 
   * @param rowModel O modelo das linhas.
   */
  private void setRowModel(RowModel<R> rowModel) {
    if (rowModel == null) {
      throw new IllegalArgumentException("O parmetro rowModel est nulo.");
    }
    this.rowModel = rowModel;
    rowModel.addRowModelListener(new RowModelListener<R>() {
      public void rowWasAdded(R newRowValue, int newRowIndex) {
        createCell(newRowIndex, newRowValue);
        createCellModelListener(newRowIndex);
        abstractTableModel.fireTableRowsInserted(newRowIndex, newRowIndex);
        adjustRowHeight();
        selectRow(newRowIndex);
        for (TableListener<R> listener : listeners) {
          listener.rowWasAdded(newRowValue);
        }
      }

      public void rowWasRemoved(R oldRowValue, int oldRowIndex) {
        for (TableColumn<R> column : columns) {
          column.removeCell(oldRowIndex);
        }
        abstractTableModel.fireTableRowsDeleted(oldRowIndex, oldRowIndex);
        for (TableListener<R> listener : listeners) {
          listener.rowWasRemoved(oldRowIndex);
        }
      }
    });
  }

  /**
   * Atualiza as cores de uma viso de clula.
   * 
   * @param cellView A viso de clula (No aceita {@code null}).
   * @param isSelected Indica se a clula est selecionada.
   * @param isEditable Indica se a clula  editvel.
   */
  private void updateCellViewColors(CellView cellView, boolean isSelected,
    boolean isEditable) {
    if (isSelected) {
      cellView.setForegroundColor(jtable.getSelectionForeground());
      cellView.setBackgroundColor(jtable.getSelectionBackground());
    }
    else if (isEditable) {
      cellView.setForegroundColor(jtable.getForeground());
      cellView.setBackgroundColor(jtable.getBackground());
    }
    else {
      cellView.setForegroundColor(jtable.getForeground());
      Component component = jtable.getParent();
      if (component == null) {
        component = jtable;
      }
      cellView.setBackgroundColor(component.getBackground());
    }
  }

  /**
   * Atualiza o Boto Remover Todos.
   */
  private void updateRemoveAllButton() {
    if (removeAllButton != null) {
      final boolean canRemoveAll = canRemoveAll();
      removeAllButton.setEnabled(canRemoveAll);
    }
  }

  /**
   * Verifica se o usurio pode remover todos os elementos.
   * 
   * @return indicativo
   */
  private boolean canRemoveAll() {
    if (!isEnabled()) {
      return false;
    }
    if (!hasRows()) {
      return false;
    }
    if (minRowCount != null && minRowCount > 0) {
      return false;
    }
    return true;
  }

  /**
   * Verifica se o usurio pode remover um elemento.
   * 
   * @return indicativo
   */
  private boolean canRemove() {
    if (!isEnabled()) {
      return false;
    }
    if (!hasSelectedRows()) {
      return false;
    }
    final int nRows = getRowCount();
    if (minRowCount != null && nRows == minRowCount) {
      return false;
    }
    return true;
  }

  /**
   * Verifica se o usurio pode adicionar um elemento.
   * 
   * @return indicativo
   */
  private boolean canAdd() {
    if (!isEnabled()) {
      return false;
    }
    final int nRows = getRowCount();
    if (maxRowCount != null && nRows == maxRowCount) {
      return false;
    }
    return true;
  }

  /**
   * Atualiza o Boto Remover.
   */
  private void updateRemoveButton() {
    if (removeButton != null) {
      final boolean canRemove = canRemove();
      removeButton.setEnabled(canRemove);
    }
  }

  /**
   * Atualiza o Boto Remover.
   */
  private void updateAddButton() {
    if (addButton != null) {
      final boolean canAdd = canAdd();
      addButton.setEnabled(canAdd);
    }
  }

}
