package csbase.client.applications.sgamonitor;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.Action;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JToolBar;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import csbase.client.Client;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.sgamonitor.actions.DetailsSGAClusterAction;
import csbase.client.applications.sgamonitor.actions.RestartSGAClusterAction;
import csbase.client.applications.sgamonitor.actions.StopSGAClusterAction;
import csbase.client.facilities.configurabletable.TableConfigUtil;
import csbase.client.facilities.configurabletable.UI.TabbedPane4Tables;
import csbase.client.facilities.configurabletable.UI.UI4Tables;
import csbase.client.facilities.configurabletable.UIFactory;
import csbase.client.facilities.configurabletable.model.Config;
import csbase.client.facilities.configurabletable.stringprovider.ApplicationStringProvider;
import csbase.client.facilities.configurabletable.stringprovider.IStringProvider;
import csbase.client.facilities.configurabletable.table.ConfigurableTable;
import csbase.client.facilities.configurabletable.table.RowToKey;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.types.PVTables;
import csbase.client.remote.srvproxies.messageservice.MessageProxy;
import csbase.logic.AdminPermission;
import csbase.logic.Permission;
import csbase.logic.SGAAdminPermission;
import csbase.logic.SGANotification;
import csbase.logic.SGASet;
import csbase.logic.User;
import csbase.logic.Utilities;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SGAServiceInterface;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;
import tecgraf.javautils.gui.StatusBar;
import tecgraf.javautils.gui.SwingThreadDispatcher;

/**
 * Aplicao encarregada de monitorar SGAs (simples e clusters). <br>
 * Essa aplicao faz uso das tabelas configurveis, {@link ConfigurableTable},
 * geradas pela fbrica {@link UIFactory}. Esta aplicao adota que todas as
 * tabelas exibidas no frame principal sejam tabelas que exibem elementos do
 * tipo {@link SGASet}.
 * 
 * @see ConfigurableTable UIFactory
 * 
 * @author Tecgraf
 */
public class SGAMonitor extends Application {

  /**
   * Identificador do painel que est no XML de configurao de tabelas.
   */
  private static final String MAIN_PANEL = "main-panel";

  /**
   * Ao que exibe os detalhes de um SGA/Cluster.
   */
  private Action detailsSGAClusterAction;

  /**
   * Ao que reinicia um SGA/Cluster.
   */
  private Action restartSGAClusterAction;

  /**
   * Ao que desativa um SGA/Cluster.
   */
  private Action stopSGAClusterAction;

  /**
   * Container de tabelas da aplicao.
   */
  private UI4Tables ui4Tables;

  /**
   * Configurao das tabelas da aplicao.
   */
  private Config config;

  /**
   * Ouvinte das notificaes dos SGAs - SGANotification -.<br>
   *  o encarregado de atualizar os dados das tabelas de monitorao.
   */
  private IMessageListener sgaListener;

  /**
   * Indica se o usurio logado tem permisso de reiniciar e desativar SGAs
   */
  private boolean SGAAdministrationPermission;

  /**
   * Construtor padro.
   * 
   * @param id - identificador da aplicao.
   * @throws ApplicationException - exceo lanada caso no consiga criar as
   *         tabelas.
   * @throws RemoteException - caso no consiga referncia para os SGAs.
   */
  public SGAMonitor(String id) throws ApplicationException, RemoteException {
    super(id);
    this.config = getTablesConfig();

    IStringProvider stringProvider = new ApplicationStringProvider(this);
    UIFactory uiFactory = new UIFactory(config, stringProvider);
    ui4Tables = uiFactory.getUI4Tables(MAIN_PANEL);

    buildActions();
    buildInterface();
    addTableSelectionObject();
    addTableListeners();
    loadTablePreferences();

    SGAAdministrationPermission = checkSGAAdministrationPermission();

    // Registra o ouvinte de notificaes de SGA.
    sgaListener = new IMessageListener() {
      @Override
      public void onMessagesReceived(final Message... messages)
        throws Exception {

        final List<SGASet> newRows = new ArrayList<SGASet>();
        for (Message aMessage : messages) {
          SGANotification notification = (SGANotification) aMessage.getBody();
          newRows.add(notification.getMainInfo());
        }

        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            List<ConfigurableTable<SGASet>> tables =
              ui4Tables.getAllTables(SGASet.class);
            if (tables.size() == 0) {
              return;
            }

            // Cria um mapa contendo os antigos SGASet dado seus nomes.
            Map<String, SGASet> rows = new HashMap<String, SGASet>();
            List<SGASet> oldRows = new ArrayList<SGASet>();

            for (ConfigurableTable<SGASet> table : tables) {
              oldRows.addAll(table.getRows());
            }

            for (SGASet anOldRow : oldRows) {
              rows.put(anOldRow.getName(), anOldRow);
            }
            // Sobrescreve no mapa os SGASet que foram atualizados.
            for (SGASet aNewRow : newRows) {
              rows.put(aNewRow.getName(), aNewRow);
            }

            setData(new ArrayList<SGASet>(rows.values()));
          }
        });
      }
    };
    MessageProxy.addListener(sgaListener, SGANotification.class);

    // Faz a atualizao inicial da lista.
    final List<SGASet> data = getSGASetList();
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        setData(data);
      }
    });
  }

  /**
   * Obtm todos os SGAs selecionados.
   * 
   * @return todos os SGAs selecionados.
   */
  public List<SGASet> getSelectedSGAs() {
    List<SGASet> selectedSGAs = new ArrayList<SGASet>();

    for (ConfigurableTable<SGASet> table : ui4Tables
      .getTablesFromSelectedComponent(SGASet.class)) {
      selectedSGAs.addAll(table.getSelectedObjects());
    }

    return selectedSGAs;
  }

  /**
   * Obtm a configurao das tabelas.
   * 
   * @return configurao das tabelas.
   */
  public Config getConfig() {
    return this.config;
  }

  /**
   * Atualiza todas as tabelas da aplicao.
   */
  public void refreshTables() {
    for (ConfigurableTable<?> table : ui4Tables.getAllTables()) {
      table.updateRows();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() {
    // salvando as preferncias da(s) tabela(s) do menu principal
    PreferenceCategory appPrefs = getPreferences();

    PVTables pv = (PVTables) appPrefs.getPreference(SGAMonitorPref.MAIN_TABLES);
    pv.storeTables(ui4Tables.getAllTables());
    savePreferences();

    MessageProxy.removeListener(sgaListener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    return true;
  }

  /**
   * Atualiza o estado das Aes.<br>
   * Esse mtodo  utilizado no listener de seleo das tabelas da aplicao.
   */
  protected void updateActions() {
    List<SGASet> selectedObjects = getSelectedSGAs();

    detailsSGAClusterAction.setEnabled(false);
    restartSGAClusterAction.setEnabled(false);
    stopSGAClusterAction.setEnabled(false);

    boolean allAlive = true;
    for (SGASet sga : selectedObjects) {
      if (!sga.getAlive()) {
        allAlive = false;
        break;
      }
    }

    if (SGAAdministrationPermission && selectedObjects.size() > 0 && allAlive) {
      restartSGAClusterAction.setEnabled(true);
      stopSGAClusterAction.setEnabled(true);
    }

    if (selectedObjects.size() == 1 && allAlive) {
      detailsSGAClusterAction.setEnabled(true);
    }

  }

  /**
   * Obtm a configurao das tabelas desta aplicao.
   * 
   * @return configurao - configurao das tabelas.
   * @throws ApplicationException - caso no consiga buscar as configuraes do
   *         servidor.
   */
  private Config getTablesConfig() throws ApplicationException {
    try (InputStream input = getResource(new String[] { "tableConfig.xml" })) {
      if (input == null) {
        throw new ApplicationException(
          "arquivo tableConfig.xml no encontrado.");
      }
      try (Reader reader = new InputStreamReader(input)) {
        return TableConfigUtil.createConfig(reader);
      }
    }
    catch (IOException e) {
      throw new ApplicationException(
        "Falha no acesso ao arquivo tableConfig.xml", e);
    }
  }

  /**
   * Cria a barra de menu da aplicao.
   * 
   * @return barra de menu da aplicao.
   */
  private JMenuBar createMenuBar() {
    JMenu serversMenu = new JMenu(getString("menu.servers"));
    serversMenu.add(detailsSGAClusterAction);
    serversMenu.add(restartSGAClusterAction);
    serversMenu.add(stopSGAClusterAction);
    serversMenu.addSeparator();
    serversMenu.add(new ApplicationExitAction(this));

    JMenu tablesMenu = new JMenu(getString("menu.tables"));
    for (ConfigurableTable<?> table : ui4Tables.getAllTables()) {
      JMenu subMenu = new JMenu(ui4Tables.getTableLabel(table.getId()));

      for (JCheckBoxMenuItem item : table.createColumnsCheckBoxes()) {
        subMenu.add(item);
      }

      tablesMenu.add(subMenu);
    }

    JMenuBar menuBar = new JMenuBar();
    menuBar.add(serversMenu);
    menuBar.add(tablesMenu);

    return menuBar;
  }

  /**
   * Cria a barra de ferramentas da aplicao.
   * 
   * @return barra de ferramentas.
   */
  private JToolBar createToolBar() {
    JToolBar toolBar = new JToolBar();
    toolBar.setFloatable(false);
    toolBar.add(detailsSGAClusterAction);
    toolBar.add(restartSGAClusterAction);
    toolBar.add(stopSGAClusterAction);
    return toolBar;
  }

  /**
   * Adiciona ouvintes nas tabelas da janela principal da aplicao.
   */
  private void addTableListeners() {
    for (final ConfigurableTable<?> table : ui4Tables.getAllTables()) {

      // ouvinte de seleo na tabela que atualiza as actions a cada
      // elemento selecionado.
      table.getSelectionModel().addListSelectionListener(
        new ListSelectionListener() {
          @Override
          public void valueChanged(ListSelectionEvent e) {
            updateActions();
          }
        });

      // ouvinte de mouse na tabela que permite a seleo de vrias
      // linhas de tabelas diferentes apenas se a tecla CONTROL estiver 
      // pressionada.
      table.addMouseListener(new MouseAdapter() {
        @Override
        public void mouseClicked(MouseEvent e) {
          if (!e.isControlDown()) {
            for (ConfigurableTable<?> otherTable : ui4Tables.getAllTables()) {
              if (!otherTable.getId().equals(table.getId())) {
                otherTable.clearSelection();
              }
            }
          }

          if (e.getClickCount() == 2) {
            detailsSGAClusterAction.actionPerformed(null);
          }
        }
      });

    }

    // verificar se a UI  um painel de abas, se for ento adicionar um 
    // ouvinte pra atualizar as actions sempre que mudar o foco da aba.
    if (ui4Tables instanceof TabbedPane4Tables) {
      // adiciona listener de seleo no painel de abas.
      ((TabbedPane4Tables) ui4Tables).addChangeListener(new ChangeListener() {
        @Override
        public void stateChanged(ChangeEvent e) {
          updateActions();
        }
      });
    }
  }

  /**
   * Como as tabelas so atualizadas constantementes, para manter a seleo das
   * linhas de cada tabela  necessrio definirmos o objeto que auxilia essa
   * operao.
   */
  private void addTableSelectionObject() {
    for (final ConfigurableTable<SGASet> table : ui4Tables
      .getAllTables(SGASet.class)) {

      table.setRowToKey(new RowToKey<SGASet>() {
        @Override
        public String getKey(SGASet row) {
          return row.getName();
        }
      });

    }
  }

  /**
   * Mtodo auxiliar que cria as aes usadas na aplicao.
   */
  private void buildActions() {
    // Aes da barra de ferramentas e do menu
    detailsSGAClusterAction = new DetailsSGAClusterAction(this);
    detailsSGAClusterAction.setEnabled(false);

    restartSGAClusterAction = new RestartSGAClusterAction(this);
    restartSGAClusterAction.setEnabled(false);

    stopSGAClusterAction = new StopSGAClusterAction(this);
    stopSGAClusterAction.setEnabled(false);
  }

  /**
   * Constri a interface que exibe as tabelas de monitorao de SGA's e
   * Clusters.
   */
  private void buildInterface() {
    final ApplicationFrame appFrame = getApplicationFrame();
    appFrame.setLayout(new BorderLayout());
    appFrame.setJMenuBar(createMenuBar());

    // adicionando e posicionando barra de ferramentas
    final Container appContentPane = appFrame.getContentPane();
    appContentPane.add(createToolBar(), BorderLayout.NORTH);

    appContentPane.add((Component) ui4Tables, BorderLayout.CENTER);
    appFrame.pack();
  }

  /**
   * Carrega as preferncias do usurio de cada tabela.
   */
  private void loadTablePreferences() {
    PreferenceCategory appPrefs = getPreferences();
    PVTables pv = (PVTables) appPrefs.getPreference(SGAMonitorPref.MAIN_TABLES);

    pv.loadTables(ui4Tables.getAllTables());
  }

  /**
   * Atualiza os dados mostrados nas tabelas.
   * 
   * @param data Novos dados.
   */
  private void setData(List<SGASet> data) {
    List<ConfigurableTable<SGASet>> tables =
      ui4Tables.getAllTables(SGASet.class);

    for (ConfigurableTable<SGASet> table : tables) {
      table.updateRows(data);
    }

    StatusBar st = getApplicationFrame().getStatusBar();
    String timeText = Utilities.getFormattedDate((new Date()).getTime());
    st.setInfo(getString("console.last.update") + timeText);

    setChanged();
    notifyObservers(data);
  }

  /**
   * Lista com todos os objetos do tipo SGASet.
   * 
   * @return lista com todos os objetos do tipo SGASet.
   * @throws RemoteException
   */
  private List<SGASet> getSGASetList() throws RemoteException {
    SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    List<String> sgaNames = sgaService.getAllSGANames();

    final List<SGASet> newRows = new ArrayList<SGASet>();

    for (int i = 0; i < sgaNames.size(); i++) {
      String name = sgaNames.get(i);
      SGASet sga = sgaService.getSGASet(name);
      newRows.add(sga);
    }

    return newRows;
  }

  /**
   * Verifica se  o administrador ou, caso contrrio, se o usurio tem
   * permisso de administrao sobre o servio de SGA do servidor corrente.
   * 
   * @return true se tem permisso ou false caso contrrio
   */
  private boolean checkSGAAdministrationPermission() {
    User user = User.getLoggedUser();

    if (user.isAdmin()) {
      return true;
    }

    String serverName =
      AdminPermission.LOCAL + Client.getInstance().getSystemName();

    Permission p = null;

    try {
      p =
        user.getMatchAttributesPermission(SGAAdminPermission.class, serverName);
    }
    catch (Exception e) {
    }
    return p != null;
  }
}
