package csbase.client.rest;

import csbase.client.Client;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.NotificationPanel;
import csbase.client.desktop.RemoteTask;
import csbase.client.login.LoginInterface;
import csbase.client.login.PreLogin;
import csbase.logic.SecureKey;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.RestServiceInterface;
import io.swagger.jaxrs.config.BeanConfig;
import org.glassfish.grizzly.http.server.CLStaticHttpHandler;
import org.glassfish.grizzly.http.server.HttpServer;
import org.glassfish.jersey.grizzly2.httpserver.GrizzlyHttpServerFactory;
import org.glassfish.jersey.jackson.JacksonFeature;
import org.glassfish.jersey.server.ResourceConfig;
import org.glassfish.jersey.server.spi.Container;
import org.glassfish.jersey.server.spi.ContainerLifecycleListener;
import org.glassfish.jersey.servlet.ServletContainer;
import tecgraf.javautils.core.lng.LNG;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.URI;
import java.rmi.RemoteException;
import java.security.DigestException;
import java.util.*;

/**
 * Controlador REST para o cliente.
 */
public class RestController {
  /**
   * Singleton
   */
  private static RestController instance;

  /**
   * Configurao de resources oferecidos.
   */
  private ResourceConfig resourceConfig;

  /**
   * Servidor HTTP
   */
  private HttpServer httpServer;

  /**
   * Container Jersey para atualizao dinmica de recursos
   */
  private Container container;

  /**
   * Token
   */
  private String token;

  /**
   * Usurio
   */
  private final LoginInterface userLogin;

  /**
   * Indicativo interno de developer mode (no demanda token)
   */
  private boolean developMode = false;

  /**
   * Pacotes de recursos da API rest
   */
  private final List<String> resourcePackages;

  /**
   * Recursos de instancias de aplicao
   */
  private final Map<String,List<Object>> appResources;

  /**
   * Diretrio de recursos para interface web da documentao
   */
  private String docFrontEndDirectory;
  private BeanConfig beanConfig;

  /**
   * Define o diretrio de recursos para interface web da documentao
   */
  public void setDocFrontEndDirectory(String docFrontEndDirectory) {
    if(httpServer!=null && httpServer.isStarted()) {
      throw new IllegalStateException("Server is already started");
    }
    this.docFrontEndDirectory = docFrontEndDirectory;
  }

  /**
   * Pporta do servidor HTTP
   */
  private int httpPort;

  /**
   * Mtodo para recuperar a instncia do singleton do controlador REST.
   *
   * @return A instncia nica do controlador REST
   */
  public static RestController getInstance() {
    if (instance == null) {
      instance = new RestController();
    }
    return instance;
  }

  /**
   * Host padro para acesso.
   */
  private static final String DEFAULT_HOST = "127.0.0.1";

  /**
   * Porta padro do servidor REST
   */
  private static final int DEFAULT_PORT = 15000;

  /**
   * Determina uma porta TCP livre no host local
   *
   * @return Uma porta TCP que no est sendo usada
   */
  private static int getAvailablePort() {
    ServerSocket socket = null;
    int port = -1;
    try {
      socket = new ServerSocket(0);
      port = socket.getLocalPort();
    }
    catch (IOException ex) {
    }
    finally {
      if (socket != null) {
        try {
          socket.close();
        }
        catch (IOException e) {
          throw new RuntimeException("You should handle this error.", e);
        }
      }
    }

    return port;
  }

  /**
   * Construtor privado do controlador REST, somente acessvel via getInstance
   */
  private RestController() {
    this.userLogin = Client.getInstance().getLoginObject();

    //pacotes de recursos REST do cliente
    this.resourcePackages = new ArrayList<>();
    resourcePackages.add("csbase.client.rest.resources");

    //objetos de recursos de aplicaes
    appResources=new HashMap<>();

    //Diretorio padro do frontend swagger para documentao
    this.docFrontEndDirectory ="csbase/client/rest/swagger-ui/";

    if (userLogin instanceof PreLogin) {
      this.token = ((PreLogin) userLogin).getClientInstanceId();
      if (this.token.startsWith("TOKENID:")) {
        this.token = this.token.substring("TOKENID:".length());
      }
    }
    else {
      SecureKey sessionKey = new SecureKey();
      try {
        this.token = sessionKey.digest();
      }
      catch (DigestException e) {
        e.printStackTrace();
        this.token = userLogin.getClientInstanceId();
      }
    }
  }

  /**
   * Recupera o token de autenticao do usurio.
   *
   * @return O token de login do usurio corrente.
   */
  public String getAuthToken() {
    return token;
  }

  /**
   * Verifica se o servidor est em modo Developer.
   *
   * @return Retorna true no caso de estar no modo de desenvolvimento, ou false
   *         caso contrario
   */
  public boolean isDeveloperMode() {
    return developMode;
  }

  /**
   * Verifica se um token de usurio  vlido. No caso de estar no modo de
   * desenvolvimento, retorna true independentemente do token.
   *
   * @param checkToken O token a ser verificado
   * @return true se o token passado  vlido, ou false caso contrario
   */
  public boolean isTokenAuthorized(String checkToken) {
    return isDeveloperMode() || this.token.equals(checkToken);
  }

  /**
   * Recupera o login do usuario corrente.
   *
   * @return a interface de login
   */
  public LoginInterface getUserLogin() {
    return userLogin;
  }

  /**
   * Encerra o servidor REST.
   */
  public void stopServer() {
    if (httpServer != null && httpServer.isStarted()) {
      httpServer.shutdownNow();

      RemoteTask<Void> task = new RemoteTask<Void>() {
        @Override
        protected void performTask() throws Exception {
          ClientRemoteLocator.restService.unregisterClientHttpServer(token);
        }
      };
      final String msg = LNG.get("csbase.client.rest.server.unregistering");
      task.execute(DesktopFrame.getInstance().getDesktopFrame(), null, msg);

      if (task.getError() != null) {
        task.getError().printStackTrace();
      }
    }
  }

  /**
   * Inicia o servidor REST na porta e host padro.
   */
  public void startServer() {
    try {
      startServer(DEFAULT_HOST, DEFAULT_PORT);
    }
    catch (javax.ws.rs.ProcessingException ex) {
      startServer(DEFAULT_HOST, getAvailablePort());
    }
  }

  /**
   * Inicia o servidor REST no host padro em uma porta especfica.
   *
   * @param port porta a ser usada pelo servidor
   */
  public void startServer(int port) {
    startServer(DEFAULT_HOST, port);
  }

  /**
   * Inicia o servidor REST em uma porta e host especficos.
   *
   * @param host Host a ser usado pelo servidor.
   * @param port porta a ser usada pelo servidor
   */
  public void startServer(String host, int port) {
    this.httpPort = port;

    this.beanConfig = new BeanConfig();
    beanConfig.setVersion("1.0");
    beanConfig.setDescription(LNG.get("csbase.client.rest.server.description"));
    beanConfig.setBasePath("/");
    beanConfig.setResourcePackage(String.join(",", resourcePackages));
    beanConfig.setScan(true);

    this.resourceConfig=buildResourceConfig();

    this.resourceConfig.registerInstances(new ContainerLifecycleListener() {
      @Override
      public void onStartup(Container container) {
        RestController.this.container=container;
        System.out.println("Container recuperado");
      }

      @Override
      public void onReload(Container container) {
        System.out.println("RELOADED");
      }

      @Override
      public void onShutdown(Container container) {

      }
    });


    final String baseURL = "http://" + host + ":" + port + "/";

    this.httpServer = GrizzlyHttpServerFactory.createHttpServer(URI.create(
      baseURL), this.resourceConfig);

    //Registro de UI do swagger
    CLStaticHttpHandler staticHttpHandler = new CLStaticHttpHandler(RestController.class.getClassLoader(),
            this.docFrontEndDirectory);
    httpServer.getServerConfiguration().addHttpHandler(staticHttpHandler, "/docs/");

    NotificationPanel notificationPanel = DesktopFrame.getInstance()
      .getNotificationPanel();

    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        final RestServiceInterface restService =
          ClientRemoteLocator.restService;
        restService.registerClientHttpServer(token, baseURL);
        boolean devMode = restService.isClientDeveloperMode();
        setResult(devMode);
      }
    };

    final String taskMsg = LNG.get("csbase.client.rest.server.registering");
    task.execute(DesktopFrame.getInstance().getDesktopFrame(), null, taskMsg);

    final Exception error = task.getError();
    if (error == null) {
      this.developMode = task.getResult();
    }
    else {
      final String errorMsg = LNG.get("csbase.client.rest.server.start.error")
        + " - " + error.getLocalizedMessage();
      notificationPanel.addNotificationLine(notificationPanel
        .makeNotificationLine(new Date(), "ClientRestService", errorMsg, true));
    }

    final String logMsg = LNG.get("csbase.client.rest.server.started") + " - "
      + port + (isDeveloperMode() ? " (dev mode)" : "");
    notificationPanel.addNotificationLine(notificationPanel
      .makeNotificationLine(new Date(), "ClientRestService", logMsg, true));

    final String apiMsg = LNG.get("csbase.client.rest.server.doc.notification")
      + " " + baseURL + "docs/";
    notificationPanel.addNotificationLine(notificationPanel
      .makeNotificationLine(new Date(), "ClientRestService", apiMsg, true));
    logMessage("RestService", "Sevidor iniciado na porta " + port);
  }

  /**
   * Metodo para registro de recursos no servidor REST.
   *
   * @param classes Lista de classes de recursos a serem registradas.
   */
  public void registerService(Class<?>[] classes) {
    resourceConfig.registerClasses(classes);
  }

  /**
   * Adiciona uma lista de pacotes a lista de
   * recursos a serem registrados no servidor REST.
   *
   * @param packages Lista de pacotes com resursos REST a serem adicionados
   */
  public void addResourcePackages(String... packages) {
    resourcePackages.addAll(Arrays.asList(packages));

    beanConfig.setResourcePackage(String.join(",", resourcePackages));
    beanConfig.setScan();

    if (container!=null) {
      this.resourceConfig=buildResourceConfig();
      container.reload(this.resourceConfig);
    }
  }

    /**
   * Adiciona uma lista de instncias de recursos a serem registrados no servidor REST.
   *
   * @param appId O id da aplicao
   * @param resources Lista de pacotes com resursos REST a serem adicionados
   */
  public void addApplicationResources(String appId, List<Object> resources) {
    appResources.put(appId,resources);

    if (container!=null) {
      this.resourceConfig=buildResourceConfig();
      container.reload(this.resourceConfig);
    }
  }


  /**
   * Remove uma lista de recursos registrados no servidor REST.
   *
   * @param appId O id da aplicao para retirada de recursos
   */
  public void removeApplicationResources(String appId) {
    appResources.remove(appId);

    if (container!=null) {
      this.resourceConfig=buildResourceConfig();
      container.reload(this.resourceConfig);
    }
  }


  /**
   * Consulta a porta REST usada pelo cliente.
   *
   * @return porta.
   */
  public int getPort() {
    if (httpServer == null || !httpServer.isStarted()) {
      return -999;
    }
    return httpPort;
  }

  /**
   * Metodo para registro de pacotes com recursos no servidor REST.
   *
   * @param packages Lista de packages de recursos a serem registradas.
   */
  public void registerPackages(String... packages) {
    if (httpServer != null && httpServer.isStarted()) {
      httpServer.shutdownNow();
    }
    resourcePackages.addAll(Arrays.asList(packages));
    this.startServer();
  }

  private void logMessage(String type, String msg) {
    try {
      ClientRemoteLocator.eventLogService.addClientInformation(new String[] {
          "ClientRest" }, new String[] { type, msg });
    }
    catch (RemoteException e) {
      e.printStackTrace();
    }
  }

  /**
   * Cria a configurao de recursos de acordo com os recursos atuais definidos
   * @return Uma nova configurao de recursos
   */
  private ResourceConfig buildResourceConfig() {
    final ResourceConfig rc = new ResourceConfig();

    //Registro dos recursos REST do cliente
    rc.packages(true, resourcePackages.toArray(
            new String[resourcePackages.size()]));

    //Registro dos recursos REST de aplicaes
    appResources.entrySet().stream().forEach(e->rc.registerInstances(e.getValue().toArray()));

    //Registro de recurso para serializao JSON
    rc.register(JacksonFeature.class);

    //Registro de recursos para tratamento de requisies e respostas
    rc.register(AuthFilter.class);
    rc.register(DebugMapper.class);

    //Registro de classes do swagger
    rc.register(
            io.swagger.jersey.listing.ApiListingResourceJSON.class);
    rc.register(io.swagger.jaxrs.listing.ApiListingResource.class);
    rc.register(io.swagger.jaxrs.listing.SwaggerSerializers.class);



    return rc;
  }

}
