package csbase.client.util.gui;

import java.awt.Component;
import java.util.Arrays;
import java.util.List;

import javax.swing.JSplitPane;

/**
 * <p>
 * Cria um painel de componentes separados por divisores que permitem abrir ou
 * fechar a visualizao de um componente.
 * </p>
 * <p>
 * Esse componente cria um {@link JSplitPane}, adiciona em seu elemento
 * top_left o primeiro componente e em seu elemento bottom_right uma nova
 * instncia de {@link JSplitPane} e assim vai cascateando essas instncias at
 * que insira o ltimo componente.<br>
 * Assim, o divisor fecha sempre um componente pra cima ou ento todos os
 * componentes a baixo. Ele considera que o componente mais importante vai estar
 * sempre mais a cima ou a esquerda, dependendo da orientao.
 * </p>
 * 
 * @author Tecgraf
 */
public class JMultiSplitPane extends JSplitPane {

  /**
   * Orientao indicando que os componentes devem estar um em cima do outro.
   */
  public static final int VERTICAL_SPLIT = JSplitPane.VERTICAL_SPLIT;
  /**
   * Orientao indicando que os componentes devem estar um ao lado do outro.
   */
  public static final int HORIZONTAL_SPLIT = JSplitPane.HORIZONTAL_SPLIT;

  /**
   * Construtor.
   * 
   * @param orientation Indica se os componentes devem estar um em cima do outro -
   *        {@link #VERTICAL_SPLIT} - ou um ao lado do outro -
   *        {@link #HORIZONTAL_SPLIT} -.
   * @param continuousLayout Indica se os componentes deve mser continuamente
   *        redesenhados conforme a posio do divisor  atualizada.
   * @param cs Componentes presentes no painel. Deve conter no mnimo 2
   *        componentes.
   */
  public JMultiSplitPane(int orientation, boolean continuousLayout,
    Component... cs) {
    this(orientation, continuousLayout, Arrays.asList(cs), 0);
  }

  /**
   * Construtor.
   * 
   * @param orientation Indica se os componentes devem estar um em cima do outro -
   *        {@link #VERTICAL_SPLIT} - ou um ao lado do outro -
   *        {@link #HORIZONTAL_SPLIT} -.
   * @param continuousLayout Indica se os componentes deve mser continuamente
   *        redesenhados conforme a posio do divisor  atualizada.
   * @param cs Componentes presentes no painel. Deve conter no mnimo 2
   *        componentes.
   */
  public JMultiSplitPane(int orientation, boolean continuousLayout,
    List<Component> cs) {
    this(orientation, continuousLayout, cs, 0);
  }

  /**
   * Construtor.
   * 
   * @param orientation Indica se os componentes devem estar um em cima do outro -
   *        {@link #VERTICAL_SPLIT} - ou um ao lado do outro -
   *        {@link #HORIZONTAL_SPLIT} -.
   * @param continuousLayout Indica se os componentes deve mser continuamente
   *        redesenhados conforme a posio do divisor  atualizada.
   * @param cs Componentes presentes no painel. Deve conter no mnimo 2
   *        componentes.
   * @param offset ndice do primeiro componente da lista a ser inserido no
   *        painel. Componentes antes dele sero ignorados.
   */
  public JMultiSplitPane(int orientation, boolean continuousLayout,
    List<Component> cs, int offset) {
    super(orientation, continuousLayout, null, null);

    if (cs.size() < 2) {
      throw new IllegalArgumentException(
        "cs deve conter ao menos dois componentes.");
    }

    addComponents(orientation, continuousLayout, cs, offset);
  }

  /**
   * Adiciona os componentes no painel cascateando instncias de
   * {@link JSplitPane}.
   * 
   * @param orientation Indica se os componentes devem estar um em cima do outro -
   *        {@link #VERTICAL_SPLIT} - ou um do lado do outro -
   *        {@link #HORIZONTAL_SPLIT} -.
   * @param continuousLayout Indica se os componentes deve mser continuamente
   *        redesenhados conforme a posio do divisor  atualizada.
   * @param cs Componentes presentes no painel. Deve conter no mnimo 2
   *        componentes.
   * @param offset ndice do primeiro componente da lista a ser inserido no
   *        painel. Componentes antes dele sero ignorados.
   */
  private void addComponents(int orientation, boolean continuousLayout,
    List<Component> cs, int offset) {

    setLeftComponent(cs.get(offset));
    int next = offset + 1;
    if (next == cs.size() - 1) {
      setRightComponent(cs.get(next));
    }
    else {
      JMultiSplitPane nextPane =
        new JMultiSplitPane(orientation, continuousLayout, cs, next);
      setRightComponent(nextPane);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setContinuousLayout(boolean newContinuousLayout) {
    super.setContinuousLayout(newContinuousLayout);
    for (Component c : getComponents()) {
      if (c instanceof JMultiSplitPane) {
        JSplitPane.class.cast(c).setContinuousLayout(newContinuousLayout);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setDividerSize(int newSize) {
    super.setDividerSize(newSize);
    for (Component c : getComponents()) {
      if (c instanceof JMultiSplitPane) {
        JSplitPane.class.cast(c).setDividerSize(newSize);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setOneTouchExpandable(boolean newValue) {
    super.setOneTouchExpandable(newValue);
    for (Component c : getComponents()) {
      if (c instanceof JMultiSplitPane) {
        JSplitPane.class.cast(c).setOneTouchExpandable(newValue);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setOrientation(int orientation) {
    super.setOrientation(orientation);
    for (Component c : getComponents()) {
      if (c instanceof JMultiSplitPane) {
        JSplitPane.class.cast(c).setOrientation(orientation);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setResizeWeight(double value) {
    super.setResizeWeight(value);
    for (Component c : getComponents()) {
      if (c instanceof JMultiSplitPane) {
        JSplitPane.class.cast(c).setResizeWeight(value);
      }
    }
  }
}
