package csbase.client.algorithms.validation;

import java.util.ArrayList;
import java.util.List;

import csbase.logic.algorithms.validation.FlowNodeValidationResult;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import tecgraf.javautils.core.lng.LNG;

/**
 * Classe que auxilia a traduo de mensagens internacionalizveis.
 */
public class ValidationTranslator {

  /**
   * Traduz a mensagem da validao.
   * 
   * @param validation A validao.
   * @return A mensagem traduzida.
   */
  public static String translateMessage(Validation validation) {
    List<StatusMessage> messages = recursiveTranslateMessage(validation, null);
    StringBuilder builder = new StringBuilder();
    for (StatusMessage message : messages) {
      builder.append(message);
      builder.append("\n");
    }
    return builder.toString();
  }

  public static class StatusMessage {
    public String message;
    public boolean status;

    public StatusMessage(boolean status, String message) {
      super();
      this.message = message;
      this.status = status;
    }
  }

  /**
   * Mtodo chamado recursivamente para traduo da mensagem
   * 
   * @param validation A validao.
   * @param prefix Prefixo do elemento que deve ser includo antes da mensagem
   * @return A mensagem traduzida.
   */
  public static List<StatusMessage> recursiveTranslateMessage(
    Validation validation, String prefix) {

    List<StatusMessage> messages = new ArrayList<>();

    String newPrefix = prefix;

    if (validation instanceof FlowNodeValidationResult) {
      newPrefix = getElementName((FlowNodeValidationResult) validation);
    }

    if (validation.getChildren() != null) {
      for (Validation resultChild : validation.getChildren()) {
        messages.addAll(recursiveTranslateMessage(resultChild, newPrefix));
      }
    }
    // Como o getMessage() j  recursivo o else  obrigatrio pra no mostrar as mesmas mensagens 2 vezes
    else if (validation.getMessage() != null && validation.getMessage()
      .size() > 0) {

      messages.add(new StatusMessage(validation.isWellSucceeded(), getMessage(
        validation.getMessage(), newPrefix)));
    }

    return messages;
  }

  /**
   * Retorna o nome do elemento da validao se for FlowNodeValidationResult
   * 
   * @param validation a validacao.
   * @return o nome do elemento
   */
  public static String getElementName(FlowNodeValidationResult validation) {
    if (validation != null) {
      if (validation.getElement() != null) {
        return "(" + validation.getElement().getAlgorithmTitle() + ") ";
      }
    }
    return null;
  }

  /**
   * Indica se existe uma mensagem internacionalizada definida para a chave
   * definida.
   * 
   * @param message mensagem internacionalizvel.
   * @return verdadeiro se existe uma mensagem ou falso, caso contrrio.
   */
  public static boolean hasLocalizedMessage(LocalizedMessage message) {
    if (message != null) {
      return LNG.hasKey(message.getKey());
    }
    else {
      return false;
    }
  }

  /**
   * Obtm a mensagem j localizada.
   * 
   * @param messages Lista de mensagens internacionalizveis.
   * @param prefix Prefixo a ser includo antes de cada mensagem.
   * @return A string concatenando as mensagens.
   */
  public static String getMessage(List<LocalizedMessage> messages,
    String prefix) {

    StringBuilder builder = new StringBuilder();

    for (LocalizedMessage message : messages) {

      if (prefix != null) {

        builder.append(prefix);
      }
      if (hasLocalizedMessage(message)) {
        String string = getLocalizedMessage(message);
        string.replaceAll("\n", "\n\t");
        builder.append(string + "\n");
      }
      else {
        builder.append(message.toString() + "\n");
      }
    }

    return builder.toString();
  }

  /**
   * Obtm a mensagem j localizada.
   * 
   * @param message mensagem internacionalizvel.
   * @return a mensagem.
   */
  public static String getLocalizedMessage(LocalizedMessage message) {
    if (message != null) {
      Object[] args = message.getArgs();
      String key = message.getKey();
      if (args != null && args.length > 0) {
        return LNG.get(key, args);
      }
      else {
        return LNG.get(key);
      }
    }
    else {
      return "";
    }
  }
}
