package csbase.client.applications.algorithmsmanager.dialogs;

import javax.swing.JPanel;

import csbase.client.applications.algorithmsmanager.actions.AlgorithmManagementAction;
import csbase.client.applications.algorithmsmanager.actions.CommonManagementAction;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.Category;
import csbase.logic.algorithms.CategorySet;

/**
 * Essa classe representa um painel para alterao de uma algoritmo.
 * 
 * O algoritmo selecionad  inicialmente exibida para consulta e pode ser
 * habilitada para alterao dos dados, atravs de uma ao de Editar. Toda
 * alterao  enviada para o servidor para atualizao.
 * 
 */
public class AlgorithmEditDataPanel extends CommonEditTabbedPanel {
  /** Painel com as informaes bsicas do algoritmo */
  private AlgorithmBasicInfoPanel infoPanel;

  /** Painel com as informaes de categorias do algoritmo */
  private AlgorithmCategoryInfoPanel categoryPanel;

  /** Painel com as informaes de verso do algoritmo */
  private AlgorithmVersionInfoPanel versionPanel;

  /**
   * Algoritmo selecionado para alterao
   */
  private AlgorithmListItem selectedAlgorithm;

  /** Painel com o histrico do algoritmo */
  private AlgorithmHistoryInfoPanel historyPanel;

  /**
   * Constri o painel para alterao de uma categoria selecionada
   * 
   * @param action ao que criou esse painel
   */
  public AlgorithmEditDataPanel(CommonManagementAction action) {
    super(action);
  }

  /**
   * Obtm a ao de criar categoria que criou esse painel.
   * 
   * @return a ao de criar categoria
   */
  public AlgorithmManagementAction getAlgorithmManagementAction() {
    return (AlgorithmManagementAction) getAction();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean confirmDataChanged() {
    if (infoPanel.wasModified() || categoryPanel.wasModified()
      || versionPanel.wasModified()) {
      // Confirma o cancelamento das abas alteradas
      return confirmCancelling();
    }
    return true;
  }

  /**
   * Atribui um novo algoritmo para edio.
   * 
   * @param algoItem algoritmo selecionado
   * 
   */
  public void changeAlgorithmItemEdition(AlgorithmListItem algoItem) {
    selectedAlgorithm = algoItem;
    initializeData();
  }

  /**
   * Obtm o algoritmo selecionado para edio.
   * 
   * @return o algoritmo selecionado para edio
   */
  public AlgorithmListItem getSelectedAlgorithm() {
    return selectedAlgorithm;
  }

  /**
   * Obtm o painel de categorias contido no painel de edio do algoritmo.
   * 
   * @return o painel de categorias
   */
  public AlgorithmCategoryInfoPanel getCategoryPanel() {
    return categoryPanel;
  }

  /**
   * Obtm o painel de histrico contido no painel de edio do algoritmo.
   * 
   * @return o painel de categorias
   */
  public AlgorithmHistoryInfoPanel getHistoryPanel() {
    return historyPanel;
  }

  /**
   * Obtm o painel de informaes bsicas contido no painel de edio do
   * algoritmo.
   * 
   * @return o painel de categorias
   */
  public AlgorithmBasicInfoPanel getBasicInfoPanel() {
    return infoPanel;
  }

  /**
   * Obtm o painel de verses contido no painel de edio do algoritmo.
   * 
   * @return o painel de categorias
   */
  public AlgorithmVersionInfoPanel getVersionPanel() {
    return versionPanel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void buildAdditionalInfoPanel() {
    categoryPanel = new AlgorithmCategoryInfoPanel(this);
    versionPanel = new AlgorithmVersionInfoPanel(this);
    historyPanel = new AlgorithmHistoryInfoPanel(this);
    addInfoPanel(categoryPanel.getTitle(), categoryPanel);
    addInfoPanel(versionPanel.getTitle(), versionPanel);
    addInfoPanel(historyPanel.getTitle(), historyPanel);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JPanel buildBasicInfoPanel() {
    infoPanel = new AlgorithmBasicInfoPanel(this);
    return infoPanel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initializeData() {
    infoPanel.initializeData();
    categoryPanel.initializeData();
    versionPanel.initializeData();
    historyPanel.initializeData();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void handleAlgorithmUpdated(AlgorithmInfo algoInfo) {
    if (selectedAlgorithm != null) {
      this.selectedAlgorithm.setItem(algoInfo);
    }
    infoPanel.initializeData();
    historyPanel.initializeData();
    versionPanel.initializeData();
    categoryPanel.initializeData();
    versionPanel.handleAlgorithmUpdated(algoInfo);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void handleCategoryUpdated(CategorySet modifiedCategorySet) {
    initializeData();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void handleCategoryCreated(Category category) {
    initializeData();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void handleCategoryRemoved(Category category) {
    initializeData();
  }
}
