package csbase.client.applications.algorithmsmanager.report.core;

import java.awt.Color;
import java.awt.Container;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.BorderFactory;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.KeyStroke;
import javax.swing.border.TitledBorder;

import tecgraf.javautils.gui.GBC;
import csdk.v1_0.helper.application.ApplicationImages;

/**
 * Dilogo para selecionar testes, comandar sua execuo e ver o resultado
 *
 *
 * @author Tecgraf/PUC-Rio
 */
public class TestDialog extends BasicTestDialog implements SourceListener {

  /** Testes disponveis */
  List<ITest<?>> tests;
  /** Fbrica de sources em funo do teste */
  ISubjectFactory factory;

  /** Painel de seleo dos testes */
  TestSelectionPanel selectionPanel;
  /** Boto para rodar os testes */
  JButton runButton;
  /** Painel de resultados */
  ReportPanel reportPanel;
  /** Boto para fechar */
  JButton closeButton;
  /** Aviso de update */
  JLabel updateMessage;

  /**
   * Construtor
   *
   * @param tests Testes disponveis
   * @param factory Fbrica de sources em funo do teste
   * @param locale Locale para internacionalizao
   * @param owner Janela que comandou a abertura desta
   */
  public TestDialog(List<ITest<?>> tests, ISubjectFactory factory,
    Locale locale, Window owner) {
    super(locale, owner);

    setTitle(getString("title"));
    setModalityType(ModalityType.MODELESS);

    this.tests = tests;
    this.factory = factory;
    closeButton = new JButton(getString("close.button"));
    closeButton.addActionListener(e -> closeDialog());
    selectionPanel = new TestSelectionPanel(tests);
    selectionPanel.addItemListener(e -> runButton.setEnabled(true));
    reportPanel = new ReportPanel(locale);
    runButton = new JButton(getString("run.button"));
    runButton.addActionListener(e -> runTests());
    updateMessage = new JLabel("<html></html>");

    final Font titleFont = new Font(Font.SANS_SERIF, Font.BOLD, 16);

    JPanel topPanel = new JPanel(new GridBagLayout());
    topPanel.add(selectionPanel, new GBC(0, 0).none().insets(9, 12, 6, 12)
      .west().gridwidth(2));
    topPanel.add(updateMessage, new GBC(0, 1).horizontal().insets(3, 6, 6, 6));

    topPanel.add(runButton, new GBC(1, 1).none().east().insets(6, 6, 6, 6));
    topPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory
      .createEmptyBorder(), getString("tests.title")));
    reportPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory
      .createEmptyBorder(), getString("results.title")));

    ((TitledBorder) topPanel.getBorder()).setTitleFont(titleFont);
    ((TitledBorder) reportPanel.getBorder()).setTitleFont(titleFont);

    Container container = getContentPane();
    container.setLayout(new GridBagLayout());
    container.add(topPanel, new GBC(0, 0).horizontal().insets(12, 12, 5, 12));
    container.add(new JSeparator(), new GBC(0, 1).horizontal().insets(12, 12,
      5, 12));
    container.add(reportPanel, new GBC(0, 2).both().insets(5, 12, 5, 12));
    container.add(closeButton, new GBC(0, 3).none().east()
      .insets(5, 12, 12, 12));

    pack();
    addEscListener();
    setLocationRelativeTo(owner);
    factory.addListener(this);
  }

  /**
   * Informa o usurio que h novas fontes
   */
  private void writeUpdateMessage() {

    String text = getString("new.sources.label");

    updateMessage.setForeground(Color.RED);
    updateMessage.setText(text);
    updateMessage.setIcon(ApplicationImages.ICON_WARNING_16);
    runButton.setEnabled(true);
  }

  /**
   * Limpa a informao de que h novas fontes
   */
  private void clearUpdateMessage() {
    updateMessage.setText("<html></html>");
    updateMessage.setIcon(null);
    runButton.setEnabled(false);
  }

  /** Roda os testes e atualiza a rea de resultados */
  private void runTests() {

    List<ITest<?>> selectedTests = selectionPanel.getSelection();
    if (selectedTests == null || selectedTests.size() == 0) {
      return;
    }

    TestRunner testRunner =
      new TestRunner(factory, this,
        getString("running.algorithms.tests.title"),
        getString("running.algorithms.tests.message"));

    Map<ITest<?>, ITestResult<?>> map = testRunner.run(selectedTests);

    reportPanel.updateContent(map);
    clearUpdateMessage();
  }

  /**
   * Ajusta a tecla esc para fechar o di?logo.
   */
  private void addEscListener() {
    final AbstractAction cancelAction = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        closeDialog();
      }
    };

    final int mode = JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT;
    final InputMap inputMap = rootPane.getInputMap(mode);
    final ActionMap actionMap = rootPane.getActionMap();
    final int esc = KeyEvent.VK_ESCAPE;
    final KeyStroke cancelStroke = KeyStroke.getKeyStroke(esc, 0);
    final String actionMapKey = cancelStroke.toString();
    inputMap.put(cancelStroke, actionMapKey);
  }

  /**
   * Fecha o dialogo
   */
  private void closeDialog() {
    factory.removeListener(TestDialog.this);
    dispose();
  }

  /**
   * Avisa que as fontes mudaram {@inheritDoc}
   */
  @Override
  public void sourceChanged() {
    writeUpdateMessage();
  }
}
