/*
 * $Id$
 */

package csbase.client.applications.algorithmsmanager.versiontree.actions;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.rmi.RemoteException;
import java.security.AccessControlException;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.ZipEntry;

import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTree;
import csbase.client.desktop.LocalTask;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.externalresources.ExternalResources;
import csbase.client.externalresources.LocalFile;
import csbase.client.externalresources.StandaloneLocalFile;
import csbase.client.externalresources.ZipLocalFile;
import csbase.client.util.ClientUtilities;
import csbase.client.util.SingletonFileChooser;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.CSBaseException;
import csbase.logic.SyncRemoteFileChannel;
import csbase.util.FileSystemUtils;
import csbase.util.Unzip;

/**
 * @author Tecgraf / PUC-Rio
 * 
 *         Ao abstrata de importao de arquivos.
 */
public abstract class AbstractImportFileAction extends
  AbstractVersionTreeNodeAction {

  /**
   * Modo de seleo permitido.
   */
  private int selectionMode;

  /**
   * Flag indicando se deve permitir que mais de um arquivo seja importado.
   */
  private boolean multiSelectionEnabled;

  /**
   * Construtor.
   * 
   * @param tree Fonte da ao.
   * @param name Nome da ao.
   * @param selectionMode Modo de seleo permitido. <br>
   *        Pode ser:
   *        <ul>
   *        <li>{@link JFileChooser#FILES_ONLY}</li>
   *        <li>{@link JFileChooser#DIRECTORIES_ONLY}</li>
   *        <li>{@link JFileChooser#FILES_AND_DIRECTORIES}</li>
   *        </ul>
   * @param multiSelectionEnabled <tt>true</tt> se deve permitir que mais de um
   *        arquivo seja importado.
   */
  public AbstractImportFileAction(VersionTree tree, String name,
    int selectionMode, boolean multiSelectionEnabled) {
    super(tree, name);

    this.selectionMode = selectionMode;
    this.multiSelectionEnabled = multiSelectionEnabled;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionPerformed(ActionEvent e) {
    if (FileSystemUtils.canRead()) {
      fileSystemUpload();
    }
    else {
      appletUpload(); // #TODO A remover quando todos sistemas CSBase forem
      // assinados
    }
  }

  /**
   * Exibe uma pgina para upload de um arquivo de documentao, a partir do
   * disco rgido do usurio para o servidor. Essa funcionalidade s est
   * disponvel via applet.
   */
  protected abstract void appletUpload();

  /**
   * Cria uma tarefa que indique se  permitido que um arquivo em particular
   * seja importado.<br>
   * Caso retorne <tt>null</tt>, a importao ir prosseguir normalmente como se
   * a tarefa tivesse retornado <tt>true</tt>.
   * 
   * @param fileName arquivo a ser importado.
   * 
   * @return uma tarefa que indique se  permitido que um arquivo em particular
   *         seja importado.<br>
   *         Caso retorne <tt>null</tt>, a importao ir prosseguir normalmente
   *         como se a tarefa tivesse retornado <tt>true</tt>.
   */
  protected abstract Task<Boolean> createCheckFileExistenceTask(
    final String fileName);

  /**
   * Prepara o servidor para a importao do arquivo.
   * 
   * @param filePath caminho relativo do arquivo a ser importado.
   * @param isZipLocalFile flag indicando que o arquivo a ser carregado  do
   *        tipo {@link ZipLocalFile}. Este arquivo representa um conjunto de
   *        arquivos compactados em memria e  utilizado no upload de
   *        diretrios. Essa flag  usada para diferenciar um arquivo zip que o
   *        cliente queira carregar, do arquivo zip criado pelo sistema.
   * 
   * @return informaes para a transferncia dados.
   * 
   * @throws RemoteException se ocorrer falha de RMI
   */
  protected abstract RemoteFileChannelInfo prepareUpload(String filePath,
    boolean isZipLocalFile) throws RemoteException;

  /**
   * Obtm o arquivo a ser importado e o diretrio de destino na rvore de
   * algoritmos. Cria ento uma {@link csbase.client.desktop.LocalTask} para
   * importar o arquivo.
   */
  private void fileSystemUpload() {
    final File[] sourceFiles = getSourceFiles();
    if (!validateFiles(sourceFiles)) {
      return;
    }

    final LocalFile[] localFiles = createLocalFiles(sourceFiles);

    final List<RequestTransfer> requests = new ArrayList<>();
    for (LocalFile sourceFile : localFiles) {
      RemoteFileChannelInfo requestInfo = getRequestInfo(sourceFile);
      if (requestInfo != null) {
        requests.add(new RequestTransfer(sourceFile, requestInfo));
      }
    }

    Task<Void> task = new LocalTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        for (RequestTransfer request : requests) {
          request.transfer();
        }
      }

      @Override
      protected void handleError(Exception error) {
        if (error instanceof IOException) {
          error.printStackTrace();
          StandardDialogs.showErrorDialog(getWindow(), getName(),
            LNG.get("algomanager.error.upload_fatal"));
          return;
        }
        super.handleError(error);
      }
    };
    String waitMsg = LNG.get("algomanager.msg.upload_wait");
    task.execute(getWindow(), getName(), waitMsg, false, true);
  }

  /**
   * <p>
   * Obtm um {@link RemoteFileChannelInfo}, isto , um objeto contendo as
   * informaes necessrias para solicitar a transferncia.
   * </p>
   * <p>
   * Verifica primeiro se o arquivo j existe no servidor. Em caso positivo,
   * confirma se o usurio deseja ou no sobrescrever o arquivo remoto.
   * </p>
   * 
   * @param file arquivo local a ser importado.
   * 
   * @return objeto contendo as informaes necessrias para a transferncia.
   */
  private RemoteFileChannelInfo getRequestInfo(LocalFile file) {
    /*
     * Queremos expandir apenas zips que foram criados pelo sistema para
     * facilitar a transferncia de um diretrio.
     */
    final boolean isZipLocalFile = file instanceof ZipLocalFile;
    final String fileName;
    try {
      fileName = file.getName();
    }
    catch (IOException e) {
      StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
        LNG.get("algomanager.error.upload_io"), e);
      return null;
    }

    Task<Boolean> checkFileExistenceTask =
      createCheckFileExistenceTask(fileName);
    Task<RemoteFileChannelInfo> prepareUploadTask =
      new RemoteTask<RemoteFileChannelInfo>() {
        @Override
        protected void performTask() throws Exception {
          setResult(prepareUpload(fileName, isZipLocalFile));
        }
      };

    String waitMsg;
    if (checkFileExistenceTask != null) {
      waitMsg = LNG.get("algomanager.msg.upload_wait");
      if (!checkFileExistenceTask.execute(getWindow(), getName(), waitMsg,
        false, true)) {
        return null;
      }
      if (checkFileExistenceTask.getResult()) {
        if (!confirmOverwrite(fileName)) {
          return null;
        }
      }
    }
    waitMsg = LNG.get("algomanager.msg.upload_wait");
    if (!prepareUploadTask
      .execute(getWindow(), getName(), waitMsg, false, true)) {
      return null;
    }
    return prepareUploadTask.getResult();
  }

  /**
   * Retorna os arquivos locais a serem importados.
   * 
   * @return arquivo de origem.
   */
  private File[] getSourceFiles() {
    // Acesso ao sistema de arquivos permitido: obtm arquivos com o
    // JFileChooser
    JFileChooser chooser = SingletonFileChooser.getInstance();
    chooser.setFileSelectionMode(selectionMode);
    chooser.setMultiSelectionEnabled(multiSelectionEnabled);

    int returnVal = chooser.showOpenDialog(getWindow());
    if (returnVal == JFileChooser.CANCEL_OPTION) {
      return new File[0];
    }

    if (multiSelectionEnabled) {
      return chooser.getSelectedFiles();
    }
    else {
      return new File[] { chooser.getSelectedFile() };
    }
  }

  /**
   * Cria instncias de {@link LocalFile} representando os arquivos no sistema
   * do cliente.
   * 
   * @param files Os arquivos do cliente.
   * 
   * @return instncias de {@link LocalFile} representando os arquivos no
   *         sistema do cliente.
   */
  private LocalFile[] createLocalFiles(File[] files) {
    try {
      // Se tiver algum diretrio, envia tudo como um arquivo zipado.
      for (File file : files) {
        if (file.isDirectory()) {
          try {
            String fileName =
              String.format("%s.zip", file.getParentFile().getName());
            return new LocalFile[] { new ZipLocalFile(fileName, files) };
          }
          catch (OutOfMemoryError e) {
            String fatalErrorMsg =
              LNG.get("algomanager.error.upload_fatal.outOfMemory");
            StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
              fatalErrorMsg, e);
            return new LocalFile[0];
          }
          catch (IOException e) {
            String fatalErrorMsg = LNG.get("algomanager.error.upload_fatal.io");
            StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
              fatalErrorMsg, e);
            return new LocalFile[0];
          }
          catch (Exception e) {
            String fatalErrorMsg = LNG.get("algomanager.error.upload_fatal");
            StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
              fatalErrorMsg, e);
            return new LocalFile[0];
          }
        }
      }

      List<LocalFile> sourceFiles = new ArrayList<>();
      for (File selFile : files) {
        if (selFile == null) {
          continue;
        }
        StandaloneLocalFile standAloneFile = new StandaloneLocalFile(selFile);
        sourceFiles.add(standAloneFile);
      }

      return sourceFiles.toArray(new LocalFile[sourceFiles.size()]);
    }
    catch (AccessControlException ex1) {
      // Acesso ao sistema de arquivos negado (rodando em "sandbox"): obtm
      // arquivos com a API JNLP. No  possvel a importao de diretrios.
      String fatalErrorMsg = LNG.get("algomanager.error.upload_fatal");
      if (!ExternalResources.getInstance().isEnabled()) {
        StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
          fatalErrorMsg);
        return null;
      }
      try {
        return ExternalResources.getInstance().openMultiFileDialog(".", null);
      }
      catch (CSBaseException ex2) {
        StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
          fatalErrorMsg, ex2);
        return null;
      }
      catch (IOException ex3) {
        StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
          LNG.get("algomanager.error.upload_io"), ex3);
        return null;
      }
    }
  }

  /**
   * Exibe um dilogo para confirmar com o usurio se este deseja sobrescrever
   * um arquivo existente no servidor.
   * 
   * @param name nome do arquivo.
   * 
   * @return <code>true</code> se o usurio confirmar, <code>false</code> caso
   *         contrrio.
   */
  private boolean confirmOverwrite(String name) {
    String msg =
      String.format(LNG.get("algomanager.msg.confirm.file_exists"), name);
    int answer = StandardDialogs.showYesNoDialog(getWindow(), getName(), msg);
    return (answer == JOptionPane.YES_OPTION);
  }

  /**
   * Verifica se os arquivos especificados so vlido, isto , se possuem nomes
   * aceitos pelas regras de nomenclatura dos arquivos do CSBase e se realmente
   * existem no sistema local de arquivos.
   * 
   * @param files Os arquivos a serem verificados.
   * 
   * @return <tt>true</tt> se todos os arquivos tem seus nomes vlidos e existem
   *         no cliente.
   */
  private boolean validateFiles(File[] files) {
    return 0 < files.length && validateFileExists(files) && validateZip(files)
      && validateFileName(files);
  }

  /**
   * Valida os arquivos que forem zip.<br>
   * Verifica se os arquivos contidos em cada zip, seguem as regras de
   * nomenclatura dos arquivos do CSBase.
   * 
   * @param files lista de arquivos
   * 
   * @return <tt>true</tt> se no forem encontrados arquivos invlidos dentro
   *         dos zips.
   */
  private boolean validateZip(File[] files) {

    try {
      for (File file : files) {
        if (!file.getName().toLowerCase().endsWith("zip")) {
          break;
        }

        Unzip unzipped = new Unzip(file);
        List<ZipEntry> entries = unzipped.listZipEntries();
        for (ZipEntry entry : entries) {

          // O nome da entrada representa o caminho de um arquivo compactado.
          // O replace abaixo pega o ltimo elemento deste caminho.
          //
          // Por exemplo, se tivessemos um zip com as seguintes entradas:
          // f0.xxx
          // dir1/
          // dir1/f1.xxx
          // dir1/dir2/
          // dir1/dir2/f2.xxx
          // dir1/dir2/dir3/
          // 
          // Aplicando o replace abaixo a cada uma das entradas, teramos:
          // f0.xxx
          // dir1
          // f1.xxx
          // dir2
          // f2.xxx
          // dir3
          String fileName =
            entry.getName().replaceAll("^(.*/)?([^/]+)/?$", "$2");
          if (!ClientUtilities.isValidFileName(fileName)) {
            StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
              LNG.get("UTIL_NAME_CHARACTER_ERROR"));
            return false;
          }
        }
      }
      return true;
    }
    catch (IOException e) {
      StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
        LNG.get("algomanager.error.upload_io"), e);
      return false;
    }
    catch (Exception e) {
      e.printStackTrace();
      StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
        LNG.get("UTIL_NAME_CHARACTER_ERROR"));
      return false;
    }
  }

  /**
   * Valida se o nome de cada arquivo possui possui apenas caracteres aceitos
   * pelas regras de nomenclatura dos arquivos do CSBase.
   *
   * @param files arquivos cujo nomes sero validados
   *
   * @return <tt>true</tt> se o nome de todos os arquivos forem vlidos.
   */
  private boolean validateFileName(File... files) {
    List<String> invalidFileNames = new ArrayList<>();
    fillInvalidFileNameList(invalidFileNames, files);
    if (!invalidFileNames.isEmpty()) {
      StandardErrorDialogs.showErrorDialog(getWindow(), LNG.get(
        "algomanager.error.upload_invalid_file_names"), getScrollPaneMessage(
          invalidFileNames));
      return false;
    }
    return true;
  }

  /**
   * Preenche uma lista com nomes de arquivos invlidos
   *
   * @param fileNames lista com nomes a ser preenchida
   * @param files Arquivos cujo nomes sero verificados
   */
  private void fillInvalidFileNameList(List<String> fileNames, File... files) {
    for (File file : files) {
      if (!ClientUtilities.isValidFileName(file.getName())) {
        fileNames.add(file.getPath());
      }
      // Se for um diretrio, valida o nome dos filhos.
      if (file.isDirectory()) {
        fillInvalidFileNameList(fileNames, file.listFiles());
      }
    }
  }

  /**
   * Retorna um scroll pane com a rea de texto contendo a mensagem de erro.
   *
   * @param invalidFileNames Mensagem de erro para ser exibida
   * @return scroll pane com a rea de texto contendo a mensagem de erro.
   */
  private JPanel getScrollPaneMessage(List<String> invalidFileNames) {
    JLabel errorLabel = new JLabel(LNG.get("UTIL_NAME_CHARACTER_ERROR"));
    JList<String> listOfFiles = new JList<String>(invalidFileNames.toArray(
      new String[0]));
    listOfFiles.setSelectedIndex(0);
    JPanel panel = new JPanel(new GridBagLayout());
    panel.add(errorLabel, new GBC(0, 0).west());
    panel.add(new JPanel(), new GBC(0, 1));
    panel.add(new JScrollPane(listOfFiles), new GBC(0, 2));
    return panel;
  }

  /**
   * Valida a existncia dos arquivos no cliente.
   * 
   * @param files arquivos a serem importados.
   * 
   * @return <tt>true</tt> se todos os arquivos existem no cliente.
   */
  private boolean validateFileExists(File... files) {
    List<File> invalidFiles = new ArrayList<>();
    for (File file : files) {
      if (!file.exists()) {
        invalidFiles.add(file);
      }
    }
    if (0 < invalidFiles.size()) {

      // Formatando uma lista dos arquivos invlidos para mostrar no dilogo de
      // erro.
      StringBuilder invalidFilesList = new StringBuilder();
      for (File file : invalidFiles) {
        invalidFilesList.append(String.format(
          LNG.get("algomanager.error.upload_files_not_found.files.list"),
          file.getName()));
      }
      StandardErrorDialogs.showErrorDialog(getWindow(), getName(), String
        .format(LNG.get("algomanager.error.upload_files_not_found"),
          invalidFilesList.toString()));

      return false;
    }
    else {
      return true;
    }
  }

  /**
   * Modela uma requisio de transferncia.
   */
  static class RequestTransfer {
    /**
     * O arquivo local.
     */
    LocalFile file;

    /**
     * A informao para construo do canal remoto.
     */
    RemoteFileChannelInfo info;

    /**
     * Constri uma requisio de transferncia.
     * 
     * @param file O arquivo local.
     * @param info A informao para construo do canal remoto.
     */
    RequestTransfer(LocalFile file, RemoteFileChannelInfo info) {
      this.file = file;
      this.info = info;
    }

    /**
     * @throws IOException
     */
    void transfer() throws IOException {
      try {
        SyncRemoteFileChannel channel = new SyncRemoteFileChannel(info);
        channel.open(false);
        long size = file.getLength();
        InputStream source = file.getInputStream();
        channel.syncTransferFrom(source, 0, size);
        source.close();
        channel.setSize(size);
        channel.close();
      }
      catch (Exception e) {
        throw new IOException(e.getMessage(), e);
      }
    }
  }
}