/*
 * $Id$
 */

package csbase.client.applications.algorithmsmanager.versiontree.actions;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTree;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.remote.srvproxies.PlatformProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.Platform;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionInfo;

/**
 * @author Tecgraf / PUC-Rio
 * 
 *         Ao de criao de plataforma.
 */
public class AddPlatform extends AbstractVersionTreeNodeAction {

  /** Informaes sobre a verso na qual ser includa uma plataforma. */
  private AlgorithmVersionInfo version;

  /**
   * Construtor.
   * 
   * @param tree rvore que detm o n fonte da ao.
   * @param version Verso que receber a nova plataforma.
   */
  public AddPlatform(VersionTree tree, AlgorithmVersionInfo version) {
    super(tree, LNG.get("algomanager.menu.platform.add"));

    this.version = version;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionPerformed(ActionEvent e) {
    new AddPlatformDialog().show();
  }

  private class AddPlatformDialog {

    /** Janela principal */
    private DesktopComponentDialog mainDialog;
    private String dialogTitle;
    private String includedPlatform;

    /** Plataformas ainda no cadastradas */
    Vector<String> candidatePlatforms;

    /**
     * Cria o dilogo.
     */
    public AddPlatformDialog() {
      dialogTitle =
        MessageFormat.format(LNG.get("algomanager.title.include_platform"),
          new Object[] { version.getInfo().getId(), version.getId() });
      Vector<String> platforms = version.getSupportedPlatforms();
      Collections.sort(platforms);
      candidatePlatforms = getCandidatePlatforms(platforms);
    }

    /**
     * Obtm as plataformas ainda no includas na verso.
     * 
     * @param platforms plataformas j includas
     * 
     * @return plataformas candidatas  incluso.
     */
    private Vector<String> getCandidatePlatforms(Vector<String> platforms) {
      String taskTitle = LNG.get("algomanager.title");
      Vector<Platform> allPlatforms =
        PlatformProxy.getAllPlatforms(mainDialog, taskTitle,
          LNG.get("algomanager.msg.waiting_platforms"));
      if (allPlatforms.size() == 0) {
        StandardErrorDialogs.showErrorDialog(mainDialog, dialogTitle,
          LNG.get("algomanager.error.no_platforms"));
        return null;
      }
      Vector<String> candidates = new Vector<>();
      for (int i = 0; i < allPlatforms.size(); i++) {
        Platform plat = allPlatforms.get(i);
        String platName = (String) plat.getId();
        if (!platforms.contains(platName)) {
          candidates.add(platName);
        }
      }
      Collections.sort(candidates);
      return candidates;
    }

    /**
     * Mostra um dilogo de incluso de plataforma.
     * 
     * @return plataforma includa ou null se operao cancelada
     */
    public String show() {
      if (candidatePlatforms == null) {
        return null;
      }
      if (candidatePlatforms.size() == 0) {
        StandardErrorDialogs.showErrorDialog(getWindow(), dialogTitle,
          LNG.get("algomanager.error.no_more_platforms"));
        return null;
      }
      final String includeTitle =
        MessageFormat.format(dialogTitle, new Object[] {
            version.getInfo().getName(), version.getId() });
      final DesktopComponentDialog dialog =
        new DesktopComponentDialog(getWindow());
      dialog.setTitle(includeTitle);
      dialog.addWindowListener(new WindowAdapter() {
        @Override
        public void windowClosing(WindowEvent e) {
          dialog.close();
        }
      });

      final JComboBox platCombo = new JComboBox(candidatePlatforms);
      JComponent[][] info =
        new JComponent[][] { {
            new JLabel(LNG.get("algomanager.label.platform")), platCombo } };
      JPanel infoPanel = GUIUtils.createBasicGridPanel(info);
      JPanel buttonsPanel = new JPanel();
      JButton includeButton =
        new JButton(LNG.get("algomanager.button.include"));
      includeButton.addActionListener(e -> {
        String platform = (String) platCombo.getSelectedItem();
        if (addPlatform(platform, dialog, includeTitle)) {
          includedPlatform = platform;
        }
        else {
          includedPlatform = null;
        }
        dialog.close();
      });

      JButton cancelButton = new JButton(LNG.get("algomanager.button.cancel"));
      cancelButton.addActionListener(e -> {
        includedPlatform = null;
        dialog.close();
      });
      ClientUtilities.adjustEqualSizes(includeButton, cancelButton);
      buttonsPanel.add(includeButton);
      buttonsPanel.add(cancelButton);
      Container cp = dialog.getContentPane();
      cp.add(infoPanel, BorderLayout.CENTER);
      cp.add(buttonsPanel, BorderLayout.SOUTH);
      dialog.pack();
      dialog.center(getWindow());
      dialog.setVisible(true);
      return includedPlatform;
    }

    /**
     * Inclui uma plataforma para uma verso de algoritmo.
     * 
     * @param platform plataforma a incluir
     * @param dialog .
     * @param dialogTitle .
     * 
     * @return true se a incluso teve sucesso, false caso contrrio
     */
    private boolean addPlatform(String platform, DesktopComponentDialog dialog,
      String dialogTitle) {
      AlgorithmInfo algo =
        AlgorithmManagementProxy.includePlatform(version.getInfo().getId(),
          version.getId(), platform, dialog);
      if (algo == null) {
        StandardErrorDialogs.showErrorDialog(dialog,
          LNG.get("algomanager.title.error") + " - " + dialogTitle,
          MessageFormat.format(LNG.get("algomanager.error.platform_include"),
            new Object[] { platform }));
        return false;
      }
      return true;
    }
  }
}
