package csbase.client.applications.diskmonitor;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.rmi.RemoteException;
import java.util.Collections;
import java.util.List;

import javax.swing.ButtonGroup;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;

import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationAboutAction;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.diskmonitor.actions.DiskMonitorChangeAreaAction;
import csbase.client.applications.diskmonitor.actions.DiskMonitorRefreshAction;
import csbase.client.applications.diskmonitor.panels.DiskMonitorPieChartPanel;
import csbase.client.remote.srvproxies.DiskUsageProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.diskusageservice.DiskOccupation;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.DiskUsageServiceInterface;

/**
 * Aplicativo de monitorao de disco.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DiskMonitor extends Application {

  /**
   * Thread de atualizaco.
   */
  final private DiskMonitorThread thread;

  /**
   * Painel do grfico de pizza.
   */
  final private DiskMonitorPieChartPanel pieChartPanel;

  /**
   * Identificadores das reas existentes.
   */
  final private List<String> areasIds;

  /**
   * Id da rea corrente em visualizao.
   */
  private String currentAreaId;

  /**
   * Construtor
   * 
   * @param id identificador da aplicao.
   */
  public DiskMonitor(String id) {
    super(id);

    this.pieChartPanel = new DiskMonitorPieChartPanel(this);
    this.thread = new DiskMonitorThread(this);
    this.areasIds = DiskUsageProxy.getAllAreasIds();
    Collections.unmodifiableList(areasIds);
    this.currentAreaId = areasIds.get(0);

    final JFrame mainFrame = getApplicationFrame();
    mainFrame.setJMenuBar(buildMenuBar());
    mountMainPanel(mainFrame.getContentPane());
    mainFrame.pack();

    thread.start();
  }

  /**
   * Constroi o menu de help
   * 
   * @return o menu
   */
  private JMenu buildHelpMenu() {
    final JMenu helpMenu = new JMenu(getMenuString("help"));
    helpMenu.add(new ApplicationAboutAction(this));
    return helpMenu;
  }

  /**
   * Monta menu de reas.
   * 
   * @return menu
   */
  private JMenu buildAreasMenu() {
    final JMenu menu = new JMenu(getMenuString("areas"));
    final int[] keys =
      new int[] { KeyEvent.VK_1, KeyEvent.VK_2, KeyEvent.VK_3, KeyEvent.VK_4,
          KeyEvent.VK_5, KeyEvent.VK_6, KeyEvent.VK_7, KeyEvent.VK_8,
          KeyEvent.VK_9 };

    int i = 0;
    final ButtonGroup grp = new ButtonGroup();
    boolean checked = false;
    for (String id : areasIds) {
      final DiskMonitorChangeAreaAction ac =
        new DiskMonitorChangeAreaAction(this, id);
      final JRadioButtonMenuItem item = new JRadioButtonMenuItem(ac);
      grp.add(item);
      menu.add(item);
      if (!checked) {
        item.setSelected(true);
        checked = true;
      }
      if (i < keys.length) {
        final KeyStroke keyStroke =
          KeyStroke.getKeyStroke(keys[i], InputEvent.CTRL_DOWN_MASK);
        item.setAccelerator(keyStroke);
      }
      i++;
    }
    return menu;
  }

  /**
   * Montagem do menu principal.
   * 
   * @return menu
   */
  private JMenuBar buildMenuBar() {
    final JMenuBar menuBar = new JMenuBar();

    final JMenu optionsMenu = buildOptionsMenu();
    final JMenu areasMenu = buildAreasMenu();
    final JMenu helpMenu = buildHelpMenu();

    menuBar.add(optionsMenu);
    menuBar.add(areasMenu);
    menuBar.add(helpMenu);

    return menuBar;
  }

  /**
   * Constroi o menu de opes.
   * 
   * @return o menu
   */
  private JMenu buildOptionsMenu() {
    final JMenu menu = new JMenu(getMenuString("options"));
    menu.add(new DiskMonitorRefreshAction(this));
    menu.addSeparator();
    menu.add(new ApplicationExitAction(this));
    return menu;
  }

  /**
   * Monta texto de internacionalizao de menus.
   * 
   * @param tag tag
   * @return texto
   */
  private String getMenuString(String tag) {
    final String prefix = getClass().getSimpleName();
    return getString(prefix + "." + tag + ".menu");
  }

  /**
   * Tratador de exceo da aplicao.
   * 
   * @param e exceo
   */
  final public void handleException(final Exception e) {
    final ApplicationFrame frame = getApplicationFrame();
    final String title = getName();
    final String msg = e.getMessage();
    StandardErrorDialogs.showErrorDialog(frame, title, msg, e);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() {
    if (thread != null) {
      thread.killThread();
    }
  }

  /**
   * Criao do painel principal da aplicao.
   * 
   * @param container container do dilogo.
   */
  private void mountMainPanel(Container container) {
    container.setLayout(new BorderLayout());
    container.add(pieChartPanel, BorderLayout.CENTER);
  }

  /**
   * Atualiza grfico corrente
   */
  public void refresh() {
    final boolean isEDT = SwingThreadDispatcher.isEventDispatchThread();
    DiskOccupation occ;
    if (isEDT) {
      occ = DiskUsageProxy.getAreaOccupation(currentAreaId);
      updateChartWithEDT(currentAreaId, occ);
    }
    else {
      final DiskUsageServiceInterface service =
        ClientRemoteLocator.diskUsageService;
      try {
        occ = service.getAreaOccupation(currentAreaId);
      }
      catch (RemoteException re) {
        occ = null;
      }

      final DiskOccupation auxOcc = occ;
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        public void run() {
          updateChartWithEDT(currentAreaId, auxOcc);
        }
      });
    }
  }

  /**
   * Atualiza grfico corrente
   * 
   * @param id identificador da rea.
   * @param occupation ocupao
   */
  private void updateChartWithEDT(final String id, DiskOccupation occupation) {
    if (currentAreaId == null) {
      pieChartPanel.resetChart(id);
    }
    pieChartPanel.updateChart(currentAreaId, occupation);
  }

  /**
   * Ajusta o valor: rea corrente de visualizao ({@link #currentAreaId} )
   * 
   * @param id id a ser ajustado
   */
  public final void setCurrentAreaId(String id) {
    this.currentAreaId = id;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    return true;
  }
}
