package csbase.client.applications.fileexchanger.logic;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.rmi.RemoteException;

import csbase.client.applications.fileexchanger.FileExchangerUI;
import csbase.exception.project.FileLockedException;
import csbase.logic.ClientProjectFile;
import csbase.logic.ProjectFileChannelLoadListener;
import csbase.logic.ProjectFileFilter;

/**
 * @author Tecgraf/PUC-Rio
 */
public class ExchangeExportThread extends ExchangeThread {

  /**
   * {@inheritDoc}
   */
  @Override
  public void run() {
    final ExchangeExport exchange = (ExchangeExport) getExchange();
    final ClientProjectFile remoteFile = exchange.getRemoteFile();
    final File localFile = exchange.getLocalFile();
    final ExchangeMode transferMode = exchange.getTransferMode();
    final ProjectFileFilter filter = exchange.getFilter();
    try {
      if (remoteFile.isDirectory()) {
        // Toda a subrvore do diretrio  carregada do servidor para se 
        // calcular o tamanho total do diretrio.
        exchange.setTotalTransferSize(remoteFile.getTotalSize(filter));
      }
      switch (transferMode) {
        case STREAM:
          exportAsStream(remoteFile, localFile, filter);
          break;
        case OPERATION:
          exportAsDownload(remoteFile, localFile, filter);
          break;
      }
    }
    catch (Exception e) {
      exchange.signalEnded(e);
      return;
    }
    exchange.signalEnded(null);
  }

  /**
   * Realiza o download de um arquivo de projeto.
   * 
   * @param remoteFile O arquivo do projeto a ser exportado.
   * @param localFile Representao do arquivo local.
   * @param filter Filtro usado para exportar os arquivos filhos de um
   *        diretrio. Somente os arquivos filhos aceitos pelo filtro sero
   *        exportados.
   * 
   * @throws IOException
   * @throws FileLockedException
   */
  private void exportAsDownload(final ClientProjectFile remoteFile,
    final File localFile, final ProjectFileFilter filter)
    throws FileLockedException, IOException {
    final Exchange exchange = getExchange();
    if (remoteFile.isDirectory()) {
      // Cria o diretrio local.
      createDirectory(localFile);
      exchange.addCurrentTransferSize(localFile.length());
      // Todos os filhos j foram carregados do servidor para se obter o tamanho 
      // total da transferncia (ver run()), por isso podemos usar o mtodo 
      // getLocalChildren().
      final ClientProjectFile[] children = remoteFile.getLocalChildren();
      for (final ClientProjectFile child : children) {
        final String NewNameLocalFile =
          localFile.getAbsolutePath() + File.separator + child.getName();
        // Exporta o filho (download).
        if (filter == null || filter.accept(child)) {
          exportAsDownload(child, new File(NewNameLocalFile), filter);
        }
      }
    }
    else {
      // Exporta o arquivo (download).
      final int chunkSize = exchange.getBlockSize().getSize();
      OutputStream out =
        new BufferedOutputStream(new FileOutputStream(localFile));
      remoteFile.download(out, chunkSize, new ProjectFileChannelLoadListener() {
        @Override
        public void transferedBytes(long currentSize, long numBytes) {
          exchange.addCurrentTransferSize(numBytes);
        }
      });
      out.flush();
      out.close();
    }
  }

  /**
   * Exporta um arquivo de projeto como stream.
   * 
   * @param remoteFile O arquivo do projeto a ser exportado.
   * @param localFile Representao do arquivo local.
   * 
   * @param filter Filtro usado para exportar os arquivos filhos de um
   *        diretrio. Somente os arquivos filhos aceitos pelo filtro sero
   *        exportados. Somente os arquivos filhos aceitos pelo filtro sero
   *        exportados.
   * 
   * @throws IOException
   * @throws RemoteException
   */
  private void exportAsStream(final ClientProjectFile remoteFile,
    final File localFile, final ProjectFileFilter filter)
    throws RemoteException, IOException {
    final Exchange exchange = getExchange();
    if (remoteFile.isDirectory()) {
      // Cria o diretrio local.
      createDirectory(localFile);
      exchange.addCurrentTransferSize(localFile.length());
      // Todos os filhos j foram carregados do servidor para se obter o tamanho 
      // total da transferncia (ver run()), por isso podemos usar o mtodo 
      // getLocalChildren().
      final ClientProjectFile[] children = remoteFile.getLocalChildren();
      for (final ClientProjectFile child : children) {
        final String NewNameLocalFile =
          localFile.getAbsolutePath() + File.separator + child.getName();
        if (filter == null || filter.accept(child)) {
          // Exporta o filho (stream).
          exportAsStream(child, new File(NewNameLocalFile), filter);
        }
      }
    }
    else {
      // Exporta o arquivo (stream).
      final int bufferSize = exchange.getBlockSize().getSize();
      final InputStream prjStream = remoteFile.getInputStream();
      final BufferedInputStream in = new BufferedInputStream(prjStream);
      final FileOutputStream fileStream = new FileOutputStream(localFile);
      BufferedOutputStream out = new BufferedOutputStream(fileStream);
      byte[] buffer = new byte[bufferSize];
      int len = 0;
      while ((len = in.read(buffer)) > 0) {
        exchange.addCurrentTransferSize(len);
        out.write(buffer, 0, len);
      }
      out.flush();
      out.close();
      in.close();
    }
  }

  /**
   * Cria um diretrio localmente.
   * 
   * @param localFile Diretrio a ser criado localmente.
   * @return True se o diretrio foi criado, false, caso contrrio.
   * @throws IOException Exceo levantada.
   */
  private boolean createDirectory(final File localFile) throws IOException {
    if (!localFile.exists()) {
      final boolean result = localFile.mkdir();
      if (!result) {
        final String[] args = new String[] { localFile.getAbsolutePath() };
        final String msg =
          FileExchangerUI.getString("ExchangeExportThread.mkdir.error", args);
        throw new IOException(msg);
      }
    }
    return true;
  }

  /**
   * Construtor
   * 
   * @param exchange operao
   */
  public ExchangeExportThread(final Exchange exchange) {
    super(exchange);
  }
}
