package csbase.client.applications.flowapplication.graph.actions;

import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.geom.Point2D;
import java.text.MessageFormat;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.WindowConstants;

import csbase.client.applications.flowapplication.FlowApplicationUI;
import csbase.client.applications.flowapplication.Workspace;
import csbase.client.applications.flowapplication.filters.AddNodePopupActionFilter;
import csbase.client.applications.flowapplication.filters.WorkspaceFilter;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.desktop.DesktopComponentDialog;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;

/**
 *
 * Action para definir um label para um n no woerkspace do construtor de fluxos
 * ou restaurar o estado de no existir label definido.
 *
 * @author Tecgraf/PUC-Rio
 */
public class LabelAction extends GraphElementAction {

  /**
   * N corrente.
   */
  private GraphNode currentNode;

  /**
   * A rea de trabalho do editor de fluxos.
   */
  Workspace workspace;

  /**
   * Campo de texto em que ser inserido o label do n.
   */
  JTextField nameField;

  /**
   * Construtor
   *
   * @param element o n a ter o label definido.
   *
   * @param workspace workspace
   * 
   */
  public LabelAction(final GraphNode element, final Workspace workspace) {
    super(element, LabelAction.class.getName() + ".name",
      FlowApplicationUI.LABEL_ICON);
    this.workspace = workspace;
    currentNode = element;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionPerformed(final ActionEvent event) {

    DesktopComponentDialog newNameDialog = new DesktopComponentDialog(workspace
      .getGraph().getParentWindow(), getDialogTitle());
    newNameDialog.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
    newNameDialog.setContentPane(createPane(newNameDialog));
    newNameDialog.pack();
    newNameDialog.setLocationRelativeTo(workspace);
    newNameDialog.setResizable(false);
    newNameDialog.setVisible(true);
  }

  /**
   * @return o ttulo para o dilogo que esra action exibe.
   */
  private String getDialogTitle() {
    final String className = LabelAction.class.getName();
    return MessageFormat.format(LNG.get(className + ".title"), currentNode
      .getAlgorithmName());
  }

  /**
   * Cria o painel do dilogo de edio do label.
   *
   * @param dialog instncia do dilogo.
   * @return Painel.
   */
  private JPanel createPane(DesktopComponentDialog dialog) {
    final String className = LabelAction.class.getName();

    // Montagem do painel de parmetros
    JLabel nameLabel = new JLabel(LNG.get(className + ".new.label"));
    nameField = new JTextField(25);
    nameField.addKeyListener(new KeyAdapter() {
      @Override
      public void keyPressed(KeyEvent evt) {
        if (evt.getKeyCode() == KeyEvent.VK_ENTER) {
          changesName(dialog);
        }
      }

    });
    nameField.setText(currentNode.getAlgorithmTitle());
    nameField.selectAll();

    JPanel paramsPanel = GUIUtils.createBasicGridPanel(new JComponent[][] { {
        nameLabel, nameField } });

    // Montagem dos botes
    final String restoreText = LNG.get(className + ".button.restore.label");
    final String restoreTip = LNG.get(className + ".button.restore.tooltip");
    JButton restoreButton = new JButton(restoreText);
    restoreButton.addActionListener(e -> restoreName(dialog));
    restoreButton.setEnabled(currentNode.getLabel() != null);
    restoreButton.setToolTipText(restoreTip);

    final String okText = LNG.get(className + ".button.ok");
    JButton okButton = new JButton(okText);
    okButton.addActionListener(e -> changesName(dialog));

    final String cancelText = LNG.get(className + ".button.cancel");
    JButton cancelButton = new JButton(cancelText);
    cancelButton.addActionListener(e -> dialog.close());

    // Montagem do painel de botes
    JPanel buttonsPanel = new JPanel(new GridBagLayout());
    GUIUtils.matchPreferredSizes(okButton, cancelButton, restoreButton);
    buttonsPanel.add(restoreButton, new GBC(0, 0).west().insets(0, 0, 0, 5));
    buttonsPanel.add(okButton, new GBC(1, 0).center().insets(0, 15, 0, 5));
    buttonsPanel.add(cancelButton, new GBC(2, 0).east().insets(0, 5, 0, 0));
    buttonsPanel.setBorder(BorderFactory.createEmptyBorder(10, 0, 10, 0));

    // Montagem do painel principal
    JPanel mainPanel = new JPanel(new BorderLayout());
    mainPanel.add(paramsPanel, BorderLayout.CENTER);
    mainPanel.add(buttonsPanel, BorderLayout.SOUTH);
    return mainPanel;
  }

  /**
   * Restaura o n para o estado em que no h label definido.
   *
   * @param dialog dilogo que contm o boto que realiza essa ao.
   */
  private void restoreName(DesktopComponentDialog dialog) {
    nameField.requestFocusInWindow();
    nameField.setText(currentNode.getAlgorithmName());
    nameField.selectAll();
  }

  /**
   * Define um label para o n.
   *
   * @param dialog dilogo que contm o boto que realiza essa ao.
   */
  private void changesName(DesktopComponentDialog dialog) {
    final String text = nameField.getText();
    if (text == null) {
      return;
    }

    final String trimmedText = text.trim();
    if (trimmedText.isEmpty()) {
      return;
    }

    currentNode.setNewTitle(trimmedText);
    currentNode.repaint();
    dialog.close();
  }

  /**
   * Cria o filtro que cadastra a ao no menu popup.
   *
   * @param workspace A rea de trabalho (No aceita {@code null}).
   *
   * @return O filtro.
   */
  public static WorkspaceFilter createFilter(final Workspace workspace) {
    if (workspace == null) {
      throw new IllegalArgumentException("O parmetro workspace est nulo.");
    }
    return new AddNodePopupActionFilter(workspace) {
      @Override
      protected Action createAction(final GraphNode graphNode,
        final Point2D point) {
        return new LabelAction(graphNode, workspace);
      }
    };
  }

}
