package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.swing.ImageIcon;

import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.core.ProjectsManagerTask;
import csbase.client.util.user.UserDialog;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Define usurios que compartilham projetos, e seus modos de acesso.
 * 
 * @author Tecgraf
 */
public class SetProjectSharedAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public SetProjectSharedAction(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionDone() throws Exception {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() == 0) {
      return;
    }

    final List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    final ApplicationFrame appFrame = projectsManager.getApplicationFrame();

    final UserDialog ud;
    if (projects.size() == 1) {
      final ProjectsManagerData pmd = projects.get(0);
      ud =
        new UserDialog(appFrame, projectsManager.getUsers(), pmd.getUsersRO(),
          pmd.getUsersRW(), true, true);
    }
    else {
      ud = new UserDialog(appFrame, projectsManager.getUsers(), true, true);
    }

    final String title = getString("SetProjectSharedAction.selection.title");
    ud.setTitle(title);
    ud.execute();

    final Set<Object> usersRO = ud.getUsersRO();
    final Set<Object> usersRW = ud.getUsersRW();

    if (usersRO.isEmpty() && usersRW.isEmpty()) {
      return;
    }

    final boolean ret = runTask(projects, usersRO, usersRW);
    if (ret) {
      projectsManager.refreshProjectsTable();
      projectsManager.refreshInfoPanel();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return ApplicationImages.ICON_SHARED_16;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    final List<ProjectsManagerData> prjList =
      new ArrayList<ProjectsManagerData>();
    prjList.addAll(projectList);

    final List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    final User loggedUser = User.getLoggedUser();
    final Object loggedUserId = loggedUser.getId();

    int i = 0;
    while (i < prjList.size()) {
      final ProjectsManagerData pmd = prjList.get(i);
      final ProjectSpaceAllocation spaceAllocation =
        pmd.getProjectSpaceAllocation();

      final boolean isAdmin = loggedUser.isAdmin();
      final boolean isOwner = pmd.getOwnerId().equals(loggedUserId);

      /*
       * Para que um usurio possa tornar um projeto pblico  preciso que: - O
       * projeto possa ser aberto (no esteja aguardando alocao /
       * desalocao). - O usurio seja o administrador ou o dono do projeto.
       */
      if (spaceAllocation.isOpenable() && (isAdmin || isOwner)) {
        i++;
      }
      else {
        failedProjects.add(pmd);
        prjList.remove(i);
      }
    }

    final ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      final String deniedMessage =
        getString("SetProjectSharedAction.project.selection.denied.message")
          + getString("SetProjectSharedAction.project.requirements.message");
      StatusDialog.showNoneOkDialog(projectsManager, deniedMessage,
        failedProjects,
        getString("SetProjectSharedAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      final String someDeniedMessage =
        getString("SetProjectSharedAction.project.selection.some.denied.message")
          + getString("SetProjectSharedAction.project.requirements.message");
      final int res =
        StatusDialog
          .showSomeOkDialog(
            projectsManager,
            someDeniedMessage,
            prjList,
            getString("SetProjectSharedAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }
    return prjList;
  }

  /**
   * Roda e trata a execuo da task.
   * 
   * @param projects Lista de projetos vlidos para a operao.
   * @param idsRO Conjunto de ids dos usurios com permisso de leitura a serem
   *        adicionados aos projetos.
   * @param idsRW Conjunto de ids dos usurios com permisso de leitura e
   *        escrita a serem adicionados aos projetos.
   * @return Indica se a operao foi realizada com sucesso.
   * @throws Exception em caso de erro.
   */
  private boolean runTask(final List<ProjectsManagerData> projects,
    final Set<Object> idsRO, final Set<Object> idsRW) throws Exception {
    final ProjectsManager projectsManager = getProjectsManager();
    final DefineUsersTask defineUsersTask =
      new DefineUsersTask(projectsManager, projects, idsRO, idsRW);

    final ApplicationFrame frame = projectsManager.getApplicationFrame();
    defineUsersTask.execute(frame, getString("SetProjectSharedAction.title"),
      getString("SetProjectSharedAction.message"));

    if (defineUsersTask.wasCancelled()) {
      final String err = getString("SetProjectSharedAction.cancelled.message");
      defineUsersTask.showError(err);
      return false;
    }

    if (defineUsersTask.getStatus() != true) {
      final Exception exception = defineUsersTask.getError();
      throw exception;
    }

    for (ProjectsManagerData pmd : projects) {
      pmd.setSharingType(SharingType.PARTIAL);
      pmd.setUsers(idsRO, idsRW);
    }

    return true;
  }

}

/**
 * Task que adiciona usurios a um conjunto de projetos (transformando o escopo
 * do projeto em "compartilhado" se este ainda no for).
 * 
 * @author Tecgraf
 */
class DefineUsersTask extends ProjectsManagerTask<Void> {

  /**
   * Lista de projetos aos quais os usurios sero adicionados.
   */
  final private List<ProjectsManagerData> projects;

  /**
   * Conjunto de usurios com acesso apenas de leitura.
   */
  final private Set<Object> usersRO;

  /**
   * Conjunto de usurios com acesso de leitura e escrita.
   */
  final private Set<Object> usersRW;

  /**
   * Construtor.
   * 
   * @param projectsManager A aplicao.
   * @param projects Lista de projetos aos quais os usurios sero adicionados.
   * @param usersRO conjunto de usurios com acesso apenas de leitura
   * @param usersRW conjunto de usurios com acesso de leitura e escrita
   */
  public DefineUsersTask(final ProjectsManager projectsManager,
    final List<ProjectsManagerData> projects, final Set<Object> usersRO,
    final Set<Object> usersRW) {
    super(projectsManager);
    this.projects = projects;
    this.usersRO = usersRO;
    this.usersRW = usersRW;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws Exception {
    if (usersRO == null && usersRW == null) {
      return;
    }
    ProjectServiceInterface psi = ClientRemoteLocator.projectService;
    for (ProjectsManagerData pmd : projects) {
      CommonClientProject ccp = psi.openProject(pmd.getProjectId(), false);
      ccp.updateUsers(SharingType.PARTIAL, usersRO, usersRW);
      ccp.close(true);
    }
  }
}
