package csbase.client.applications.projectsmanager.panels;

import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.RowSorter;
import javax.swing.RowSorter.SortKey;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.panels.renderers.ServerLocationRenderer;
import csbase.client.applications.projectsmanager.panels.renderers.TableAllocationRenderer;
import csbase.client.applications.projectsmanager.panels.renderers.TableDateRenderer;
import csbase.client.applications.projectsmanager.panels.renderers.TableOccupationRenderer;
import csbase.client.applications.projectsmanager.panels.renderers.TableScopeRenderer;
import csbase.client.applications.projectsmanager.panels.renderers.TableSpaceAllocationRenderer;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.table.SortableTable;

/**
 * Painel que contem a tabela que apresenta os projetos existentes.
 * 
 * @author jnlopes
 * 
 */
public class TablePanel extends ProjectsManagerPanel {

  /**
   * Tabela que apresenta os projetos existentes.
   */
  final private SortableTable projectsTable = new SortableTable();

  /**
   * Lista dos projetos filtrados.
   */
  private List<ProjectsManagerData> visibleProjects =
    new ArrayList<ProjectsManagerData>();

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public TablePanel(final ProjectsManager projectsManager) {
    super(projectsManager);

    initComponents();
    buildPanel();

    addSelectionListeners();
    defineColumns();
    // Clinio: Comentado para futura remoo? [CSBASE-1056]
    // addMouseListeners();

    this.setVisible(true);
    this.validate();
    this.repaint();
  }

  /**
   * Adiciona listener de seleo de campos da tabela.
   */
  private void addSelectionListeners() {
    projectsTable.setSelectionMode(
      ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);

    projectsTable.getSelectionModel().addListSelectionListener(
      new ListSelectionListener() {
        /**
         * {@inheritDoc}
         */
        @Override
        public void valueChanged(ListSelectionEvent e) {
          if (e.getValueIsAdjusting()) {
            // Este evento  intermedirio e deve ser descartado
            return;
          }
          final ProjectsManager projectsManager = getProjectsManager();
          projectsManager.setSelectedProjectsList(TablePanel.this
            .getSelectedProjects());
        }
      });
  }

  // /**
  // * Adiciona MouseListeners.
  // */
  // private void addMouseListeners() {
  //
  // this.projectsTable
  // .addMouseListener(new ProjectsTableOccupationMouseAdapter(
  // getProjectsManager(), projectsTable));
  // this.projectsTable
  // .addMouseListener(new ProjectsTableAllocationMouseAdapter(
  // getProjectsManager(), projectsTable));
  // this.projectsTable.addMouseListener(new ProjectsTableSharingMouseAdapter(
  // getProjectsManager(), projectsTable));
  // }

  /**
   * Define os renderers e propriedades gerais das colunas da tabela de
   * projetos.
   */
  private void defineColumns() {
    projectsTable.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);

    final TableColumnModel columnModel = projectsTable.getColumnModel();
    final TableColumn column0 = columnModel.getColumn(0);
    column0.setHeaderValue(getString("TablePanel.project.access.tablecolumn"));
    column0.setCellRenderer(new TableScopeRenderer());
    column0.setPreferredWidth(50);

    final ProjectsManager projectsManager = getProjectsManager();
    final boolean areaReserved = projectsManager.hasAreaReserved();

    int idx = 1;
    if (areaReserved) {
      final TableColumn column1 = columnModel.getColumn(idx);
      column1.setHeaderValue(getString(
        "TablePanel.project.allocation.tablecolumn"));
      column1.setCellRenderer(new TableSpaceAllocationRenderer());
      column1.setPreferredWidth(50);
      idx++;

      final TableColumn column2 = columnModel.getColumn(idx);
      column2.setHeaderValue(getString(
        "TablePanel.allocated.space.tablecolumn"));
      column2.setCellRenderer(new TableAllocationRenderer());
      column2.setPreferredWidth(90);
      idx++;
    }

    final TableColumn column3 = columnModel.getColumn(idx);
    column3.setHeaderValue(getString("TablePanel.project.name.tablecolumn"));
    column3.setPreferredWidth(170);
    idx++;

    final TableColumn column4 = columnModel.getColumn(idx);
    column4.setHeaderValue(getString("TablePanel.owner.name.tablecolumn"));
    column4.setPreferredWidth(60);
    idx++;

    final TableColumn column5 = columnModel.getColumn(idx);
    column5.setHeaderValue(getString("TablePanel.occupied.space.tablecolumn"));
    column5.setCellRenderer(new TableOccupationRenderer());
    column5.setPreferredWidth(90);
    idx++;

    final TableColumn column6 = columnModel.getColumn(idx);
    column6.setHeaderValue(getString("TablePanel.creation.date.tablecolumn"));
    column6.setCellRenderer(new TableDateRenderer());
    column6.setPreferredWidth(160);
    idx++;

    final TableColumn column7 = columnModel.getColumn(idx);
    column7.setHeaderValue(getString("TablePanel.last.used.date.tablecolumn"));
    column7.setCellRenderer(new TableDateRenderer());
    column7.setPreferredWidth(160);
    idx++;

    final TableColumn column8 = columnModel.getColumn(idx);
    column8.setHeaderValue(getString("TablePanel.server.name.tablecolumn"));
    column8.setPreferredWidth(160);
    idx++;

    if (areaReserved) {
      final TableColumn column9 = columnModel.getColumn(idx);
      column9.setHeaderValue(getString(
        "TablePanel.server.location.tablecolumn"));
      column9.setCellRenderer(new ServerLocationRenderer());
      column9.setPreferredWidth(250);
      idx++;
    }

  }

  /**
   * Pega todos os projetos selecionados.
   * 
   * @return array dos projetos selecionados.
   */
  public List<ProjectsManagerData> getSelectedProjects() {
    final int[] rows = projectsTable.getSelectedRows();

    final List<ProjectsManagerData> projectsManagerData =
      new ArrayList<ProjectsManagerData>();
    for (int i = 0; i < rows.length; i++) {
      projectsManagerData.add(visibleProjects.get(projectsTable
        .convertRowIndexToModel(rows[i])));
    }

    return projectsManagerData;
  }

  /**
   * Inicia os componentes.
   */
  private void initComponents() {
    projectsTable.setNoSortStateEnabled(true);
    final SimpleTableModel stm = new SimpleTableModel(this);
    projectsTable.setModel(stm);
    projectsTable.setUpdateSelectionOnSort(true);

    final ProjectsManager projectsManager = getProjectsManager();

    if (!projectsManager.hasAreaReserved()) {
      final TableColumnModel model = projectsTable.getColumnModel();
      final TableColumn column1 = model.getColumn(1);
      final TableColumn column2 = model.getColumn(2);
      final TableColumn column9 = model.getColumn(9);
      model.removeColumn(column1);
      model.removeColumn(column2);
      model.removeColumn(column9);
    }
  }

  /**
   * Monta o painel.
   */
  private void buildPanel() {
    setLayout(new GridBagLayout());
    JScrollPane projectsTablePane = new JScrollPane(this.projectsTable);
    add(projectsTablePane, new GBC(0, 0).both());
  }

  /**
   * Adiciona um novo projeto  tabela.
   * 
   * @param prj projeto
   */
  public void addProjectToTable(final ProjectsManagerData prj) {
    if (prj == null) {
      return;
    }

    final RowSorter<? extends TableModel> rs = projectsTable.getRowSorter();
    final List<? extends SortKey> l = rs.getSortKeys();
    rs.setSortKeys(null);
    visibleProjects.add(prj);
    rs.setSortKeys(l);
    refreshTable();
  }

  /**
   * Remove um projeto. O projeto  removido tanto da lista de todos os projetos
   * disponveis, quanto da dos projetos visveis.
   * 
   * @param pmd O projeto.
   */
  public void removeProjectFromTable(ProjectsManagerData pmd) {
    if (pmd == null) {
      return;
    }
    RowSorter<? extends TableModel> rs = projectsTable.getRowSorter();
    List<? extends SortKey> l = rs.getSortKeys();
    rs.setSortKeys(null);
    visibleProjects.remove(pmd);
    rs.setSortKeys(l);
    refreshTable();
  }

  /**
   * Esvazia conteudo da tabela.
   */
  final public void emptyTableData() {
    projectsTable.clearSelection();
    final ProjectsManager projectsManager = getProjectsManager();
    projectsManager.clearInfoFields();
    visibleProjects.clear();
    refreshTable();
  }

  /**
   * Realiza refresh na tabela.
   */
  public void refreshTable() {
    projectsTable.revalidate();
    projectsTable.repaint();
  }

  /**
   * Ajusta a preciso de um double para que tenha apenas 3 casas decimais.
   * 
   * @param num O nmero a ser ajustado.
   * @return public double adjustPrecision(double num) { num =
   *         (double)Math.round(num * 1000) / 1000; return num; }
   */

  /**
   * @param projects lista de projetos visveis
   */
  final public void setVisibleProjects(
    final List<ProjectsManagerData> projects) {
    RowSorter<? extends TableModel> rs = projectsTable.getRowSorter();
    final List<? extends SortKey> l = rs.getSortKeys();
    rs.setSortKeys(null);
    visibleProjects.clear();
    visibleProjects.addAll(projects);
    rs.setSortKeys(l);
  }

  /**
   * Retorna a lista de projetos filtrados.
   * 
   * @return lista de projetos filtrados. ATENO: retorna a prpria lista, e
   *         no uma cpia.
   */
  final public List<ProjectsManagerData> getVisibleProjects() {
    return visibleProjects;
  }

  /**
   * Define o intervalo de seleo da tabela de projetos.
   * 
   * @param idx0 O ndice inicial.
   * @param idx1 O ndice final.
   */
  final public void setTableSelectionInterval(final int idx0, final int idx1) {
    projectsTable.getSelectionModel().setSelectionInterval(idx0, idx1);
  }

  /**
   * Desmarca as linhas selecionadas da tabela.
   */
  public void clearTableSelection() {
    projectsTable.clearSelection();
  }

}
