/**
 * $Id$
 */

package csbase.client.applications.projectsmanager.panels.filters;

import java.awt.Color;
import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyListener;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;

import javax.swing.AbstractAction;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.models.ProjectsManagerScope;
import csbase.client.util.user.UserDialog;

/**
 * Painel de filtragem de elementos da tabela por nome de projeto, dono de
 * projeto e talvez outros a serem adicionados.
 * 
 * @author jnlopes
 */
public class ProjectBasicInfoFilterPanel extends AbstractProjectFilter {

  /**
   * Campo de nome de projeto.
   */
  private JTextField projectNameField;

  /**
   * Painel de nome de projeto.
   */
  private JPanel projectNamePanel;

  /**
   * Label de nome de projeto.
   */
  private JLabel projectNameLabel;

  /**
   * Campo para chave de dono de projeto.
   */
  private JTextField projectOwnerField;

  /**
   * Painel para chave de dono de projeto.
   */
  private JPanel projectOwnerPanel;

  /**
   * Label para chave de dono de projeto.
   */
  private JLabel projectOwnerLabel;

  /**
   * Painel para compartilhamento de projeto.
   */
  private JPanel statusComboPanel;

  /**
   * Label para status de projeto.
   */
  private JLabel projectStatusLabel;

  /**
   * Combobox para status de projeto.
   */
  private JComboBox projectStatusBox;

  /**
   * Painel para compartilhamento de projeto.
   */
  private JPanel sharingComboPanel;

  /**
   * Label para compartilhamento de projeto.
   */
  private JLabel projectSharingLabel;

  /**
   * Combobox para compartilhamento de projeto.
   */
  private JComboBox projectSharingBox;

  /**
   * Boto de busca por usurio.
   */
  private JButton searchButton;

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public ProjectBasicInfoFilterPanel(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void initComponents() {
    projectNameLabel =
      new JLabel(getString("ProjectBasicInfoFilterPanel.project.name.label"));
    projectNameField = new JTextField();
    projectOwnerField = new JTextField();
    projectOwnerLabel =
      new JLabel(
        getString("ProjectBasicInfoFilterPanel.project.owner.id.label"));
    projectStatusLabel =
      new JLabel(getString("ProjectBasicInfoFilterPanel.project.status.label"));
    projectStatusBox = new JComboBox();
    projectSharingLabel =
      new JLabel(getString("ProjectBasicInfoFilterPanel.project.sharing.label"));
    projectSharingBox = new JComboBox();

    projectNameLabel.setForeground(Color.DARK_GRAY);
    projectOwnerLabel.setForeground(Color.DARK_GRAY);
    projectStatusLabel.setForeground(Color.DARK_GRAY);
    projectSharingLabel.setForeground(Color.DARK_GRAY);

    projectStatusBox.addItem(null);
    for (ProjectSpaceAllocation p : ProjectSpaceAllocation.values()) {
      projectStatusBox.addItem(p);
    }
    projectStatusBox.setRenderer(new BoxRenderer(true));

    projectSharingBox.addItem(null);

    for (ProjectsManagerScope p : ProjectsManagerScope.values()) {
      if (p != ProjectsManagerScope.NOT_APPLICABLE) {
        projectSharingBox.addItem(p);
      }
    }
    projectSharingBox.setRenderer(new BoxRenderer(false));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void buildPanel() {

    searchButton = new JButton(new AbstractAction() {

      @Override
      public void actionPerformed(ActionEvent e) {
        final ProjectsManager projectsManager = getProjectsManager();
        final ApplicationFrame appFrame = projectsManager.getApplicationFrame();
        final String title =
          getString("ProjectBasicInfoFilterPanel.user.selection.title");
        try {
          final Set<Object> ids =
            UserDialog.showUserSelectionDialog(appFrame, projectsManager
              .getUsers(), false, title);

          if (ids != null) {
            final Object userId = ids.toArray()[0];
            projectOwnerField.setText((String) userId);
            final KeyListener keyListener =
              projectOwnerField.getKeyListeners()[0];
            keyListener.keyReleased(null);
          }
        }
        catch (Exception e1) {
          e1.printStackTrace();
          final String errMsg =
            getString("ProjectBasicInfoFilterPanel.user.retrieval.error.message");
          StandardDialogs.showErrorDialog(appFrame, "Error", errMsg);
          return;
        }
      }
    });

    searchButton
      .setToolTipText(getString("ProjectBasicInfoFilterPanel.tooltip.search.button"));
    searchButton.setIcon(ApplicationImages.ICON_FIND_USER_16);

    GUIUtils.trimImageButton(searchButton);

    setLayout(new GridBagLayout());

    projectNamePanel = new JPanel(new GridBagLayout());
    projectNamePanel.add(this.projectNameField, new GBC(0, 0).horizontal());

    projectOwnerPanel = new JPanel(new GridBagLayout());
    projectOwnerPanel.add(this.projectOwnerField, new GBC(0, 0).horizontal());
    projectOwnerPanel.add(searchButton, new GBC(1, 0).insets(0, 10, 0, 0));

    sharingComboPanel = new JPanel(new GridBagLayout());
    sharingComboPanel.add(this.projectSharingBox, new GBC(0, 0).horizontal());

    statusComboPanel = new JPanel(new GridBagLayout());
    statusComboPanel.add(this.projectStatusBox, new GBC(0, 0).horizontal());

    add(projectNamePanel, new GBC(0, 0).horizontal());
    add(projectOwnerPanel, new GBC(1, 0).horizontal());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean projectMatchesFilter(final ProjectsManagerData prj) {
    final Pattern namePattern = convertToPattern(projectNameField);
    final Pattern ownerPattern = convertToPattern(projectOwnerField);
    if (namePattern == null || ownerPattern == null) {
      return false;
    }
    return projectMatchesFilter(namePattern, ownerPattern, prj);
  }

  /**
   * Verifica se um projeto  aceito por este filtro.
   * 
   * @param namePattern padro para validao do nome do projeto
   * @param ownerPattern padro para validao do dono do projeto
   * @param prj projeto
   * @return <code>true</code> se o projeto  aceito por este filtro
   */
  private boolean projectMatchesFilter(final Pattern namePattern,
    final Pattern ownerPattern, final ProjectsManagerData prj) {
    if (namePattern.matcher(prj.getProjectName()).matches()
      && ownerPattern.matcher(prj.getOwnerId().toString()).matches()) {
      final ProjectsManagerScope prjScopeFilter =
        (ProjectsManagerScope) this.projectSharingBox.getSelectedItem();
      final ProjectSpaceAllocation prjSpaceAllocationFilter =
        (ProjectSpaceAllocation) this.projectStatusBox.getSelectedItem();

      if (prjScopeFilter == null || prjScopeFilter == prj.getScope()) {
        if (prjSpaceAllocationFilter == null
          || prjSpaceAllocationFilter == prj.getProjectSpaceAllocation()) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void runSpecificFilter(final List<ProjectsManagerData> allProjects,
    final List<ProjectsManagerData> filteredProjects) {
    final Pattern namePattern = convertToPattern(projectNameField);
    final Pattern ownerPattern = convertToPattern(projectOwnerField);
    final boolean validPattern = namePattern != null && ownerPattern != null;

    if (!validPattern) {
      final ProjectsManager projectsManager = getProjectsManager();
      projectsManager.showError("<<Erro na especificao do filtro>>");
      return;
    }

    for (ProjectsManagerData pmd : allProjects) {
      if (projectMatchesFilter(namePattern, ownerPattern, pmd)) {
        filteredProjects.add(pmd);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addListener(KeyListener kl) {
    projectNameField.addKeyListener(kl);
    projectOwnerField.addKeyListener(kl);
  }

  /**
   * Adiciona ItemListener as combo boxes.
   * 
   * @param itemListener o listener.
   */
  public void addItemListener(final ItemListener itemListener) {
    projectStatusBox.addItemListener(itemListener);
    projectSharingBox.addItemListener(itemListener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void emptyFilterData() {
    projectNameField.setText("");
    projectOwnerField.setText("");
    projectStatusBox.setSelectedItem(null);
    projectSharingBox.setSelectedItem(null);
  }

  /**
   * Retorna o label do nome do projeto.
   * 
   * @return O label do nome do projeto.
   */
  public JLabel getProjectNameLabel() {
    return projectNameLabel;
  }

  /**
   * Retorna o painel do dono do projeto.
   * 
   * @return O painel do dono do projeto.
   */
  public JPanel getProjectOwnerPanel() {
    return projectOwnerPanel;
  }

  /**
   * Retorna o label do dono do projeto.
   * 
   * @return O label do dono do projeto.
   */
  public JLabel getProjectOwnerLabel() {
    return projectOwnerLabel;
  }

  /**
   * Retorna o label do status do projeto.
   * 
   * @return O label do status do projeto.
   */
  public JLabel getProjectStatusLabel() {
    return projectStatusLabel;
  }

  /**
   * Retorna o label de compartilhamento de projeto.
   * 
   * @return O label de compartilhamento de projeto.
   */
  public JLabel getProjectSharingLabel() {
    return projectSharingLabel;
  }

  /**
   * Retorna o painel de compartilhamento.
   * 
   * @return O painel de compartilhamento.
   */
  public JPanel getSharingComboPanel() {
    return sharingComboPanel;
  }

  /**
   * Retorna o painel de status.
   * 
   * @return O painel de status.
   */
  public JPanel getStatusComboPanel() {
    return statusComboPanel;
  }

  /**
   * Retorna o campo de nome de projeto.
   * 
   * @return O campo de nome de projeto.
   */
  public JTextField getProjectNameField() {
    return projectNameField;
  }

}

/**
 * Renderer das combo boxes.
 * 
 * @author jnlopes
 * 
 */
class BoxRenderer extends DefaultListCellRenderer {

  /**
   * Indica se  combo de status de projeto ou no.
   */
  private boolean isStatusBox;

  /**
   * Construtor
   * 
   * @param isStatusBox Indica se  combo de status de projeto ou no.
   */
  public BoxRenderer(final boolean isStatusBox) {
    this.isStatusBox = isStatusBox;
    /*
     * setOpaque(true); setHorizontalAlignment(CENTER);
     * setVerticalAlignment(CENTER);
     */
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getListCellRendererComponent(JList list, Object value,
    int index, boolean isSelected, boolean cellHasFocus) {

    if (value == null) {
      return super.getListCellRendererComponent(list, "-", index, isSelected,
        cellHasFocus);
    }

    if (!isStatusBox) {
      final ProjectsManagerScope pms = (ProjectsManagerScope) value;
      final JLabel label =
        (JLabel) super.getListCellRendererComponent(list, pms.getTitle(),
          index, isSelected, cellHasFocus);
      label.setIcon(pms.getIcon());
      return label;
    }

    final ProjectSpaceAllocation psa = (ProjectSpaceAllocation) value;
    final JLabel label =
      (JLabel) super.getListCellRendererComponent(list, psa.getTitle(), index,
        isSelected, cellHasFocus);
    label.setIcon(psa.getIcon());
    return label;
  }

}
