package csbase.client.csdk.v2.command;

import java.awt.Window;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import csbase.client.csdk.v2.core.CSDKAbstractContext;
import csbase.client.csdk.v2.filesystem.CSDKFile;
import csbase.client.desktop.RemoteTask;
import csbase.client.facilities.algorithms.executor.AlgorithmCommandExecutor;
import csbase.client.facilities.algorithms.executor.CommandObserver;
import csbase.client.facilities.algorithms.executor.FlowCommandExecutor;
import csbase.client.facilities.algorithms.parameters.FlowParametersValues;
import csbase.client.facilities.algorithms.parameters.NodeParametersValues;
import csbase.logic.ClientFile;
import csbase.logic.CommandNotification;
import csbase.logic.algorithms.AlgorithmVersionId;
import csdk.v2.api.command.CommandException;
import csdk.v2.api.command.IAlgorithmTemplate;
import csdk.v2.api.command.ICommandContext;
import csdk.v2.api.command.ICommandObserver;
import csdk.v2.api.command.IFlowTemplate;
import tecgraf.javautils.core.lng.LNG;

/**
 * Implementao do contexto de execuo de comandos do CSDK, que permite a
 * submisso e o acompanhamento de algoritmos, fluxos instalados ou no nos SGAs
 * conectados ao servidor.
 */
public class CSDKCommandContext extends CSDKAbstractContext implements
  ICommandContext {

  /**
   * {@inheritDoc}
   */
  @Override
  public String executeAlgorithm(IAlgorithmTemplate template,
    final String description, final String sgaServerName, final Window owner,
    final ICommandObserver observer) throws CommandException {
    final AlgorithmCommandExecutor executor = new AlgorithmCommandExecutor();
    final NodeParametersValues paramValues =
      createAlgorithmParameterConfiguration(template);
    final CommandObserver commandObserver = createObserver(observer);
    RemoteTask<String> submissionTask = new RemoteTask<String>() {
      @Override
      protected void performTask() throws Exception {
        setResult(executor.executeCommand(paramValues, description,
          sgaServerName, owner, commandObserver));
      }
    };
    String msg =
      LNG.get(CSDKCommandContext.class.getSimpleName()
        + ".algorithm.submission.task");
    if (!submissionTask.execute(owner, null, msg)) {
      String errorMsg =
        LNG.get(CSDKCommandContext.class.getSimpleName()
          + ".algorithm.configuration.error");
      throw new CommandException(errorMsg);
    }
    return submissionTask.getResult();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String executeFlow(IFlowTemplate template, final String description,
    final String sgaServerName, final Window owner,
    final ICommandObserver observer) throws CommandException {
    if (template.getFlowDefinitionFile() != null) {
      return executeFlowFromFile(template, description, sgaServerName, owner,
        observer);
    }
    else {
      return executeInstalledFlow(template, description, sgaServerName, owner,
        observer);
    }
  }

  /**
   * Executa um fluxo instalado no servidor (no repositrio de algoritmos).
   * 
   * @param template definio do fluxo a ser executado e dos valores dos seus
   *        parmetros
   * @param description descrio do comando
   * @param sgaServerName nome do servidor do SGA a ser usado para execuo. Se
   *        for null, indica que a seleo do servidor ser automtica.
   * @param owner janela-me da execuo.
   * @param observer Observador do comando executado.
   * @return o identificador do comando submetido.
   * 
   * @throws CommandException em caso de erro durante a submisso do comando.
   */
  private String executeInstalledFlow(final IFlowTemplate template,
    final String description, final String sgaServerName, final Window owner,
    final ICommandObserver observer) throws CommandException {
    final FlowCommandExecutor executor = new FlowCommandExecutor();
    final CommandObserver commandObserver = createObserver(observer);
    final FlowParametersValues paramValues =
      createFlowParameterConfig(template);
    RemoteTask<String> submissionTask = new RemoteTask<String>() {
      @Override
      protected void performTask() throws Exception {
        setResult(executor.executeCommand(paramValues, description, false,
          sgaServerName, owner, commandObserver));
      }
    };
    String msg =
      LNG.get(CSDKCommandContext.class.getSimpleName()
        + ".flow.submission.task");
    if (!submissionTask.execute(owner, null, msg)) {
      String errorMsg =
        LNG.get(CSDKCommandContext.class.getSimpleName()
          + ".flow.configuration.error");
      throw new CommandException(errorMsg);
    }
    return submissionTask.getResult();
  }

  /**
   * Executa um fluxo a partir de um arquivo.
   * 
   * @param template definio do fluxo a ser executado e dos valores dos seus
   *        parmetros
   * @param description descrio do comando
   * @param sgaServerName nome do servidor do SGA a ser usado para execuo. Se
   *        for null, indica que a seleo do servidor ser automtica.
   * @param owner janela-me da execuo.
   * @param observer Observador do comando executado.
   * @return o identificador do comando submetido.
   * 
   * @throws CommandException em caso de erro durante a submisso do comando.
   */
  private String executeFlowFromFile(final IFlowTemplate template,
    final String description, final String sgaServerName, final Window owner,
    final ICommandObserver observer) throws CommandException {
    final CSDKFile flowFile = (CSDKFile) template.getFlowDefinitionFile();
    final ClientFile file = flowFile.getFile();
    final FlowCommandExecutor executor = new FlowCommandExecutor();
    final CommandObserver commandObserver = createObserver(observer);
    final Map<Integer, NodeParametersValues> paramValues =
      createFlowParameterValues(template);
    RemoteTask<String> submissionTask = new RemoteTask<String>() {
      @Override
      protected void performTask() throws Exception {
        setResult(executor.executeCommand(file, paramValues, description,
          false, sgaServerName, owner, commandObserver));
      }
    };
    String msg =
      LNG.get(CSDKCommandContext.class.getSimpleName()
        + ".flow.submission.task");
    if (!submissionTask.execute(owner, null, msg)) {
      String errorMsg =
        LNG.get(CSDKCommandContext.class.getSimpleName()
          + ".flow.configuration.error");
      throw new CommandException(errorMsg);
    }
    return submissionTask.getResult();
  }

  /**
   * Cria o conjunto de parmetros do algoritmo a ser executado a partir da
   * descrio do algoritmo no CSDK.
   * 
   * @param template a descrio do algoritmo no CSDK.
   * @return o conjunto de parmetros.
   */
  private NodeParametersValues createAlgorithmParameterConfiguration(
    IAlgorithmTemplate template) {
    AlgorithmVersionId algorithmVersionId =
      createVersionId(template.getAlgorithmVersionId());
    NodeParametersValues paramValues =
      new NodeParametersValues(template.getAlgorithmName(), algorithmVersionId,
        template.getParametersValues());
    return paramValues;
  }

  /**
   * Cria o conjunto de parmetros do fluxo a ser executado a partir da
   * descrio do fluxo no CSDK.
   * 
   * @param template a descrio do fluxo no CSDK.
   * @return o conjunto de parmetros.
   */
  private FlowParametersValues createFlowParameterConfig(IFlowTemplate template) {
    Map<Integer, NodeParametersValues> values =
      createFlowParameterValues(template);
    AlgorithmVersionId flowVersionId =
      createVersionId(template.getFlowVersionId());
    return new FlowParametersValues(template.getFlowName(), flowVersionId,
      values);
  }

  /**
   * Cria o conjunto de parmetros do fluxo a ser executado a partir da
   * descrio do fluxo no CSDK.
   * 
   * @param template a descrio do fluxo no CSDK.
   * @return o conjunto de parmetros.
   */
  private Map<Integer, NodeParametersValues> createFlowParameterValues(
    IFlowTemplate template) {
    Map<Integer, IAlgorithmTemplate> parametersValuesList =
      template.getParametersValuesByNodeId();
    Map<Integer, NodeParametersValues> values =
      new HashMap<Integer, NodeParametersValues>();
    for (Entry<Integer, IAlgorithmTemplate> nodeTemplate : parametersValuesList
      .entrySet()) {
      values.put(nodeTemplate.getKey(),
        createAlgorithmParameterConfiguration(nodeTemplate.getValue()));
    }
    return values;
  }

  /**
   * Gera o identificador da verso do algoritmo a partir da sua representao
   * textual.
   * 
   * @param id a verso em formato textual.
   * @return o identificador.
   */
  private AlgorithmVersionId createVersionId(String id) {
    AlgorithmVersionId algorithmVersionId = AlgorithmVersionId.create(id);
    return algorithmVersionId;
  }

  /**
   * Encapsula o observador de comandos do CSDK em um observador do sistema.
   * 
   * @param observer o observador do CSDK.
   * @return o observador do CSBASE.
   */
  private CommandObserver createObserver(final ICommandObserver observer) {
    if (observer == null) {
      return null;
    }
    CommandObserver commandObserver = new CommandObserver() {
      @Override
      public void notify(CommandNotification data) {
        CSDKCommandInfo cmdInfo = new CSDKCommandInfo(data);
        observer.onCommandEnd(cmdInfo);
      }
    };
    return commandObserver;
  }

}
